import type { Express } from "express";
import { createServer, type Server } from "http";
import { storage } from "./storage";
import { analyzeToken, chatWithAI } from "./lib/deepseek";

const CHAIN_IDS: Record<string, string> = {
  ethereum: "1",
  bsc: "56",
  polygon: "137",
  arbitrum: "42161",
  base: "8453",
  avalanche: "43114",
  fantom: "250",
  optimism: "10",
};

const CHAIN_NAMES: Record<string, string> = {
  "1": "Ethereum",
  "56": "BSC",
  "137": "Polygon",
  "42161": "Arbitrum",
  "8453": "Base",
  "43114": "Avalanche",
  "250": "Fantom",
  "10": "Optimism",
};

const TRUST_WALLET_CHAIN_NAMES: Record<string, string> = {
  "1": "ethereum",
  "56": "smartchain",
  "137": "polygon",
  "42161": "arbitrum",
  "8453": "base",
  "43114": "avalanchec",
  "250": "fantom",
  "10": "optimism",
};

function getTokenLogo(address: string, chainId: string): string {
  const chainName = TRUST_WALLET_CHAIN_NAMES[chainId] || "ethereum";
  return `https://raw.githubusercontent.com/trustwallet/assets/master/blockchains/${chainName}/assets/${address}/logo.png`;
}

const DEXSCREENER_CHAIN_IDS: Record<string, string> = {
  "1": "ethereum",
  "56": "bsc",
  "137": "polygon",
  "42161": "arbitrum",
  "8453": "base",
  "43114": "avalanche",
  "250": "fantom",
  "10": "optimism",
  "solana": "solana",
};

async function fetchDexScreenerData(tokenAddress: string, chainId: string) {
  try {
    const dexChain = DEXSCREENER_CHAIN_IDS[chainId] || "bsc";
    const response = await fetch(
      `https://api.dexscreener.com/latest/dex/tokens/${tokenAddress}`
    );
    
    if (response.ok) {
      const data = await response.json();
      if (data.pairs && data.pairs.length > 0) {
        const matchingPair = data.pairs.find((p: any) => p.chainId === dexChain) || data.pairs[0];
        const txns = matchingPair?.txns || {};
        const priceChange = matchingPair?.priceChange || {};
        
        const buys24h = txns.h24?.buys || 0;
        const sells24h = txns.h24?.sells || 0;
        const totalTxns = buys24h + sells24h;
        const buyPressure = totalTxns > 0 ? ((buys24h / totalTxns) * 100).toFixed(1) : "50.0";
        
        let sentiment = "Neutral";
        if (parseFloat(buyPressure) > 60) sentiment = "Bullish";
        else if (parseFloat(buyPressure) < 40) sentiment = "Bearish";
        
        return {
          logo: matchingPair?.info?.imageUrl || null,
          website: matchingPair?.info?.websites?.[0]?.url || null,
          twitter: matchingPair?.info?.socials?.find((s: any) => s.type === "twitter")?.url || null,
          telegram: matchingPair?.info?.socials?.find((s: any) => s.type === "telegram")?.url || null,
          discord: matchingPair?.info?.socials?.find((s: any) => s.type === "discord")?.url || null,
          baseToken: {
            name: matchingPair?.baseToken?.name || null,
            symbol: matchingPair?.baseToken?.symbol || null,
          },
          technical: {
            priceUsd: matchingPair?.priceUsd || null,
            priceChange5m: priceChange.m5 || 0,
            priceChange1h: priceChange.h1 || 0,
            priceChange6h: priceChange.h6 || 0,
            priceChange24h: priceChange.h24 || 0,
            volume24h: matchingPair?.volume?.h24 || 0,
            liquidity: matchingPair?.liquidity?.usd || 0,
            marketCap: matchingPair?.marketCap || matchingPair?.fdv || 0,
          },
          sentiment: {
            overall: sentiment,
            buyPressure: `${buyPressure}%`,
            buys24h,
            sells24h,
            totalTxns24h: totalTxns,
          },
        };
      }
    }
  } catch (e) {
    console.log("DexScreener fetch failed, continuing without social data");
  }
  return null;
}

async function analyzeEvmToken(tokenAddress: string, chainId: string) {
  const [goPlusResponse, dexScreenerData] = await Promise.all([
    fetch(`https://api.gopluslabs.io/api/v1/token_security/${chainId}?contract_addresses=${tokenAddress}`, {
      headers: { "Accept": "application/json" },
    }),
    fetchDexScreenerData(tokenAddress, chainId),
  ]);

  if (!goPlusResponse.ok) {
    throw new Error("Failed to fetch token data from GoPlus");
  }

  const data = await goPlusResponse.json();
  
  if (data.code !== 1 || !data.result || !data.result[tokenAddress.toLowerCase()]) {
    throw new Error("Token not found or invalid address");
  }

  const tokenData = data.result[tokenAddress.toLowerCase()];
  const checksumAddress = tokenAddress;

  const socials: Record<string, string> = {};
  if (dexScreenerData?.website) socials.website = dexScreenerData.website;
  if (dexScreenerData?.twitter) socials.twitter = dexScreenerData.twitter;
  if (dexScreenerData?.telegram) socials.telegram = dexScreenerData.telegram;
  if (dexScreenerData?.discord) socials.discord = dexScreenerData.discord;

  return {
    chain: CHAIN_NAMES[chainId] || `Chain ${chainId}`,
    address: tokenAddress,
    name: tokenData.token_name || "Unknown",
    symbol: tokenData.token_symbol || "N/A",
    logo: dexScreenerData?.logo || getTokenLogo(checksumAddress, chainId),
    socials,
    technical: dexScreenerData?.technical || null,
    sentiment: dexScreenerData?.sentiment || null,
    simulation: {
      buyTax: tokenData.buy_tax ? `${(parseFloat(tokenData.buy_tax) * 100).toFixed(1)}%` : "0.0%",
      sellTax: tokenData.sell_tax ? `${(parseFloat(tokenData.sell_tax) * 100).toFixed(1)}%` : "0.0%",
      transferTax: tokenData.transfer_tax ? `${(parseFloat(tokenData.transfer_tax) * 100).toFixed(1)}%` : "0.0%",
      buyGas: tokenData.buy_gas || "N/A",
      sellGas: tokenData.sell_gas || "N/A",
      buyLimit: tokenData.cannot_buy === "1" ? "BLOCKED" : "NONE DETECTED",
      sellLimit: tokenData.cannot_sell_all === "1" ? "RESTRICTED" : "NONE DETECTED",
      isHoneypot: tokenData.is_honeypot === "1",
      sourceCode: tokenData.is_open_source === "1" ? "OPEN SOURCE" : "NOT VERIFIED",
    },
    holders: {
      total: parseInt(tokenData.holder_count) || 0,
      canSell: parseInt(tokenData.holder_count) - (parseInt(tokenData.holders_cannot_sell_count) || 0),
      cantSell: parseInt(tokenData.holders_cannot_sell_count) || 0,
      siphoned: 0,
      averageTax: tokenData.sell_tax ? `${(parseFloat(tokenData.sell_tax) * 100).toFixed(1)}%` : "0.0%",
      highestTax: tokenData.sell_tax ? `${(parseFloat(tokenData.sell_tax) * 100).toFixed(1)}%` : "0.0%",
      averageGas: tokenData.sell_gas || "N/A",
    },
    security: {
      isProxy: tokenData.is_proxy === "1",
      isMintable: tokenData.is_mintable === "1",
      canTakeBackOwnership: tokenData.can_take_back_ownership === "1",
      ownerCanChangeBalance: tokenData.owner_change_balance === "1",
      hiddenOwner: tokenData.hidden_owner === "1",
      hasTradingCooldown: tokenData.trading_cooldown === "1",
      isBlacklisted: tokenData.is_blacklisted === "1",
      isWhitelisted: tokenData.is_whitelisted === "1",
      antiWhaleModifiable: tokenData.anti_whale_modifiable === "1",
      lpLocked: tokenData.lp_holder_count ? parseInt(tokenData.lp_holder_count) > 0 : false,
    },
    raw: tokenData,
  };
}

async function analyzeSolanaToken(tokenAddress: string) {
  const HELIUS_API_KEY = process.env.HELIUS_API_KEY;
  if (!HELIUS_API_KEY) {
    throw new Error("Helius API key not configured");
  }

  try {
    const [metadataResponse, dexScreenerData] = await Promise.all([
      fetch(
        `https://api.helius.xyz/v0/token-metadata?api-key=${HELIUS_API_KEY}`,
        {
          method: "POST",
          headers: { "Content-Type": "application/json" },
          body: JSON.stringify({ mintAccounts: [tokenAddress] }),
        }
      ),
      fetchDexScreenerData(tokenAddress, "solana")
    ]);

    if (!metadataResponse.ok) {
      throw new Error("Failed to fetch token metadata from Helius");
    }

    const metadataData = await metadataResponse.json();
    const tokenInfo = metadataData && metadataData.length > 0 ? metadataData[0] : null;
    
    const parsedInfo = tokenInfo?.onChainAccountInfo?.accountInfo?.data?.parsed?.info;
    const offChainMeta = tokenInfo?.offChainMetadata?.metadata;
    const legacyMeta = tokenInfo?.legacyMetadata;

    const socials: Record<string, string> = {};
    if (dexScreenerData?.website) socials.website = dexScreenerData.website;
    else if (offChainMeta?.external_url) socials.website = offChainMeta.external_url;
    else if (legacyMeta?.website) socials.website = legacyMeta.website;

    if (dexScreenerData?.twitter) socials.twitter = dexScreenerData.twitter;
    else if (legacyMeta?.twitter) socials.twitter = legacyMeta.twitter;

    if (dexScreenerData?.telegram) socials.telegram = dexScreenerData.telegram;
    else if (legacyMeta?.telegram) socials.telegram = legacyMeta.telegram;

    if (dexScreenerData?.discord) socials.discord = dexScreenerData.discord;
    else if (legacyMeta?.discord) socials.discord = legacyMeta.discord;

    return {
      chain: "Solana",
      address: tokenAddress,
      name: parsedInfo?.name || offChainMeta?.name || legacyMeta?.name || dexScreenerData?.baseToken?.name || "Unknown",
      symbol: parsedInfo?.symbol || offChainMeta?.symbol || legacyMeta?.symbol || dexScreenerData?.baseToken?.symbol || "N/A",
      logo: dexScreenerData?.logo || offChainMeta?.image || legacyMeta?.logoURI || null,
      socials,
      technical: dexScreenerData?.technical || null,
      sentiment: dexScreenerData?.sentiment || null,
      simulation: {
        buyTax: "0.0%",
        sellTax: "0.0%",
        transferTax: "0.0%",
        buyGas: "5000",
        sellGas: "5000",
        buyLimit: "NONE DETECTED",
        sellLimit: "NONE DETECTED",
        isHoneypot: false,
        sourceCode: "N/A (Solana)",
      },
      holders: {
        total: 0,
        canSell: 0,
        cantSell: 0,
        siphoned: 0,
        averageTax: "0.0%",
        highestTax: "0.0%",
        averageGas: "5000",
      },
      security: {
        mintAuthority: parsedInfo?.mintAuthority || null,
        freezeAuthority: parsedInfo?.freezeAuthority || null,
        isMintable: !!parsedInfo?.mintAuthority,
        isFreezable: !!parsedInfo?.freezeAuthority,
      },
    };
  } catch (error) {
    console.error("Solana analysis error:", error);
    throw error;
  }
}

function detectChain(address: string): "solana" | "evm" {
  if (address.startsWith("0x") && address.length === 42) {
    return "evm";
  }
  return "solana";
}

export async function registerRoutes(
  httpServer: Server,
  app: Express
): Promise<Server> {
  app.post("/api/analyze-token", async (req, res) => {
    try {
      const { tokenAddress, chain } = req.body;

      if (!tokenAddress) {
        return res.status(400).json({ error: "Token address is required" });
      }

      const detectedChain = chain || (detectChain(tokenAddress) === "evm" ? "bsc" : "solana");
      
      let result;
      
      if (detectedChain === "solana") {
        result = await analyzeSolanaToken(tokenAddress);
      } else {
        const chainId = CHAIN_IDS[detectedChain] || "56";
        result = await analyzeEvmToken(tokenAddress, chainId);
      }

      return res.json(result);
    } catch (error) {
      console.error("Token analysis error:", error);
      const message = error instanceof Error ? error.message : "Internal server error";
      return res.status(500).json({ error: message });
    }
  });

  app.get("/api/supported-chains", (_req, res) => {
    res.json({
      chains: [
        { id: "solana", name: "Solana", icon: "☀️" },
        { id: "ethereum", name: "Ethereum", icon: "⟠" },
        { id: "bsc", name: "BSC", icon: "🔶" },
        { id: "polygon", name: "Polygon", icon: "🟣" },
        { id: "arbitrum", name: "Arbitrum", icon: "🔵" },
        { id: "base", name: "Base", icon: "🔷" },
        { id: "avalanche", name: "Avalanche", icon: "🔺" },
        { id: "fantom", name: "Fantom", icon: "👻" },
        { id: "optimism", name: "Optimism", icon: "🔴" },
      ],
    });
  });

  // Free Smart Contract Audit API (BSC BEP-20)
  app.get("/api/audit/:contractAddress", async (req, res) => {
    try {
      const { contractAddress } = req.params;
      
      if (!contractAddress || !contractAddress.startsWith("0x")) {
        return res.status(400).json({ error: "Invalid contract address" });
      }

      // Fetch security data from GoPlus for BSC (chain id 56)
      const goplusResponse = await fetch(
        `https://api.gopluslabs.io/api/v1/token_security/56?contract_addresses=${contractAddress}`
      );
      
      if (!goplusResponse.ok) {
        throw new Error("Failed to fetch security data");
      }

      const goplusData = await goplusResponse.json();
      const tokenData = goplusData.result?.[contractAddress.toLowerCase()];

      if (!tokenData) {
        return res.status(404).json({ error: "Contract not found or not verified" });
      }

      // Transform GoPlus data into audit findings
      const findings: any[] = [];
      let critical = 0, major = 0, medium = 0, minor = 0, info = 0;

      // Check for critical issues
      if (tokenData.is_honeypot === "1") {
        findings.push({
          severity: "critical",
          title: "Honeypot detected - cannot sell tokens",
          description: "This contract is identified as a honeypot. Users cannot sell tokens after purchase.",
          type: "honeypot-detection",
          confidence: "high",
          file: "contract.sol",
        });
        critical++;
      }

      if (tokenData.is_blacklisted === "1") {
        findings.push({
          severity: "critical",
          title: "Contract is blacklisted",
          description: "This contract has been blacklisted for malicious activity.",
          type: "blacklist-status",
          confidence: "high",
          file: "contract.sol",
        });
        critical++;
      }

      // Check for major issues
      if (tokenData.owner_change_balance === "1") {
        findings.push({
          severity: "major",
          title: "Owner can modify token balances",
          description: "The contract owner has the ability to change user balances arbitrarily.",
          type: "centralization-risk",
          confidence: "high",
          file: "contract.sol",
          lines: "owner functions",
        });
        major++;
      }

      if (tokenData.hidden_owner === "1") {
        findings.push({
          severity: "major",
          title: "Hidden owner mechanism detected",
          description: "The contract has hidden owner functionality that may allow unauthorized control.",
          type: "hidden-owner",
          confidence: "high",
          file: "contract.sol",
        });
        major++;
      }

      if (tokenData.can_take_back_ownership === "1") {
        findings.push({
          severity: "major",
          title: "Ownership can be reclaimed",
          description: "Previous owner can reclaim ownership after renouncing.",
          type: "ownership-risk",
          confidence: "high",
          file: "contract.sol",
        });
        major++;
      }

      if (tokenData.selfdestruct === "1") {
        findings.push({
          severity: "major",
          title: "Contract can self-destruct",
          description: "The contract contains selfdestruct functionality which could destroy all funds.",
          type: "selfdestruct",
          confidence: "high",
          file: "contract.sol",
        });
        major++;
      }

      if (tokenData.external_call === "1") {
        findings.push({
          severity: "major",
          title: "External calls detected",
          description: "Contract makes external calls which could be exploited.",
          type: "external-call",
          confidence: "medium",
          file: "contract.sol",
        });
        major++;
      }

      // Check for medium issues
      if (tokenData.is_mintable === "1") {
        findings.push({
          severity: "medium",
          title: "Token is mintable",
          description: "New tokens can be minted, potentially diluting holder value.",
          type: "mint-function",
          confidence: "high",
          file: "contract.sol",
        });
        medium++;
      }

      if (tokenData.trading_cooldown === "1") {
        findings.push({
          severity: "medium",
          title: "Trading cooldown enabled",
          description: "There is a cooldown period between trades which may affect trading.",
          type: "trading-restriction",
          confidence: "high",
          file: "contract.sol",
        });
        medium++;
      }

      if (tokenData.transfer_pausable === "1") {
        findings.push({
          severity: "medium",
          title: "Transfers can be paused",
          description: "Token transfers can be paused by the owner.",
          type: "pausable",
          confidence: "high",
          file: "contract.sol",
        });
        medium++;
      }

      if (tokenData.anti_whale_modifiable === "1") {
        findings.push({
          severity: "medium",
          title: "Anti-whale limits can be modified",
          description: "The anti-whale restrictions can be changed by the owner.",
          type: "modifiable-limits",
          confidence: "medium",
          file: "contract.sol",
        });
        medium++;
      }

      // Check for minor issues
      if (parseFloat(tokenData.buy_tax || "0") > 5) {
        findings.push({
          severity: "minor",
          title: `High buy tax detected: ${tokenData.buy_tax}%`,
          description: "The buy tax is higher than typical, which affects profitability.",
          type: "high-tax",
          confidence: "high",
          file: "contract.sol",
        });
        minor++;
      }

      if (parseFloat(tokenData.sell_tax || "0") > 5) {
        findings.push({
          severity: "minor",
          title: `High sell tax detected: ${tokenData.sell_tax}%`,
          description: "The sell tax is higher than typical, which affects profitability.",
          type: "high-tax",
          confidence: "high",
          file: "contract.sol",
        });
        minor++;
      }

      if (tokenData.is_anti_whale === "1") {
        findings.push({
          severity: "minor",
          title: "Anti-whale mechanism enabled",
          description: "There are limits on maximum transaction or holding amounts.",
          type: "anti-whale",
          confidence: "high",
          file: "contract.sol",
        });
        minor++;
      }

      if (tokenData.slippage_modifiable === "1") {
        findings.push({
          severity: "minor",
          title: "Slippage can be modified",
          description: "Trading slippage/tax can be changed by the owner.",
          type: "modifiable-slippage",
          confidence: "medium",
          file: "contract.sol",
        });
        minor++;
      }

      // Info items
      if (tokenData.is_open_source === "0") {
        findings.push({
          severity: "info",
          title: "Contract source code not verified",
          description: "The contract source code is not publicly verified on BscScan.",
          type: "unverified-source",
          confidence: "high",
          file: "contract.sol",
        });
        info++;
      }

      if (tokenData.is_proxy === "1") {
        findings.push({
          severity: "info",
          title: "Proxy contract detected",
          description: "This is a proxy contract - the implementation can be upgraded.",
          type: "proxy-contract",
          confidence: "high",
          file: "contract.sol",
        });
        info++;
      }

      // Add holder concentration info
      if (tokenData.holder_count) {
        findings.push({
          severity: "info",
          title: `Token has ${tokenData.holder_count} holders`,
          description: "Total number of unique addresses holding this token.",
          type: "holder-info",
          confidence: "high",
          file: "contract.sol",
        });
        info++;
      }

      // Add ownership info
      if (tokenData.owner_address) {
        const isRenounced = tokenData.owner_address === "0x0000000000000000000000000000000000000000";
        findings.push({
          severity: "info",
          title: isRenounced ? "Ownership is renounced" : "Contract has active owner",
          description: isRenounced 
            ? "The contract ownership has been renounced to the null address."
            : `Owner address: ${tokenData.owner_address.slice(0, 10)}...`,
          type: "ownership-info",
          confidence: "high",
          file: "contract.sol",
        });
        info++;
      }

      // Calculate audit score (100 = perfect, deductions based on severity)
      let auditScore = 100;
      auditScore -= critical * 25; // Critical issues: -25 each
      auditScore -= major * 15;    // Major issues: -15 each
      auditScore -= medium * 8;    // Medium issues: -8 each
      auditScore -= minor * 3;     // Minor issues: -3 each
      auditScore -= info * 0;      // Info: no deduction
      auditScore = Math.max(0, Math.min(100, auditScore)); // Clamp 0-100

      // Determine grade
      let grade = "A+";
      if (auditScore < 95) grade = "A";
      if (auditScore < 90) grade = "A-";
      if (auditScore < 85) grade = "B+";
      if (auditScore < 80) grade = "B";
      if (auditScore < 75) grade = "B-";
      if (auditScore < 70) grade = "C+";
      if (auditScore < 65) grade = "C";
      if (auditScore < 60) grade = "C-";
      if (auditScore < 55) grade = "D";
      if (auditScore < 50) grade = "F";

      return res.json({
        success: true,
        contractAddress,
        chain: "BSC",
        tokenName: tokenData.token_name || "Unknown",
        tokenSymbol: tokenData.token_symbol || "???",
        auditScore,
        grade,
        summary: {
          critical,
          major,
          medium,
          minor,
          info,
        },
        findings,
        rawData: {
          totalSupply: tokenData.total_supply,
          holderCount: tokenData.holder_count,
          lpHolderCount: tokenData.lp_holder_count,
          buyTax: tokenData.buy_tax,
          sellTax: tokenData.sell_tax,
          isOpenSource: tokenData.is_open_source === "1",
          isProxy: tokenData.is_proxy === "1",
          isMintable: tokenData.is_mintable === "1",
          ownerAddress: tokenData.owner_address,
        },
      });
    } catch (error) {
      console.error("Contract audit error:", error);
      const message = error instanceof Error ? error.message : "Failed to audit contract";
      return res.status(500).json({ error: message });
    }
  });

  // Four.meme Token Generator API endpoints
  const FOUR_MEME_API = "https://four.meme/meme-api";
  const FEE_WALLET = "0xa6652bd6c6ff77ebae15a14861467a5366918504";

  // Get nonce for wallet authentication
  app.post("/api/token-generator/nonce", async (req, res) => {
    try {
      const { walletAddress } = req.body;
      
      if (!walletAddress) {
        return res.status(400).json({ error: "Wallet address is required" });
      }

      const response = await fetch(`${FOUR_MEME_API}/v1/private/user/nonce/generate`, {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({
          accountAddress: walletAddress,
          verifyType: "LOGIN",
          networkCode: "BSC"
        }),
      });

      const data = await response.json();
      
      if (data.code !== "0") {
        return res.status(400).json({ error: data.msg || "Failed to generate nonce" });
      }

      return res.json({ nonce: data.data });
    } catch (error) {
      console.error("Nonce generation error:", error);
      return res.status(500).json({ error: "Failed to generate nonce" });
    }
  });

  // Login with wallet signature
  app.post("/api/token-generator/login", async (req, res) => {
    try {
      const { walletAddress, signature, nonce } = req.body;
      
      if (!walletAddress || !signature) {
        return res.status(400).json({ error: "Wallet address and signature are required" });
      }

      const response = await fetch(`${FOUR_MEME_API}/v1/private/user/login/dex`, {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({
          region: "WEB",
          langType: "EN",
          loginIp: "",
          inviteCode: "",
          verifyInfo: {
            address: walletAddress,
            networkCode: "BSC",
            signature: signature,
            verifyType: "LOGIN"
          },
          walletName: "MetaMask"
        }),
      });

      const data = await response.json();
      
      if (data.code !== "0") {
        return res.status(400).json({ error: data.msg || "Login failed" });
      }

      return res.json({ accessToken: data.data });
    } catch (error) {
      console.error("Login error:", error);
      return res.status(500).json({ error: "Login failed" });
    }
  });

  // Get token creation config
  app.get("/api/token-generator/config", async (_req, res) => {
    try {
      const response = await fetch(`${FOUR_MEME_API}/v1/public/config`);
      const data = await response.json();
      
      return res.json({
        ...data,
        feeWallet: FEE_WALLET,
        deployCost: "0.01",
        labels: ["Meme", "AI", "DeFi", "Games", "Infra", "De-Sci", "Social", "DePIN", "Charity", "Others"],
        fixedParams: {
          totalSupply: "1000000000",
          raisedAmount: "24",
          saleRate: "0.8",
          lpTradingFee: "0.0025",
        }
      });
    } catch (error) {
      console.error("Config fetch error:", error);
      return res.status(500).json({ error: "Failed to fetch config" });
    }
  });

  // Create token parameters (returns signature for blockchain interaction)
  app.post("/api/token-generator/create", async (req, res) => {
    try {
      const { accessToken, tokenData } = req.body;
      
      if (!accessToken || !tokenData) {
        return res.status(400).json({ error: "Access token and token data are required" });
      }

      const createPayload = {
        name: tokenData.name,
        shortName: tokenData.symbol,
        desc: tokenData.description,
        imgUrl: tokenData.imageUrl,
        launchTime: Date.now(),
        label: tokenData.label || "Meme",
        lpTradingFee: 0.0025,
        webUrl: tokenData.website || "",
        twitterUrl: tokenData.twitter || "",
        telegramUrl: tokenData.telegram || "",
        preSale: tokenData.preSale || "0",
        onlyMPC: false,
        totalSupply: 1000000000,
        raisedAmount: 24,
        saleRate: 0.8,
        reserveRate: 0,
        funGroup: false,
        clickFun: false,
        symbol: "BNB",
        raisedToken: {
          symbol: "BNB",
          nativeSymbol: "BNB",
          symbolAddress: "0xbb4cdb9cbd36b01bd1cbaebf2de08d9173bc095c",
          deployCost: "0",
          buyFee: "0.01",
          sellFee: "0.01",
          minTradeFee: "0",
          b0Amount: "8",
          totalBAmount: "24",
          totalAmount: "1000000000",
          tradeLevel: ["0.1", "0.5", "1"],
          status: "PUBLISH",
          reservedNumber: 10,
          saleRate: "0.8",
          networkCode: "BSC",
          platform: "MEME"
        }
      };

      const response = await fetch(`${FOUR_MEME_API}/v1/private/token/create`, {
        method: "POST",
        headers: { 
          "Content-Type": "application/json",
          "meme-web-access": accessToken,
        },
        body: JSON.stringify(createPayload),
      });

      const data = await response.json();
      
      if (data.code !== "0") {
        return res.status(400).json({ error: data.msg || "Token creation failed" });
      }

      return res.json({
        success: true,
        createArg: data.data?.createArg,
        signature: data.data?.signature,
        contractAddress: "0x277204675524B49D417bD0B374A1FB09465F7777",
        feeWallet: FEE_WALLET,
      });
    } catch (error) {
      console.error("Token creation error:", error);
      return res.status(500).json({ error: "Token creation failed" });
    }
  });

  // Block Explorer API - BSC Block Explorer using NodeReal
  const NODEREAL_BSC_ENDPOINT = "https://bsc-mainnet.nodereal.io/v1/64a9df0874fb4a93b9d0a3849de012d3";
  
  // Explorer homepage - latest blocks and transactions
  app.get("/api/explorer", async (req, res) => {
    try {
      // Get latest block number
      const blockNumRes = await fetch(NODEREAL_BSC_ENDPOINT, {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({
          jsonrpc: "2.0",
          method: "eth_blockNumber",
          params: [],
          id: 1,
        }),
      });
      const blockNumData = await blockNumRes.json();
      const latestBlockNum = parseInt(blockNumData.result, 16);
      
      // Get latest 10 blocks
      const blockPromises = [];
      for (let i = 0; i < 10; i++) {
        const blockNum = latestBlockNum - i;
        blockPromises.push(
          fetch(NODEREAL_BSC_ENDPOINT, {
            method: "POST",
            headers: { "Content-Type": "application/json" },
            body: JSON.stringify({
              jsonrpc: "2.0",
              method: "eth_getBlockByNumber",
              params: [`0x${blockNum.toString(16)}`, false],
              id: i + 10,
            }),
          }).then(r => r.json())
        );
      }
      const blockResults = await Promise.all(blockPromises);
      
      const blocks = blockResults.map((b: any) => {
        const block = b.result;
        if (!block) return null;
        const blockNum = parseInt(block.number, 16);
        const timestamp = parseInt(block.timestamp, 16);
        const txCount = block.transactions?.length || 0;
        const reward = (parseInt(block.gasUsed, 16) * parseInt(block.baseFeePerGas || "0", 16) / 1e18).toFixed(8);
        return {
          number: blockNum,
          timestamp,
          timeAgo: getTimeAgo(timestamp),
          validator: block.miner,
          txCount,
          reward: reward !== "NaN" ? reward : "0.00050000",
          gasUsed: parseInt(block.gasUsed, 16),
        };
      }).filter(Boolean);
      
      // Get latest transactions from recent block
      const latestBlockRes = await fetch(NODEREAL_BSC_ENDPOINT, {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({
          jsonrpc: "2.0",
          method: "eth_getBlockByNumber",
          params: ["latest", true],
          id: 100,
        }),
      });
      const latestBlockData = await latestBlockRes.json();
      const latestTxs = (latestBlockData.result?.transactions || []).slice(0, 10).map((tx: any) => ({
        hash: tx.hash,
        from: tx.from,
        to: tx.to,
        value: (parseInt(tx.value, 16) / 1e18).toFixed(8),
        blockNumber: parseInt(tx.blockNumber, 16),
        timestamp: parseInt(latestBlockData.result.timestamp, 16),
        timeAgo: getTimeAgo(parseInt(latestBlockData.result.timestamp, 16)),
      }));
      
      // Get BNB price from CoinGecko
      let bnbPrice = 0;
      try {
        const priceRes = await fetch("https://api.coingecko.com/api/v3/simple/price?ids=binancecoin&vs_currencies=usd&include_24hr_change=true");
        const priceData = await priceRes.json();
        bnbPrice = priceData.binancecoin?.usd || 0;
      } catch (e) {
        bnbPrice = 600; // Fallback
      }
      
      return res.json({
        latestBlock: latestBlockNum,
        bnbPrice,
        activeValidators: 45,
        votingPower: "25,291,104.81",
        blocks,
        transactions: latestTxs,
      });
    } catch (error) {
      console.error("Explorer homepage error:", error);
      return res.status(500).json({ error: "Failed to fetch explorer data" });
    }
  });
  
  // Helper function for time ago
  function getTimeAgo(timestamp: number): string {
    const now = Math.floor(Date.now() / 1000);
    const diff = now - timestamp;
    if (diff < 60) return `${diff}s ago`;
    if (diff < 3600) return `${Math.floor(diff / 60)}m ago`;
    if (diff < 86400) return `${Math.floor(diff / 3600)}h ago`;
    return `${Math.floor(diff / 86400)}d ago`;
  }
  
  app.get("/api/explorer/:address", async (req, res) => {
    try {
      const { address } = req.params;
      
      if (!address || !/^0x[a-fA-F0-9]{40}$/.test(address)) {
        return res.status(400).json({ error: "Invalid BSC address" });
      }
      
      // Get BNB balance
      const balanceRes = await fetch(NODEREAL_BSC_ENDPOINT, {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({
          jsonrpc: "2.0",
          method: "eth_getBalance",
          params: [address, "latest"],
          id: 1,
        }),
      });
      const balanceData = await balanceRes.json();
      const bnbBalance = balanceData.result 
        ? (parseInt(balanceData.result, 16) / 1e18).toFixed(6) 
        : "0";
      
      // Get token holdings using nr_getTokenHoldings
      const holdingsRes = await fetch(NODEREAL_BSC_ENDPOINT, {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({
          jsonrpc: "2.0",
          method: "nr_getTokenHoldings",
          params: [address, "0x1", "0x14"], // page 1, 20 items
          id: 2,
        }),
      });
      const holdingsData = await holdingsRes.json();
      
      const tokenHoldings = holdingsData.result?.details?.map((token: any) => ({
        symbol: token.tokenSymbol || "Unknown",
        name: token.tokenName || "Unknown Token",
        balance: token.tokenBalance ? (parseInt(token.tokenBalance, 16) / Math.pow(10, parseInt(token.tokenDecimal || "18", 16))).toFixed(4) : "0",
        contract: token.tokenAddress,
        decimals: parseInt(token.tokenDecimal || "18", 16),
      })) || [];
      
      // Get recent transactions using NodeReal API
      const txRes = await fetch(NODEREAL_BSC_ENDPOINT, {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({
          jsonrpc: "2.0",
          method: "nr_getTransactionByAddress",
          params: [{
            category: ["external", "internal", "20"],
            addressType: null,
            address: address,
            order: "desc",
            excludeZeroValue: false,
            maxCount: "0x14",
            fromBlock: "0x1",
            toBlock: "latest",
          }],
          id: 3,
        }),
      });
      const txData = await txRes.json();
      
      const now = Date.now();
      let transactions: any[] = [];
      
      if (txData.result?.transfers && Array.isArray(txData.result.transfers)) {
        transactions = txData.result.transfers.map((tx: any) => {
          const timestamp = tx.blockTimestamp ? parseInt(tx.blockTimestamp, 16) * 1000 : now;
          const diffMs = now - timestamp;
          const diffMins = Math.floor(diffMs / 60000);
          let timeAgo = "Just now";
          if (diffMins >= 60 * 24) {
            timeAgo = `${Math.floor(diffMins / (60 * 24))}d ago`;
          } else if (diffMins >= 60) {
            timeAgo = `${Math.floor(diffMins / 60)}h ago`;
          } else if (diffMins > 0) {
            timeAgo = `${diffMins}m ago`;
          }
          
          const gasUsed = tx.gasUsed ? parseInt(tx.gasUsed, 16) : 21000;
          const gasPrice = tx.gasPrice ? parseInt(tx.gasPrice, 16) : 3000000000;
          const txnFee = ((gasUsed * gasPrice) / 1e18).toFixed(8);
          
          let method = tx.category === "20" ? "Transfer" : (tx.functionName?.split("(")[0] || "Transfer");
          
          return {
            hash: tx.hash,
            from: tx.from,
            to: tx.to,
            value: tx.value ? (parseInt(tx.value, 16) / 1e18).toFixed(6) : "0",
            asset: tx.asset || "BNB",
            blockNumber: parseInt(tx.blockNumber || "0", 16),
            timestamp: new Date(timestamp).toISOString(),
            timeAgo,
            type: tx.from?.toLowerCase() === address.toLowerCase() ? "out" : "in",
            method,
            txnFee,
          };
        });
      }
      
      // Get transaction count
      const txCountRes = await fetch(NODEREAL_BSC_ENDPOINT, {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({
          jsonrpc: "2.0",
          method: "eth_getTransactionCount",
          params: [address, "latest"],
          id: 4,
        }),
      });
      const txCountData = await txCountRes.json();
      const txCount = txCountData.result ? parseInt(txCountData.result, 16) : 0;
      
      // Check if address is a contract
      const codeRes = await fetch(NODEREAL_BSC_ENDPOINT, {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({
          jsonrpc: "2.0",
          method: "eth_getCode",
          params: [address, "latest"],
          id: 5,
        }),
      });
      const codeData = await codeRes.json();
      const isContract = codeData.result && codeData.result !== "0x";
      
      return res.json({
        address,
        isContract,
        bnbBalance,
        txCount,
        tokenHoldings,
        transactions,
        bscscanUrl: `https://bscscan.com/address/${address}`,
      });
    } catch (error) {
      console.error("Block explorer error:", error);
      return res.status(500).json({ error: "Failed to fetch address data" });
    }
  });

  // Bundle Checker API - Detect bundle signals in Solana and BSC token launches
  app.post("/api/bundle-checker", async (req, res) => {
    try {
      const { mintAddress, chain = "solana" } = req.body;

      if (!mintAddress) {
        return res.status(400).json({ error: "Token address is required" });
      }

      if (chain === "bsc") {
        // BSC Bundle Analysis using GoPlus API and BscScan
        const goPlusRes = await fetch(`https://api.gopluslabs.io/api/v1/token_security/56?contract_addresses=${mintAddress}`);
        
        let tokenInfo = null;
        let bundleAnalysis: any = {
          isBundled: false,
          confidence: 0,
          bundleCount: 0,
          details: "Analysis complete",
          signals: [],
          riskLevel: "Low",
        };

        if (goPlusRes.ok) {
          const goPlusData = await goPlusRes.json();
          const tokenData = goPlusData.result?.[mintAddress.toLowerCase()];
          
          if (tokenData) {
            tokenInfo = {
              name: tokenData.token_name || "Unknown",
              symbol: tokenData.token_symbol || "???",
              decimals: 18,
            };

            // Analyze bundle signals from GoPlus data
            const signals: string[] = [];
            let bundleScore = 0;

            // Check creator ownership percentage
            if (tokenData.creator_percent && parseFloat(tokenData.creator_percent) > 10) {
              bundleScore += 25;
              signals.push(`Creator holds ${(parseFloat(tokenData.creator_percent) * 100).toFixed(1)}% of supply`);
            }

            // Check top 10 holder concentration
            if (tokenData.holder_count && parseInt(tokenData.holder_count) < 50) {
              bundleScore += 20;
              signals.push(`Low holder count: ${tokenData.holder_count} holders`);
            }

            // Check if honeypot
            if (tokenData.is_honeypot === "1") {
              bundleScore += 30;
              signals.push("Token flagged as honeypot");
            }

            // Check sell tax
            if (tokenData.sell_tax && parseFloat(tokenData.sell_tax) > 0.1) {
              bundleScore += 15;
              signals.push(`High sell tax: ${(parseFloat(tokenData.sell_tax) * 100).toFixed(1)}%`);
            }

            // Check if trading is open
            if (tokenData.is_open_source === "0") {
              bundleScore += 10;
              signals.push("Contract is not open source");
            }

            // Check owner privileges
            if (tokenData.can_take_back_ownership === "1" || tokenData.owner_change_balance === "1") {
              bundleScore += 20;
              signals.push("Owner has dangerous privileges");
            }

            // LP analysis
            if (tokenData.lp_holder_count && parseInt(tokenData.lp_holder_count) <= 1) {
              bundleScore += 15;
              signals.push("Single LP holder detected");
            }

            bundleAnalysis = {
              isBundled: bundleScore >= 40,
              confidence: Math.min(bundleScore, 100),
              bundleCount: 0,
              uniqueWallets: parseInt(tokenData.holder_count) || 0,
              totalTransactions: 0,
              details: bundleScore >= 40 
                ? "Bundle/coordination signals detected - exercise caution with this token"
                : "No significant bundle patterns detected",
              signals,
              riskLevel: bundleScore >= 70 ? "High" : bundleScore >= 40 ? "Medium" : "Low",
            };
          }
        }

        return res.json({
          mintAddress,
          chain: "bsc",
          tokenInfo,
          bundleAnalysis,
          transactionCount: 0,
          timestamp: new Date().toISOString(),
        });
      }

      // Solana Bundle Analysis using Helius API
      const heliusApiKey = process.env.HELIUS_API_KEY;
      if (!heliusApiKey) {
        return res.status(500).json({ error: "Helius API key not configured" });
      }

      // Fetch token metadata and transaction history
      const [metadataRes, signaturesRes] = await Promise.all([
        fetch(`https://api.helius.xyz/v0/token-metadata?api-key=${heliusApiKey}`, {
          method: "POST",
          headers: { "Content-Type": "application/json" },
          body: JSON.stringify({ mintAccounts: [mintAddress] }),
        }),
        fetch(`https://api.helius.xyz/v0/addresses/${mintAddress}/transactions?api-key=${heliusApiKey}&limit=50`),
      ]);

      let tokenInfo = null;
      if (metadataRes.ok) {
        const metadataData = await metadataRes.json();
        if (metadataData && metadataData.length > 0) {
          tokenInfo = metadataData[0];
        }
      }

      let transactions: any[] = [];
      if (signaturesRes.ok) {
        transactions = await signaturesRes.json();
      }

      // Analyze for bundle patterns
      const bundleAnalysis = analyzeBundlePatterns(transactions);

      return res.json({
        mintAddress,
        chain: "solana",
        tokenInfo: tokenInfo ? {
          name: tokenInfo.onChainMetadata?.metadata?.data?.name || tokenInfo.legacyMetadata?.name || "Unknown",
          symbol: tokenInfo.onChainMetadata?.metadata?.data?.symbol || tokenInfo.legacyMetadata?.symbol || "???",
          decimals: tokenInfo.onChainMetadata?.metadata?.tokenStandard || 9,
        } : null,
        bundleAnalysis,
        transactionCount: transactions.length,
        timestamp: new Date().toISOString(),
      });
    } catch (error) {
      console.error("Bundle checker error:", error);
      return res.status(500).json({ error: "Failed to analyze bundle" });
    }
  });

  // AI-powered token analysis using DeepSeek
  app.post("/api/ai-analyze", async (req, res) => {
    try {
      const { tokenAddress, chain, tokenData } = req.body;

      if (!tokenData) {
        return res.status(400).json({ error: "Token data is required for AI analysis" });
      }

      const analysis = await analyzeToken({
        name: tokenData.name || "Unknown",
        symbol: tokenData.symbol || "N/A",
        address: tokenAddress || tokenData.address || "",
        chain: chain || tokenData.chain || "BSC",
        price: tokenData.technical?.priceUsd ? parseFloat(tokenData.technical.priceUsd) : undefined,
        marketCap: tokenData.technical?.marketCap,
        volume24h: tokenData.technical?.volume24h,
        holders: tokenData.holders?.total,
        liquidity: tokenData.technical?.liquidity,
        contractVerified: tokenData.simulation?.sourceCode === "OPEN SOURCE",
        topHolderPercentage: undefined,
      });

      return res.json({
        success: true,
        analysis,
        timestamp: new Date().toISOString(),
      });
    } catch (error) {
      console.error("AI analysis error:", error);
      const message = error instanceof Error ? error.message : "AI analysis failed";
      return res.status(500).json({ error: message });
    }
  });

  // AI Chat endpoint
  app.post("/api/ai-chat", async (req, res) => {
    try {
      const { messages, systemPrompt } = req.body;

      if (!messages || !Array.isArray(messages)) {
        return res.status(400).json({ error: "Messages array is required" });
      }

      const response = await chatWithAI(messages, systemPrompt);

      return res.json({
        success: true,
        response,
        timestamp: new Date().toISOString(),
      });
    } catch (error) {
      console.error("AI chat error:", error);
      const message = error instanceof Error ? error.message : "AI chat failed";
      return res.status(500).json({ error: message });
    }
  });

  return httpServer;
}

// Helper function to analyze bundle patterns
function analyzeBundlePatterns(transactions: any[]) {
  if (!transactions || transactions.length === 0) {
    return {
      isBundled: false,
      confidence: 0,
      bundleCount: 0,
      details: "No transactions found to analyze",
      signals: [],
    };
  }

  const signals: string[] = [];
  let bundleScore = 0;

  // Group transactions by slot (same block = potential bundle)
  const slotGroups: Record<number, any[]> = {};
  transactions.forEach((tx) => {
    const slot = tx.slot || 0;
    if (!slotGroups[slot]) slotGroups[slot] = [];
    slotGroups[slot].push(tx);
  });

  // Check for multiple transactions in same slot
  const multiTxSlots = Object.entries(slotGroups).filter(([_, txs]) => txs.length > 2);
  if (multiTxSlots.length > 0) {
    bundleScore += 30;
    signals.push(`${multiTxSlots.length} slots with multiple transactions detected`);
  }

  // Check for rapid consecutive transactions (within first few slots)
  const sortedSlots = Object.keys(slotGroups).map(Number).sort((a, b) => a - b);
  if (sortedSlots.length >= 3) {
    const slotDiff = sortedSlots[2] - sortedSlots[0];
    if (slotDiff < 10) {
      bundleScore += 25;
      signals.push("Rapid transaction clustering in early slots");
    }
  }

  // Check for similar transaction patterns (same source wallets)
  const sourceWallets = new Set<string>();
  transactions.forEach((tx) => {
    if (tx.feePayer) sourceWallets.add(tx.feePayer);
  });

  const uniqueWalletRatio = sourceWallets.size / Math.min(transactions.length, 20);
  if (uniqueWalletRatio < 0.3) {
    bundleScore += 20;
    signals.push("Low wallet diversity suggests coordinated activity");
  }

  // Check transaction types for swap patterns
  const swapCount = transactions.filter((tx) => 
    tx.type === "SWAP" || tx.description?.toLowerCase().includes("swap")
  ).length;

  if (swapCount > 5 && swapCount / transactions.length > 0.3) {
    bundleScore += 15;
    signals.push(`High swap ratio detected (${swapCount} swaps)`);
  }

  // Determine if bundled based on score
  const isBundled = bundleScore >= 40;
  const confidence = Math.min(bundleScore, 100);

  return {
    isBundled,
    confidence,
    bundleCount: multiTxSlots.length,
    uniqueWallets: sourceWallets.size,
    totalTransactions: transactions.length,
    details: isBundled 
      ? "Bundle signals detected - this token may have been launched with coordinated transactions"
      : "No significant bundle patterns detected",
    signals,
    riskLevel: confidence >= 70 ? "High" : confidence >= 40 ? "Medium" : "Low",
  };
}
