import OpenAI from "openai";

// Initialize lazily to avoid crash on startup if API key not set
let _deepseek: OpenAI | null = null;

function getDeepseekClient(): OpenAI {
  if (!_deepseek) {
    const apiKey = process.env.DEEPSEEK_API_KEY;
    if (!apiKey) {
      throw new Error("DEEPSEEK_API_KEY environment variable is not set");
    }
    _deepseek = new OpenAI({
      apiKey,
      baseURL: process.env.DEEPSEEK_BASE_URL || "https://api.deepseek.com",
    });
  }
  return _deepseek;
}

export async function analyzeToken(tokenData: {
  name: string;
  symbol: string;
  address: string;
  chain: string;
  price?: number;
  marketCap?: number;
  volume24h?: number;
  holders?: number;
  liquidity?: number;
  contractVerified?: boolean;
  topHolderPercentage?: number;
}): Promise<{
  sentiment: { score: number; analysis: string };
  technical: { score: number; analysis: string };
  onChain: { score: number; analysis: string };
  overall: { score: number; recommendation: string };
}> {
  const prompt = `You are an expert cryptocurrency analyst. Analyze the following token and provide a comprehensive assessment.

Token Information:
- Name: ${tokenData.name}
- Symbol: ${tokenData.symbol}
- Contract Address: ${tokenData.address}
- Chain: ${tokenData.chain}
${tokenData.price ? `- Current Price: $${tokenData.price}` : ''}
${tokenData.marketCap ? `- Market Cap: $${tokenData.marketCap.toLocaleString()}` : ''}
${tokenData.volume24h ? `- 24h Volume: $${tokenData.volume24h.toLocaleString()}` : ''}
${tokenData.holders ? `- Number of Holders: ${tokenData.holders.toLocaleString()}` : ''}
${tokenData.liquidity ? `- Liquidity: $${tokenData.liquidity.toLocaleString()}` : ''}
${tokenData.contractVerified !== undefined ? `- Contract Verified: ${tokenData.contractVerified ? 'Yes' : 'No'}` : ''}
${tokenData.topHolderPercentage ? `- Top 10 Holders Own: ${tokenData.topHolderPercentage}%` : ''}

Provide your analysis in the following JSON format:
{
  "sentiment": {
    "score": <number 0-100>,
    "analysis": "<brief sentiment analysis based on market perception, social signals, and community engagement>"
  },
  "technical": {
    "score": <number 0-100>,
    "analysis": "<brief technical analysis based on price action, volume, and market metrics>"
  },
  "onChain": {
    "score": <number 0-100>,
    "analysis": "<brief on-chain analysis based on holder distribution, liquidity, and contract security>"
  },
  "overall": {
    "score": <number 0-100>,
    "recommendation": "<brief overall recommendation: Strong Buy, Buy, Hold, Sell, or Strong Sell with reasoning>"
  }
}

Respond ONLY with the JSON object, no additional text.`;

  try {
    const response = await getDeepseekClient().chat.completions.create({
      model: "deepseek-chat",
      messages: [
        {
          role: "system",
          content: "You are a cryptocurrency analysis expert. Always respond with valid JSON only."
        },
        {
          role: "user",
          content: prompt
        }
      ],
      temperature: 0.7,
      max_tokens: 1000,
    });

    const content = response.choices[0]?.message?.content || "";
    
    // Parse the JSON response
    const jsonMatch = content.match(/\{[\s\S]*\}/);
    if (!jsonMatch) {
      throw new Error("Failed to parse AI response");
    }
    
    return JSON.parse(jsonMatch[0]);
  } catch (error) {
    console.error("DeepSeek analysis error:", error);
    throw error;
  }
}

export async function chatWithAI(
  messages: Array<{ role: "user" | "assistant" | "system"; content: string }>,
  systemPrompt?: string
): Promise<string> {
  const allMessages = systemPrompt
    ? [{ role: "system" as const, content: systemPrompt }, ...messages]
    : messages;

  const response = await getDeepseekClient().chat.completions.create({
    model: "deepseek-chat",
    messages: allMessages,
    temperature: 0.7,
    max_tokens: 2000,
  });

  return response.choices[0]?.message?.content || "";
}
