import { cpSync, mkdirSync, existsSync } from 'fs';
import { execSync } from 'child_process';
import { join } from 'path';

const ROOT = process.cwd();
const STANDALONE = join(ROOT, 'standalone');
const DIST = join(ROOT, 'dist');

console.log('🚀 Building Astherafi for cPanel deployment...\n');

// Step 1: Build the project
console.log('📦 Step 1: Building project...');
try {
  execSync('npm run build', { stdio: 'inherit', cwd: ROOT });
  console.log('✅ Build completed!\n');
} catch (error) {
  console.error('❌ Build failed!');
  process.exit(1);
}

// Step 2: Check if dist folder exists
if (!existsSync(DIST)) {
  console.error('❌ dist/ folder not found. Build may have failed.');
  process.exit(1);
}

// Step 3: Copy built files to standalone
console.log('📁 Step 2: Copying files to standalone/...');

// Copy server bundle
const serverBundle = join(DIST, 'index.cjs');
if (existsSync(serverBundle)) {
  cpSync(serverBundle, join(STANDALONE, 'index.cjs'));
  console.log('  ✅ Copied index.cjs');
} else {
  console.error('  ❌ index.cjs not found in dist/');
  process.exit(1);
}

// Copy public folder
const publicSrc = join(DIST, 'public');
const publicDest = join(STANDALONE, 'public');

if (existsSync(publicSrc)) {
  // Remove old public folder
  if (existsSync(publicDest)) {
    execSync(`rm -rf "${publicDest}"`);
  }
  mkdirSync(publicDest, { recursive: true });
  cpSync(publicSrc, publicDest, { recursive: true });
  console.log('  ✅ Copied public/ folder');
} else {
  console.error('  ❌ public/ folder not found in dist/');
  process.exit(1);
}

console.log('\n✨ Deployment files ready in standalone/');
console.log('\n📋 Next steps:');
console.log('1. Upload all files from standalone/ to your cPanel');
console.log('2. Create .env file with your API keys');
console.log('3. Edit .htaccess - replace "username" with your cPanel username');
console.log('4. Setup Node.js app in cPanel and run npm install');
console.log('5. Start the application!\n');
console.log('📖 See standalone/README.txt for detailed instructions.');
