import { useState, useEffect, useRef } from "react";
import { motion } from "framer-motion";
import { useLocation } from "wouter";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Copy, Check, Search, Coins, Globe, Layers, Shield, Percent, Twitter, MessageCircle, Send, Menu, X, Zap, Link2, LockKeyhole } from "lucide-react";
import { useLanguage } from "@/lib/i18n";
import { LanguageSwitcher } from "@/components/LanguageSwitcher";
import astherfiLogo from "@assets/image_1768380685328.png";

const CONTRACT_ADDRESS = "0x277204675524B49D417bD0B374A1FB09465F7777";

export default function Landing() {
  const [, setLocation] = useLocation();
  const [homeSearchAddress, setHomeSearchAddress] = useState("");
  const [copied, setCopied] = useState(false);
  const [mobileNavOpen, setMobileNavOpen] = useState(false);
  const canvasRef = useRef<HTMLCanvasElement>(null);
  const { t, dir } = useLanguage();

  // Particle background effect
  useEffect(() => {
    const canvas = canvasRef.current;
    if (!canvas) return;

    const ctx = canvas.getContext("2d");
    if (!ctx) return;

    const resizeCanvas = () => {
      canvas.width = window.innerWidth;
      canvas.height = window.innerHeight;
    };
    resizeCanvas();
    window.addEventListener("resize", resizeCanvas);

    interface Particle {
      x: number;
      y: number;
      size: number;
      speedX: number;
      speedY: number;
      opacity: number;
    }

    const particles: Particle[] = [];
    const particleCount = 80;

    for (let i = 0; i < particleCount; i++) {
      particles.push({
        x: Math.random() * canvas.width,
        y: Math.random() * canvas.height,
        size: Math.random() * 2 + 0.5,
        speedX: (Math.random() - 0.5) * 0.3,
        speedY: (Math.random() - 0.5) * 0.3,
        opacity: Math.random() * 0.5 + 0.2,
      });
    }

    const animate = () => {
      ctx.clearRect(0, 0, canvas.width, canvas.height);

      particles.forEach((particle) => {
        particle.x += particle.speedX;
        particle.y += particle.speedY;

        if (particle.x < 0) particle.x = canvas.width;
        if (particle.x > canvas.width) particle.x = 0;
        if (particle.y < 0) particle.y = canvas.height;
        if (particle.y > canvas.height) particle.y = 0;

        ctx.beginPath();
        ctx.arc(particle.x, particle.y, particle.size, 0, Math.PI * 2);
        ctx.fillStyle = `rgba(255, 255, 255, ${particle.opacity})`;
        ctx.fill();
      });

      // Draw connections between nearby particles
      particles.forEach((p1, i) => {
        particles.slice(i + 1).forEach((p2) => {
          const dx = p1.x - p2.x;
          const dy = p1.y - p2.y;
          const distance = Math.sqrt(dx * dx + dy * dy);

          if (distance < 120) {
            ctx.beginPath();
            ctx.moveTo(p1.x, p1.y);
            ctx.lineTo(p2.x, p2.y);
            ctx.strokeStyle = `rgba(48, 208, 242, ${0.1 * (1 - distance / 120)})`;
            ctx.lineWidth = 0.5;
            ctx.stroke();
          }
        });
      });

      requestAnimationFrame(animate);
    };

    animate();

    return () => {
      window.removeEventListener("resize", resizeCanvas);
    };
  }, []);

  const scrollToSection = (id: string) => {
    const element = document.getElementById(id);
    if (element) {
      element.scrollIntoView({ behavior: "smooth" });
    }
    setMobileNavOpen(false);
  };

  const navItems = [
    { label: t("nav.home"), id: "hero" },
    { label: t("nav.tools"), id: "tools" },
    { label: t("nav.tokenomics"), id: "tokenomics" },
    { label: t("nav.roadmap"), id: "roadmap" },
    { label: t("nav.community"), id: "community" },
  ];

  const copyCA = () => {
    navigator.clipboard.writeText(CONTRACT_ADDRESS);
    setCopied(true);
    setTimeout(() => setCopied(false), 2000);
  };

  const handleHomeSearch = () => {
    if (!homeSearchAddress.trim()) return;
    // Navigate to dashboard token analyzer with the search query
    setLocation(`/dashboard?token=${encodeURIComponent(homeSearchAddress.trim())}`);
  };

  const features = [
    {
      icon: <Search className="w-6 h-6" />,
      title: t("tools.tokenAnalyzer"),
      description: t("tools.tokenAnalyzerDesc")
    },
    {
      icon: <Coins className="w-6 h-6" />,
      title: t("tools.tokenGenerator"),
      description: t("tools.tokenGeneratorDesc")
    },
    {
      icon: <Globe className="w-6 h-6" />,
      title: t("tools.websiteBuilder"),
      description: t("tools.websiteBuilderDesc")
    },
    {
      icon: <Layers className="w-6 h-6" />,
      title: t("tools.bundleChecker"),
      description: t("tools.bundleCheckerDesc")
    },
    {
      icon: <Shield className="w-6 h-6" />,
      title: t("tools.passwordScore"),
      description: t("tools.passwordScoreDesc")
    },
    {
      icon: <Percent className="w-6 h-6" />,
      title: t("tools.staking"),
      description: t("tools.stakingDesc")
    }
  ];

  return (
    <div className="w-full min-h-screen flex flex-col items-center text-center relative overflow-x-hidden" style={{ background: "linear-gradient(180deg, #0a0a0f 0%, #0d1117 30%, #0a0a0f 100%)" }}>
      {/* Particle Canvas Background */}
      <canvas
        ref={canvasRef}
        className="fixed inset-0 pointer-events-none z-0"
      />

      {/* Aurora Background Effects - Moving */}
      <div className="fixed inset-0 overflow-hidden pointer-events-none z-0">
        <div 
          className="absolute w-[800px] h-[800px] bg-gradient-to-br from-cyan-500/25 via-cyan-400/10 to-transparent rounded-full blur-3xl"
          style={{ 
            top: "-20%", 
            left: "-10%",
            animation: "aurora1 20s ease-in-out infinite"
          }} 
        />
        <div 
          className="absolute w-[600px] h-[600px] bg-gradient-to-bl from-purple-600/20 via-purple-500/10 to-transparent rounded-full blur-3xl"
          style={{ 
            top: "10%", 
            right: "-15%",
            animation: "aurora2 25s ease-in-out infinite"
          }} 
        />
        <div 
          className="absolute w-[700px] h-[700px] bg-gradient-to-tr from-blue-500/15 via-cyan-500/5 to-transparent rounded-full blur-3xl"
          style={{ 
            bottom: "0%", 
            left: "-20%",
            animation: "aurora3 22s ease-in-out infinite"
          }} 
        />
        <div 
          className="absolute w-[500px] h-[500px] bg-gradient-to-tl from-purple-400/15 via-cyan-400/10 to-transparent rounded-full blur-3xl"
          style={{ 
            bottom: "20%", 
            right: "-10%",
            animation: "aurora4 18s ease-in-out infinite"
          }} 
        />
      </div>

      {/* Wave Lines Effect - Enhanced */}
      <div className="fixed inset-0 pointer-events-none z-0 overflow-hidden">
        {/* Primary Wave Group */}
        <svg 
          className="absolute w-full h-full" 
          viewBox="0 0 1440 800" 
          preserveAspectRatio="xMidYMid slice"
          style={{ animation: "waveFloat 15s ease-in-out infinite" }}
        >
          <defs>
            <linearGradient id="waveGradient1" x1="0%" y1="0%" x2="100%" y2="0%">
              <stop offset="0%" stopColor="rgba(48, 208, 242, 0)" />
              <stop offset="30%" stopColor="rgba(48, 208, 242, 0.15)" />
              <stop offset="70%" stopColor="rgba(48, 208, 242, 0.15)" />
              <stop offset="100%" stopColor="rgba(48, 208, 242, 0)" />
            </linearGradient>
            <linearGradient id="waveGradient2" x1="0%" y1="0%" x2="100%" y2="0%">
              <stop offset="0%" stopColor="rgba(93, 92, 222, 0)" />
              <stop offset="40%" stopColor="rgba(93, 92, 222, 0.1)" />
              <stop offset="60%" stopColor="rgba(93, 92, 222, 0.1)" />
              <stop offset="100%" stopColor="rgba(93, 92, 222, 0)" />
            </linearGradient>
          </defs>
          {[...Array(30)].map((_, i) => (
            <path
              key={`wave1-${i}`}
              d={`M-100 ${350 + i * 2} C200 ${280 + i * 2 + Math.sin(i * 0.2) * 80}, 400 ${420 + i * 2 - Math.cos(i * 0.3) * 60}, 720 ${350 + i * 2} S1100 ${280 + i * 2 + Math.sin(i * 0.4) * 70}, 1540 ${350 + i * 2}`}
              fill="none"
              stroke="url(#waveGradient1)"
              strokeWidth="0.8"
              opacity={0.3 + i * 0.02}
            />
          ))}
        </svg>
        
        {/* Secondary Wave Group */}
        <svg 
          className="absolute w-full h-full" 
          viewBox="0 0 1440 800" 
          preserveAspectRatio="xMidYMid slice"
          style={{ animation: "waveFloat2 18s ease-in-out infinite reverse" }}
        >
          {[...Array(25)].map((_, i) => (
            <path
              key={`wave2-${i}`}
              d={`M-50 ${450 + i * 2.5} C300 ${380 + i * 2.5 - Math.sin(i * 0.25) * 90}, 600 ${520 + i * 2.5 + Math.cos(i * 0.35) * 70}, 900 ${450 + i * 2.5} S1200 ${380 + i * 2.5 - Math.sin(i * 0.45) * 80}, 1490 ${450 + i * 2.5}`}
              fill="none"
              stroke="url(#waveGradient2)"
              strokeWidth="0.6"
              opacity={0.25 + i * 0.015}
            />
          ))}
        </svg>

        {/* Tertiary Accent Wave */}
        <svg 
          className="absolute w-full h-full" 
          viewBox="0 0 1440 800" 
          preserveAspectRatio="xMidYMid slice"
          style={{ animation: "waveFloat3 22s ease-in-out infinite" }}
        >
          {[...Array(15)].map((_, i) => (
            <path
              key={`wave3-${i}`}
              d={`M-200 ${300 + i * 3} C150 ${220 + i * 3 + Math.sin(i * 0.3) * 100}, 450 ${380 + i * 3 - Math.cos(i * 0.4) * 80}, 750 ${300 + i * 3} S1050 ${220 + i * 3 + Math.sin(i * 0.5) * 90}, 1640 ${300 + i * 3}`}
              fill="none"
              stroke="rgba(48, 208, 242, 0.08)"
              strokeWidth="1"
              opacity={0.4 + i * 0.03}
            />
          ))}
        </svg>
      </div>

      <style>{`
        @keyframes aurora1 {
          0%, 100% { transform: translate(0, 0) scale(1); opacity: 0.6; }
          25% { transform: translate(10%, 15%) scale(1.1); opacity: 0.8; }
          50% { transform: translate(5%, 25%) scale(1.05); opacity: 0.5; }
          75% { transform: translate(-5%, 10%) scale(0.95); opacity: 0.7; }
        }
        @keyframes aurora2 {
          0%, 100% { transform: translate(0, 0) scale(1); opacity: 0.5; }
          25% { transform: translate(-15%, 10%) scale(1.05); opacity: 0.7; }
          50% { transform: translate(-20%, -5%) scale(1.1); opacity: 0.6; }
          75% { transform: translate(-5%, -10%) scale(0.95); opacity: 0.8; }
        }
        @keyframes aurora3 {
          0%, 100% { transform: translate(0, 0) scale(1); opacity: 0.5; }
          25% { transform: translate(20%, -10%) scale(1.1); opacity: 0.7; }
          50% { transform: translate(15%, -20%) scale(1); opacity: 0.6; }
          75% { transform: translate(5%, -5%) scale(1.05); opacity: 0.8; }
        }
        @keyframes aurora4 {
          0%, 100% { transform: translate(0, 0) scale(1); opacity: 0.6; }
          25% { transform: translate(-10%, -15%) scale(0.95); opacity: 0.8; }
          50% { transform: translate(-5%, 10%) scale(1.1); opacity: 0.5; }
          75% { transform: translate(10%, 5%) scale(1); opacity: 0.7; }
        }
        @keyframes waveFloat {
          0%, 100% { transform: translateX(0) translateY(0) rotate(0deg); }
          25% { transform: translateX(2%) translateY(-15px) rotate(0.5deg); }
          50% { transform: translateX(0) translateY(5px) rotate(0deg); }
          75% { transform: translateX(-2%) translateY(-10px) rotate(-0.5deg); }
        }
        @keyframes waveFloat2 {
          0%, 100% { transform: translateX(0) translateY(0) rotate(0deg); }
          25% { transform: translateX(-3%) translateY(10px) rotate(-0.3deg); }
          50% { transform: translateX(1%) translateY(-8px) rotate(0.2deg); }
          75% { transform: translateX(2%) translateY(5px) rotate(0deg); }
        }
        @keyframes waveFloat3 {
          0%, 100% { transform: translateX(0) translateY(0) scale(1); }
          33% { transform: translateX(3%) translateY(-20px) scale(1.02); }
          66% { transform: translateX(-2%) translateY(15px) scale(0.98); }
        }
      `}</style>
      {/* Header */}
      <header className="fixed top-0 left-0 right-0 z-50 bg-background/80 backdrop-blur-md border-b border-primary/10">
        <div className="max-w-6xl mx-auto px-4 h-16 flex items-center justify-between">
          <div className="flex items-center gap-2 cursor-pointer" onClick={() => scrollToSection("hero")}>
            <img src={astherfiLogo} alt="Astherafi" className="w-8 h-8 rounded-lg" />
            <span className="font-display font-bold text-base text-foreground">Astherafi</span>
          </div>
          
          {/* Desktop Nav - Centered */}
          <nav className="hidden md:flex items-center gap-8 absolute left-1/2 transform -translate-x-1/2">
            {navItems.map((item) => (
              <button
                key={item.id}
                onClick={() => scrollToSection(item.id)}
                className="text-sm font-medium text-muted-foreground hover:text-foreground transition-colors"
                data-testid={`nav-${item.id}`}
              >
                {item.label}
              </button>
            ))}
          </nav>

          {/* Right side - Language, X, Docs, Launch App */}
          <div className="flex items-center gap-2 md:gap-3">
            <LanguageSwitcher />
            <a
              href="https://x.com/astherafihq"
              target="_blank"
              rel="noopener noreferrer"
              className="text-muted-foreground hover:text-foreground transition-colors"
              data-testid="nav-twitter"
            >
              <svg stroke="currentColor" fill="currentColor" strokeWidth="0" role="img" viewBox="0 0 24 24" className="w-4 h-4" xmlns="http://www.w3.org/2000/svg"><path d="M18.901 1.153h3.68l-8.04 9.19L24 22.846h-7.406l-5.8-7.584-6.638 7.584H.474l8.6-9.83L0 1.154h7.594l5.243 6.932ZM17.61 20.644h2.039L6.486 3.24H4.298Z"></path></svg>
            </a>
            <button
              onClick={() => setLocation("/docs")}
              className="text-sm font-medium text-muted-foreground hover:text-foreground transition-colors"
              data-testid="nav-docs"
            >
              {t("nav.docs")}
            </button>
            <Button 
              onClick={() => setLocation("/dashboard")}
              size="sm"
              className="hidden md:flex font-semibold text-white border-0"
              style={{ background: "linear-gradient(135deg, rgb(48, 208, 242), rgb(93, 92, 222))" }}
              data-testid="nav-launch-app"
            >
              {t("nav.launchApp")}
            </Button>
            {/* Mobile Menu Button */}
            <Button
              variant="ghost"
              size="icon"
              className="md:hidden"
              onClick={() => setMobileNavOpen(!mobileNavOpen)}
              data-testid="button-mobile-nav"
            >
              {mobileNavOpen ? <X className="w-5 h-5" /> : <Menu className="w-5 h-5" />}
            </Button>
          </div>
        </div>

        {/* Mobile Nav */}
        {mobileNavOpen && (
          <motion.div
            initial={{ opacity: 0, y: -10 }}
            animate={{ opacity: 1, y: 0 }}
            className="md:hidden bg-background border-b border-primary/10 px-4 py-4"
          >
            <nav className="flex flex-col gap-2">
              {navItems.map((item) => (
                <button
                  key={item.id}
                  onClick={() => scrollToSection(item.id)}
                  className="text-left py-2 px-3 text-sm font-medium text-muted-foreground hover:text-foreground hover:bg-muted/50 rounded-lg transition-colors"
                >
                  {item.label}
                </button>
              ))}
              <button
                onClick={() => setLocation("/docs")}
                className="text-left py-2 px-3 text-sm font-medium text-muted-foreground hover:text-foreground hover:bg-muted/50 rounded-lg transition-colors"
              >
                {t("nav.docs")}
              </button>
              <Button 
                onClick={() => setLocation("/dashboard")}
                className="mt-2 font-semibold text-white border-0"
                style={{ background: "linear-gradient(135deg, rgb(48, 208, 242), rgb(93, 92, 222))" }}
              >
                {t("nav.launchApp")}
              </Button>
            </nav>
          </motion.div>
        )}
      </header>

      {/* Spacer for fixed header */}
      <div className="h-24" />


      <motion.div
        id="hero"
        initial={{ opacity: 0, y: 20 }}
        animate={{ opacity: 1, y: 0 }}
        transition={{ duration: 0.5 }}
        className="relative z-10 max-w-4xl w-full space-y-8 p-6 pt-10"
      >
        {/* AI Badge */}
        <div className="flex justify-center mb-6">
          <div className="flex items-center gap-2 px-4 py-2 bg-primary/10 border border-primary/30 rounded-full">
            <Zap className="w-4 h-4 text-primary" />
            <span className="text-sm font-medium text-primary">{t("hero.badgeAlt")}</span>
          </div>
        </div>

        <div className="flex justify-center mb-8">
          <img src={astherfiLogo} alt="Astherafi" className="w-24 h-24 rounded-2xl animate-[spin_8s_linear_infinite]" />
        </div>

        <h1 className="font-display font-bold text-6xl text-foreground tracking-tighter">
          Astherafi
        </h1>
        
        <h2 
          className="font-display font-bold text-4xl bg-clip-text text-transparent"
          style={{ backgroundImage: "linear-gradient(135deg, rgb(44, 206, 240), rgb(48, 208, 242), rgb(93, 92, 222))" }}
        >
          {t("hero.tagline")}
        </h2>

        <p className="text-muted-foreground text-xl max-w-2xl mx-auto leading-relaxed">
          {t("hero.description")}
        </p>

        <div className="flex flex-wrap justify-center gap-4">
          <Button 
            data-testid="button-launch-app"
            onClick={() => setLocation("/dashboard")}
            className="px-10 h-14 font-semibold text-lg text-white border-0 rounded-xl shadow-xl hover:scale-105 transition-transform"
            style={{ background: "linear-gradient(135deg, rgb(48, 208, 242), rgb(93, 92, 222))" }}
          >
            {t("hero.launchDashboard")}
          </Button>
          <Button 
            variant="outline"
            className="px-10 h-14 font-semibold text-lg rounded-xl border-primary/20 hover:bg-primary/5 transition-all"
          >
            {t("hero.viewChart")}
          </Button>
        </div>

        <div className="flex w-full max-w-2xl mx-auto mt-8 group">
          <Input
            data-testid="input-landing-search"
            placeholder={t("hero.search")}
            className="flex-1 h-14 bg-card/50 backdrop-blur-sm rounded-l-2xl rounded-r-none text-foreground placeholder:text-muted-foreground px-8 border-2 border-primary/30 focus:border-primary transition-colors"
            value={homeSearchAddress}
            onChange={(e) => setHomeSearchAddress(e.target.value)}
            onKeyDown={(e) => e.key === "Enter" && handleHomeSearch()}
          />
          <Button
            data-testid="button-landing-search"
            className="h-14 px-10 rounded-l-none rounded-r-2xl font-semibold text-white border-0"
            style={{ background: "linear-gradient(135deg, rgb(48, 208, 242), rgb(93, 92, 222))" }}
            onClick={handleHomeSearch}
          >
            {t("hero.searchBtn")}
          </Button>
        </div>

      </motion.div>

      {/* Copy CA Section */}
      <motion.div
        initial={{ opacity: 0, y: 30 }}
        animate={{ opacity: 1, y: 0 }}
        transition={{ duration: 0.5, delay: 0.2 }}
        className="relative z-10 w-full max-w-2xl mt-16 px-4"
      >
        <div className="flex flex-col items-center gap-3">
          <p className="text-sm text-muted-foreground">Contract Address (CA)</p>
          <div 
            onClick={copyCA}
            data-testid="button-copy-ca"
            className="flex items-center gap-3 px-4 py-3 bg-card/80 backdrop-blur-sm border border-primary/20 rounded-xl cursor-pointer hover:border-primary/40 transition-all group max-w-full"
          >
            <span className="font-mono text-xs sm:text-sm text-foreground/80 group-hover:text-foreground transition-colors truncate">
              {CONTRACT_ADDRESS}
            </span>
            <Button 
              size="icon" 
              variant="ghost" 
              className="h-8 w-8 flex-shrink-0 text-muted-foreground group-hover:text-primary transition-colors"
            >
              {copied ? <Check className="w-4 h-4 text-green-400" /> : <Copy className="w-4 h-4" />}
            </Button>
          </div>
          {copied && (
            <motion.p 
              initial={{ opacity: 0, y: -5 }}
              animate={{ opacity: 1, y: 0 }}
              className="text-xs text-green-400"
            >
              Copied to clipboard!
            </motion.p>
          )}
        </div>
      </motion.div>

      {/* About Section - Features Grid */}
      <motion.div
        id="tools"
        initial={{ opacity: 0, y: 40 }}
        animate={{ opacity: 1, y: 0 }}
        transition={{ duration: 0.6, delay: 0.3 }}
        className="relative z-10 w-full max-w-6xl mt-16 mb-12 px-6"
      >
        <h3 className="font-display font-bold text-3xl text-foreground mb-12">
          {t("tools.title")}
        </h3>
        <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-6">
          {features.map((feature, index) => (
            <motion.div
              key={feature.title}
              initial={{ opacity: 0, y: 20 }}
              animate={{ opacity: 1, y: 0 }}
              transition={{ duration: 0.4, delay: 0.1 * index }}
              onClick={() => setLocation("/dashboard")}
              className="p-6 bg-card/60 backdrop-blur-sm border border-primary/10 rounded-xl text-left cursor-pointer hover:border-primary/30 hover:bg-card/80 transition-all group"
            >
              <div className="w-12 h-12 rounded-lg bg-primary/10 text-primary flex items-center justify-center mb-4 group-hover:bg-primary/20 transition-colors">
                {feature.icon}
              </div>
              <h4 className="font-display font-semibold text-lg text-foreground mb-2">
                {feature.title}
              </h4>
              <p className="text-sm text-muted-foreground leading-relaxed">
                {feature.description}
              </p>
            </motion.div>
          ))}
        </div>
      </motion.div>

      {/* Tokenomics Section */}
      <motion.div
        id="tokenomics"
        initial={{ opacity: 0, y: 40 }}
        animate={{ opacity: 1, y: 0 }}
        transition={{ duration: 0.6, delay: 0.4 }}
        className="relative z-10 w-full max-w-4xl mt-16 mb-12 px-6"
      >
        <h3 className="font-display font-bold text-3xl text-foreground mb-12 text-center">
          {t("tokenomics.title")}
        </h3>
        
        <div className="grid grid-cols-2 md:grid-cols-4 gap-4 md:gap-6">
          <motion.div
            initial={{ opacity: 0, scale: 0.9 }}
            animate={{ opacity: 1, scale: 1 }}
            transition={{ duration: 0.4, delay: 0.5 }}
            className="p-6 bg-card/60 backdrop-blur-sm border border-primary/20 rounded-xl text-center"
          >
            <div className="w-12 h-12 mx-auto mb-4 rounded-full bg-primary/10 flex items-center justify-center">
              <Coins className="w-6 h-6 text-primary" />
            </div>
            <div className="text-2xl md:text-3xl font-bold text-foreground mb-1">1B</div>
            <div className="text-sm text-muted-foreground">{t("tokenomics.totalSupply")}</div>
          </motion.div>

          <motion.div
            initial={{ opacity: 0, scale: 0.9 }}
            animate={{ opacity: 1, scale: 1 }}
            transition={{ duration: 0.4, delay: 0.6 }}
            className="p-6 bg-card/60 backdrop-blur-sm border border-purple-500/20 rounded-xl text-center"
          >
            <div className="w-12 h-12 mx-auto mb-4 rounded-full bg-purple-500/10 flex items-center justify-center">
              <Percent className="w-6 h-6 text-purple-400" />
            </div>
            <div className="text-2xl md:text-3xl font-bold text-foreground mb-1">3%</div>
            <div className="text-sm text-muted-foreground">{t("tokenomics.tax")}</div>
          </motion.div>

          <motion.div
            initial={{ opacity: 0, scale: 0.9 }}
            animate={{ opacity: 1, scale: 1 }}
            transition={{ duration: 0.4, delay: 0.7 }}
            className="p-6 bg-card/60 backdrop-blur-sm border border-yellow-500/20 rounded-xl text-center"
          >
            <div className="w-12 h-12 mx-auto mb-4 rounded-full bg-yellow-500/10 flex items-center justify-center">
              <Link2 className="w-6 h-6 text-yellow-400" />
            </div>
            <div className="text-2xl md:text-3xl font-bold text-foreground mb-1">BSC</div>
            <div className="text-sm text-muted-foreground">{t("tokenomics.chain")}</div>
          </motion.div>

          <motion.div
            initial={{ opacity: 0, scale: 0.9 }}
            animate={{ opacity: 1, scale: 1 }}
            transition={{ duration: 0.4, delay: 0.8 }}
            className="p-6 bg-card/60 backdrop-blur-sm border border-green-500/20 rounded-xl text-center"
          >
            <div className="w-12 h-12 mx-auto mb-4 rounded-full bg-green-500/10 flex items-center justify-center">
              <LockKeyhole className="w-6 h-6 text-green-400" />
            </div>
            <div className="text-2xl md:text-3xl font-bold text-foreground mb-1">{t("tokenomics.locked")}</div>
            <div className="text-sm text-muted-foreground">{t("tokenomics.liquidity")}</div>
          </motion.div>
        </div>
      </motion.div>

      {/* Roadmap Section */}
      <motion.div
        id="roadmap"
        initial={{ opacity: 0, y: 40 }}
        animate={{ opacity: 1, y: 0 }}
        transition={{ duration: 0.6, delay: 0.5 }}
        className="relative z-10 w-full max-w-4xl mt-16 mb-20 px-6"
      >
        <h3 className="font-display font-bold text-3xl text-foreground mb-12">
          {t("roadmap.title")}
        </h3>
        
        <div className="relative">
          {/* Vertical Line */}
          <div className="absolute left-1/2 transform -translate-x-1/2 w-1 h-full bg-gradient-to-b from-primary/50 via-secondary/50 to-primary/20 rounded-full hidden md:block" />
          
          {/* Phase 1 */}
          <motion.div 
            initial={{ opacity: 0, x: -30 }}
            animate={{ opacity: 1, x: 0 }}
            transition={{ duration: 0.4, delay: 0.6 }}
            className="relative flex flex-col md:flex-row items-center mb-12"
          >
            <div className="md:w-1/2 md:pr-12 md:text-right mb-4 md:mb-0">
              <div className="p-6 bg-card/60 backdrop-blur-sm border border-primary/20 rounded-xl">
                <span className="text-xs font-semibold text-primary uppercase tracking-wider">{t("roadmap.phase1")}</span>
                <h4 className="font-display font-bold text-xl text-foreground mt-2 mb-3">{t("roadmap.foundation")}</h4>
                <ul className="text-sm text-muted-foreground space-y-2 text-left md:text-right">
                  <li>{t("roadmap.phase1Item1")}</li>
                  <li>{t("roadmap.phase1Item2")}</li>
                  <li>{t("roadmap.phase1Item3")}</li>
                  <li>{t("roadmap.phase1Item4")}</li>
                </ul>
              </div>
            </div>
            <div className="w-4 h-4 bg-primary rounded-full border-4 border-background shadow-lg shadow-primary/30 z-10 hidden md:block" />
            <div className="md:w-1/2 md:pl-12 hidden md:block" />
          </motion.div>

          {/* Phase 2 */}
          <motion.div 
            initial={{ opacity: 0, x: 30 }}
            animate={{ opacity: 1, x: 0 }}
            transition={{ duration: 0.4, delay: 0.7 }}
            className="relative flex flex-col md:flex-row items-center mb-12"
          >
            <div className="md:w-1/2 md:pr-12 hidden md:block" />
            <div className="w-4 h-4 bg-secondary rounded-full border-4 border-background shadow-lg shadow-secondary/30 z-10 hidden md:block" />
            <div className="md:w-1/2 md:pl-12 md:text-left mb-4 md:mb-0">
              <div className="p-6 bg-card/60 backdrop-blur-sm border border-secondary/20 rounded-xl">
                <span className="text-xs font-semibold text-secondary uppercase tracking-wider">{t("roadmap.phase2")}</span>
                <h4 className="font-display font-bold text-xl text-foreground mt-2 mb-3">{t("roadmap.expansion")}</h4>
                <ul className="text-sm text-muted-foreground space-y-2 text-left">
                  <li>{t("roadmap.phase2Item1")}</li>
                  <li>{t("roadmap.phase2Item2")}</li>
                  <li>{t("roadmap.phase2Item3")}</li>
                  <li>{t("roadmap.phase2Item4")}</li>
                </ul>
              </div>
            </div>
          </motion.div>

          {/* Phase 3 */}
          <motion.div 
            initial={{ opacity: 0, x: -30 }}
            animate={{ opacity: 1, x: 0 }}
            transition={{ duration: 0.4, delay: 0.8 }}
            className="relative flex flex-col md:flex-row items-center mb-12"
          >
            <div className="md:w-1/2 md:pr-12 md:text-right mb-4 md:mb-0">
              <div className="p-6 bg-card/60 backdrop-blur-sm border border-primary/20 rounded-xl">
                <span className="text-xs font-semibold text-primary uppercase tracking-wider">{t("roadmap.phase3")}</span>
                <h4 className="font-display font-bold text-xl text-foreground mt-2 mb-3">{t("roadmap.growth")}</h4>
                <ul className="text-sm text-muted-foreground space-y-2 text-left md:text-right">
                  <li>{t("roadmap.phase3Item1")}</li>
                  <li>{t("roadmap.phase3Item2")}</li>
                  <li>{t("roadmap.phase3Item3")}</li>
                  <li>{t("roadmap.phase3Item4")}</li>
                </ul>
              </div>
            </div>
            <div className="w-4 h-4 bg-primary rounded-full border-4 border-background shadow-lg shadow-primary/30 z-10 hidden md:block" />
            <div className="md:w-1/2 md:pl-12 hidden md:block" />
          </motion.div>

          {/* Phase 4 */}
          <motion.div 
            initial={{ opacity: 0, x: 30 }}
            animate={{ opacity: 1, x: 0 }}
            transition={{ duration: 0.4, delay: 0.9 }}
            className="relative flex flex-col md:flex-row items-center"
          >
            <div className="md:w-1/2 md:pr-12 hidden md:block" />
            <div className="w-4 h-4 bg-secondary rounded-full border-4 border-background shadow-lg shadow-secondary/30 z-10 hidden md:block" />
            <div className="md:w-1/2 md:pl-12 md:text-left">
              <div className="p-6 bg-card/60 backdrop-blur-sm border border-secondary/20 rounded-xl">
                <span className="text-xs font-semibold text-secondary uppercase tracking-wider">{t("roadmap.phase4")}</span>
                <h4 className="font-display font-bold text-xl text-foreground mt-2 mb-3">{t("roadmap.innovation")}</h4>
                <ul className="text-sm text-muted-foreground space-y-2 text-left">
                  <li>{t("roadmap.phase4Item1")}</li>
                  <li>{t("roadmap.phase4Item2")}</li>
                  <li>{t("roadmap.phase4Item3")}</li>
                  <li>{t("roadmap.phase4Item4")}</li>
                </ul>
              </div>
            </div>
          </motion.div>
        </div>
      </motion.div>

      {/* Social Links Section */}
      <motion.div
        id="community"
        initial={{ opacity: 0, y: 30 }}
        animate={{ opacity: 1, y: 0 }}
        transition={{ duration: 0.5, delay: 0.6 }}
        className="relative z-10 w-full max-w-2xl mt-16 mb-8 px-6"
      >
        <h3 className="font-display font-bold text-2xl text-foreground mb-8">
          {t("community.title")}
        </h3>
        <div className="flex flex-wrap justify-center gap-4">
          <a
            href="https://x.com/astherafihq"
            target="_blank"
            rel="noopener noreferrer"
            className="flex items-center gap-3 px-6 py-4 bg-card/60 backdrop-blur-sm border border-primary/20 rounded-xl hover:border-primary/40 hover:bg-card/80 transition-all group"
            data-testid="link-twitter"
          >
            <svg stroke="currentColor" fill="currentColor" strokeWidth="0" role="img" viewBox="0 0 24 24" className="w-5 h-5 text-primary group-hover:scale-110 transition-transform" xmlns="http://www.w3.org/2000/svg"><path d="M18.901 1.153h3.68l-8.04 9.19L24 22.846h-7.406l-5.8-7.584-6.638 7.584H.474l8.6-9.83L0 1.154h7.594l5.243 6.932ZM17.61 20.644h2.039L6.486 3.24H4.298Z"></path></svg>
            <span className="font-medium text-foreground">X</span>
          </a>
          <a
            href="https://t.me/AstheraFi_Official"
            target="_blank"
            rel="noopener noreferrer"
            className="flex items-center gap-3 px-6 py-4 bg-card/60 backdrop-blur-sm border border-primary/20 rounded-xl hover:border-primary/40 hover:bg-card/80 transition-all group"
            data-testid="link-telegram"
          >
            <Send className="w-6 h-6 text-primary group-hover:scale-110 transition-transform" />
            <span className="font-medium text-foreground">Telegram</span>
          </a>
          <a
            href="https://discord.gg/astherafi"
            target="_blank"
            rel="noopener noreferrer"
            className="flex items-center gap-3 px-6 py-4 bg-card/60 backdrop-blur-sm border border-primary/20 rounded-xl hover:border-primary/40 hover:bg-card/80 transition-all group"
            data-testid="link-discord"
          >
            <MessageCircle className="w-6 h-6 text-primary group-hover:scale-110 transition-transform" />
            <span className="font-medium text-foreground">Discord</span>
          </a>
        </div>
      </motion.div>

      {/* Footer */}
      <div className="relative z-10 w-full mt-12 pt-8 border-t border-primary/10 text-center">
        <p className="text-sm text-muted-foreground">
          2025 Astherafi. All rights reserved.
        </p>
      </div>
    </div>
  );
}
