import { motion } from "framer-motion";
import { Link } from "wouter";
import { 
  ArrowLeft, 
  Shield, 
  Brain, 
  Coins, 
  Globe, 
  Lock, 
  Zap,
  TrendingUp,
  Users,
  Target,
  Rocket,
  CheckCircle2,
  ChevronRight
} from "lucide-react";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { useLanguage } from "@/lib/i18n";
import { LanguageSwitcher } from "@/components/LanguageSwitcher";

export default function Docs() {
  const { t } = useLanguage();
  
  return (
    <div className="min-h-screen bg-background text-foreground">
      <div className="fixed inset-0 overflow-hidden pointer-events-none">
        <div className="absolute top-0 left-1/4 w-96 h-96 bg-primary/5 rounded-full blur-3xl" />
        <div className="absolute bottom-1/4 right-1/4 w-80 h-80 bg-purple-500/5 rounded-full blur-3xl" />
      </div>

      <header className="sticky top-0 z-50 border-b border-primary/10 bg-background/80 backdrop-blur-xl">
        <div className="max-w-6xl mx-auto px-4 py-4 flex items-center justify-between">
          <Link href="/">
            <Button variant="ghost" className="gap-2" data-testid="button-back-home">
              <ArrowLeft className="w-4 h-4" />
              {t("docs.backToHome")}
            </Button>
          </Link>
          <div className="flex items-center gap-3">
            <LanguageSwitcher />
            <Badge variant="outline" className="text-primary border-primary/30">
              {t("docs.whitepaper")}
            </Badge>
          </div>
        </div>
      </header>

      <main className="relative z-10 max-w-6xl mx-auto px-4 py-12">
        <motion.div
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          className="text-center mb-16"
        >
          <Badge className="mb-4 bg-primary/10 text-primary border-primary/20">
            {t("docs.officialDocs")}
          </Badge>
          <h1 className="text-4xl md:text-5xl lg:text-6xl font-display font-bold mb-6">
            <span className="bg-gradient-to-r from-primary via-purple-400 to-pink-400 bg-clip-text text-transparent">
              Astherafi
            </span>
          </h1>
          <p className="text-xl text-muted-foreground max-w-3xl mx-auto">
            {t("docs.subtitle")}
          </p>
        </motion.div>

        <div className="space-y-16">
          <Section id="introduction" title={t("docs.introduction")} icon={<Rocket className="w-6 h-6" />}>
            <p className="text-muted-foreground leading-relaxed mb-4">
              {t("docs.introText1")}
            </p>
            <p className="text-muted-foreground leading-relaxed">
              {t("docs.introText2")}
            </p>
          </Section>

          <Section id="vision" title={t("docs.ourVision")} icon={<Target className="w-6 h-6" />}>
            <div className="grid md:grid-cols-2 gap-6">
              <Card className="bg-card/50 border-primary/10">
                <CardHeader>
                  <CardTitle className="flex items-center gap-2 text-lg">
                    <Shield className="w-5 h-5 text-primary" />
                    {t("docs.democratizeSecurity")}
                  </CardTitle>
                </CardHeader>
                <CardContent className="text-muted-foreground">
                  {t("docs.democratizeSecurityDesc")}
                </CardContent>
              </Card>
              <Card className="bg-card/50 border-primary/10">
                <CardHeader>
                  <CardTitle className="flex items-center gap-2 text-lg">
                    <Brain className="w-5 h-5 text-purple-400" />
                    {t("docs.aiFirstApproach")}
                  </CardTitle>
                </CardHeader>
                <CardContent className="text-muted-foreground">
                  {t("docs.aiFirstApproachDesc")}
                </CardContent>
              </Card>
              <Card className="bg-card/50 border-primary/10">
                <CardHeader>
                  <CardTitle className="flex items-center gap-2 text-lg">
                    <Globe className="w-5 h-5 text-pink-400" />
                    {t("docs.multiChainSupport")}
                  </CardTitle>
                </CardHeader>
                <CardContent className="text-muted-foreground">
                  {t("docs.multiChainSupportDesc")}
                </CardContent>
              </Card>
              <Card className="bg-card/50 border-primary/10">
                <CardHeader>
                  <CardTitle className="flex items-center gap-2 text-lg">
                    <Users className="w-5 h-5 text-green-400" />
                    {t("docs.communityDriven")}
                  </CardTitle>
                </CardHeader>
                <CardContent className="text-muted-foreground">
                  {t("docs.communityDrivenDesc")}
                </CardContent>
              </Card>
            </div>
          </Section>

          <Section id="features" title={t("docs.platformFeatures")} icon={<Zap className="w-6 h-6" />}>
            <div className="space-y-8">
              <FeatureBlock
                title={t("docs.feature.tokenAnalyzer")}
                badge={t("docs.feature.tokenAnalyzerBadge")}
                badgeColor="bg-purple-500/20 text-purple-400"
                description={t("docs.feature.tokenAnalyzerDesc")}
                features={[
                  t("docs.feature.tokenAnalyzer1"),
                  t("docs.feature.tokenAnalyzer2"),
                  t("docs.feature.tokenAnalyzer3"),
                  t("docs.feature.tokenAnalyzer4"),
                  t("docs.feature.tokenAnalyzer5"),
                  t("docs.feature.tokenAnalyzer6")
                ]}
              />

              <FeatureBlock
                title={t("docs.feature.tokenGenerator")}
                badge={t("docs.feature.tokenGeneratorBadge")}
                badgeColor="bg-yellow-500/20 text-yellow-400"
                description={t("docs.feature.tokenGeneratorDesc")}
                features={[
                  t("docs.feature.tokenGenerator1"),
                  t("docs.feature.tokenGenerator2"),
                  t("docs.feature.tokenGenerator3"),
                  t("docs.feature.tokenGenerator4"),
                  t("docs.feature.tokenGenerator5")
                ]}
              />

              <FeatureBlock
                title={t("docs.feature.bundleChecker")}
                badge={t("docs.feature.bundleCheckerBadge")}
                badgeColor="bg-red-500/20 text-red-400"
                description={t("docs.feature.bundleCheckerDesc")}
                features={[
                  t("docs.feature.bundleChecker1"),
                  t("docs.feature.bundleChecker2"),
                  t("docs.feature.bundleChecker3"),
                  t("docs.feature.bundleChecker4"),
                  t("docs.feature.bundleChecker5")
                ]}
              />

              <FeatureBlock
                title={t("docs.feature.websiteBuilder")}
                badge={t("docs.feature.websiteBuilderBadge")}
                badgeColor="bg-blue-500/20 text-blue-400"
                description={t("docs.feature.websiteBuilderDesc")}
                features={[
                  t("docs.feature.websiteBuilder1"),
                  t("docs.feature.websiteBuilder2"),
                  t("docs.feature.websiteBuilder3"),
                  t("docs.feature.websiteBuilder4"),
                  t("docs.feature.websiteBuilder5")
                ]}
              />

              <FeatureBlock
                title={t("docs.feature.passwordSecurity")}
                badge={t("docs.feature.passwordSecurityBadge")}
                badgeColor="bg-green-500/20 text-green-400"
                description={t("docs.feature.passwordSecurityDesc")}
                features={[
                  t("docs.feature.passwordSecurity1"),
                  t("docs.feature.passwordSecurity2"),
                  t("docs.feature.passwordSecurity3"),
                  t("docs.feature.passwordSecurity4"),
                  t("docs.feature.passwordSecurity5")
                ]}
              />

              <FeatureBlock
                title={t("docs.feature.tokenStaking")}
                badge={t("docs.feature.tokenStakingBadge")}
                badgeColor="bg-primary/20 text-primary"
                description={t("docs.feature.tokenStakingDesc")}
                features={[
                  t("docs.feature.tokenStaking1"),
                  t("docs.feature.tokenStaking2"),
                  t("docs.feature.tokenStaking3"),
                  t("docs.feature.tokenStaking4")
                ]}
              />

              <FeatureBlock
                title={t("docs.feature.tokenSwap")}
                badge={t("docs.feature.tokenSwapBadge")}
                badgeColor="bg-orange-500/20 text-orange-400"
                description={t("docs.feature.tokenSwapDesc")}
                features={[
                  t("docs.feature.tokenSwap1"),
                  t("docs.feature.tokenSwap2"),
                  t("docs.feature.tokenSwap3"),
                  t("docs.feature.tokenSwap4")
                ]}
              />
            </div>
          </Section>

          <Section id="security" title={t("docs.securityPrivacy")} icon={<Lock className="w-6 h-6" />}>
            <div className="bg-card/50 border border-primary/10 rounded-lg p-6">
              <div className="grid md:grid-cols-2 gap-6">
                <div>
                  <h4 className="font-semibold mb-3 text-foreground">{t("docs.dataProtection")}</h4>
                  <ul className="space-y-2 text-muted-foreground">
                    <li className="flex items-start gap-2">
                      <CheckCircle2 className="w-4 h-4 text-green-400 mt-1 shrink-0" />
                      {t("docs.noPersonalData")}
                    </li>
                    <li className="flex items-start gap-2">
                      <CheckCircle2 className="w-4 h-4 text-green-400 mt-1 shrink-0" />
                      {t("docs.walletReadOnly")}
                    </li>
                    <li className="flex items-start gap-2">
                      <CheckCircle2 className="w-4 h-4 text-green-400 mt-1 shrink-0" />
                      {t("docs.explicitApproval")}
                    </li>
                  </ul>
                </div>
                <div>
                  <h4 className="font-semibold mb-3 text-foreground">{t("docs.platformSecurity")}</h4>
                  <ul className="space-y-2 text-muted-foreground">
                    <li className="flex items-start gap-2">
                      <CheckCircle2 className="w-4 h-4 text-green-400 mt-1 shrink-0" />
                      {t("docs.httpsEncryption")}
                    </li>
                    <li className="flex items-start gap-2">
                      <CheckCircle2 className="w-4 h-4 text-green-400 mt-1 shrink-0" />
                      {t("docs.noPrivateKeys")}
                    </li>
                    <li className="flex items-start gap-2">
                      <CheckCircle2 className="w-4 h-4 text-green-400 mt-1 shrink-0" />
                      {t("docs.regularAudits")}
                    </li>
                  </ul>
                </div>
              </div>
            </div>
          </Section>

          <Section id="roadmap" title={t("docs.roadmap")} icon={<TrendingUp className="w-6 h-6" />}>
            <div className="space-y-6">
              <RoadmapPhase
                phase={t("docs.phase1")}
                title={t("docs.foundation")}
                status="completed"
                statusLabel={t("docs.completed")}
                items={[
                  t("docs.roadmap.phase1.item1"),
                  t("docs.roadmap.phase1.item2"),
                  t("docs.roadmap.phase1.item3"),
                  t("docs.roadmap.phase1.item4")
                ]}
              />
              <RoadmapPhase
                phase={t("docs.phase2")}
                title={t("docs.expansion")}
                status="in-progress"
                statusLabel={t("docs.inProgress")}
                items={[
                  t("docs.roadmap.phase2.item1"),
                  t("docs.roadmap.phase2.item2"),
                  t("docs.roadmap.phase2.item3"),
                  t("docs.roadmap.phase2.item4")
                ]}
              />
              <RoadmapPhase
                phase={t("docs.phase3")}
                title={t("docs.growth")}
                status="upcoming"
                statusLabel={t("docs.upcoming")}
                items={[
                  t("docs.roadmap.phase3.item1"),
                  t("docs.roadmap.phase3.item2"),
                  t("docs.roadmap.phase3.item3"),
                  t("docs.roadmap.phase3.item4")
                ]}
              />
              <RoadmapPhase
                phase={t("docs.phase4")}
                title={t("docs.innovation")}
                status="upcoming"
                statusLabel={t("docs.upcoming")}
                items={[
                  t("docs.roadmap.phase4.item1"),
                  t("docs.roadmap.phase4.item2"),
                  t("docs.roadmap.phase4.item3"),
                  t("docs.roadmap.phase4.item4")
                ]}
              />
            </div>
          </Section>

          <Section id="tokenomics" title={t("docs.tokenomics")} icon={<Coins className="w-6 h-6" />}>
            <Card className="bg-card/50 border-primary/10">
              <CardContent className="pt-6">
                <div className="text-center mb-8">
                  <h3 className="text-2xl font-bold text-foreground mb-2">$ASTH Token</h3>
                  <p className="text-muted-foreground">{t("docs.tokenomicsDesc")}</p>
                </div>
                <div className="grid sm:grid-cols-2 lg:grid-cols-4 gap-4">
                  <div className="text-center p-4 bg-background/50 rounded-lg border border-primary/20">
                    <div className="text-2xl font-bold text-primary">1B</div>
                    <div className="text-sm text-muted-foreground">{t("docs.totalSupply")}</div>
                  </div>
                  <div className="text-center p-4 bg-background/50 rounded-lg border border-purple-500/20">
                    <div className="text-2xl font-bold text-purple-400">3%</div>
                    <div className="text-sm text-muted-foreground">{t("docs.tax")}</div>
                  </div>
                  <div className="text-center p-4 bg-background/50 rounded-lg border border-yellow-500/20">
                    <div className="text-2xl font-bold text-yellow-400">BSC</div>
                    <div className="text-sm text-muted-foreground">{t("docs.network")}</div>
                  </div>
                  <div className="text-center p-4 bg-background/50 rounded-lg border border-green-500/20">
                    <div className="text-2xl font-bold text-green-400">{t("docs.locked")}</div>
                    <div className="text-sm text-muted-foreground">{t("docs.liquidityStatus")}</div>
                  </div>
                </div>
                <div className="mt-8 pt-6 border-t border-primary/10">
                  <h4 className="font-semibold mb-4 text-foreground">{t("docs.tokenUtilities")}</h4>
                  <div className="grid md:grid-cols-2 gap-4 text-muted-foreground">
                    <div className="flex items-start gap-2">
                      <ChevronRight className="w-4 h-4 text-primary mt-1 shrink-0" />
                      <span>{t("docs.utility1")}</span>
                    </div>
                    <div className="flex items-start gap-2">
                      <ChevronRight className="w-4 h-4 text-primary mt-1 shrink-0" />
                      <span>{t("docs.utility2")}</span>
                    </div>
                    <div className="flex items-start gap-2">
                      <ChevronRight className="w-4 h-4 text-primary mt-1 shrink-0" />
                      <span>{t("docs.utility3")}</span>
                    </div>
                    <div className="flex items-start gap-2">
                      <ChevronRight className="w-4 h-4 text-primary mt-1 shrink-0" />
                      <span>{t("docs.utility4")}</span>
                    </div>
                    <div className="flex items-start gap-2">
                      <ChevronRight className="w-4 h-4 text-primary mt-1 shrink-0" />
                      <span>{t("docs.utility5")}</span>
                    </div>
                    <div className="flex items-start gap-2">
                      <ChevronRight className="w-4 h-4 text-primary mt-1 shrink-0" />
                      <span>{t("docs.utility6")}</span>
                    </div>
                  </div>
                </div>
              </CardContent>
            </Card>
          </Section>

          <Section id="disclaimer" title={t("docs.disclaimer")} icon={<Shield className="w-6 h-6" />}>
            <div className="bg-yellow-500/10 border border-yellow-500/20 rounded-lg p-6 text-muted-foreground">
              <p className="mb-4">
                <strong className="text-yellow-400">{t("docs.disclaimerNotice")}</strong> {t("docs.disclaimerText1")}
              </p>
              <p className="mb-4">
                {t("docs.disclaimerText2")}
              </p>
              <p>
                {t("docs.disclaimerText3")}
              </p>
            </div>
          </Section>

          <motion.div
            initial={{ opacity: 0 }}
            whileInView={{ opacity: 1 }}
            viewport={{ once: true }}
            className="text-center pt-12 border-t border-primary/10"
          >
            <h3 className="text-2xl font-bold mb-4">{t("docs.readyToStart")}</h3>
            <p className="text-muted-foreground mb-6">
              {t("docs.readyToStartDesc")}
            </p>
            <div className="flex flex-wrap justify-center gap-4">
              <Link href="/dashboard">
                <Button size="lg" className="gap-2" data-testid="button-launch-app">
                  <Rocket className="w-4 h-4" />
                  {t("docs.launchApp")}
                </Button>
              </Link>
              <Link href="/">
                <Button size="lg" variant="outline" className="gap-2" data-testid="button-back-home-bottom">
                  <ArrowLeft className="w-4 h-4" />
                  {t("docs.backToHomeBtn")}
                </Button>
              </Link>
            </div>
          </motion.div>
        </div>
      </main>

      <footer className="relative z-10 border-t border-primary/10 mt-16 py-8">
        <div className="max-w-6xl mx-auto px-4 text-center text-muted-foreground">
          <p>&copy; {new Date().getFullYear()} Astherafi. All rights reserved.</p>
        </div>
      </footer>
    </div>
  );
}

function Section({ id, title, icon, children }: { id: string; title: string; icon: React.ReactNode; children: React.ReactNode }) {
  return (
    <motion.section
      id={id}
      initial={{ opacity: 0, y: 20 }}
      whileInView={{ opacity: 1, y: 0 }}
      viewport={{ once: true, margin: "-100px" }}
      className="scroll-mt-24"
    >
      <div className="flex items-center gap-3 mb-6">
        <div className="p-2 rounded-lg bg-primary/10 text-primary">{icon}</div>
        <h2 className="text-2xl md:text-3xl font-display font-bold">{title}</h2>
      </div>
      {children}
    </motion.section>
  );
}

function FeatureBlock({ 
  title, 
  badge, 
  badgeColor, 
  description, 
  features 
}: { 
  title: string; 
  badge: string; 
  badgeColor: string; 
  description: string; 
  features: string[] 
}) {
  return (
    <Card className="bg-card/50 border-primary/10">
      <CardHeader>
        <div className="flex items-center justify-between flex-wrap gap-2">
          <CardTitle className="text-xl">{title}</CardTitle>
          <Badge className={badgeColor}>{badge}</Badge>
        </div>
        <p className="text-muted-foreground">{description}</p>
      </CardHeader>
      <CardContent>
        <ul className="grid sm:grid-cols-2 gap-2">
          {features.map((feature, i) => (
            <li key={i} className="flex items-start gap-2 text-muted-foreground">
              <CheckCircle2 className="w-4 h-4 text-primary mt-1 shrink-0" />
              <span>{feature}</span>
            </li>
          ))}
        </ul>
      </CardContent>
    </Card>
  );
}

function RoadmapPhase({ 
  phase, 
  title, 
  status,
  statusLabel,
  items 
}: { 
  phase: string; 
  title: string; 
  status: "completed" | "in-progress" | "upcoming";
  statusLabel: string;
  items: string[] 
}) {
  const statusColors = {
    completed: "bg-green-500/20 text-green-400 border-green-500/30",
    "in-progress": "bg-primary/20 text-primary border-primary/30",
    upcoming: "bg-muted text-muted-foreground border-muted"
  };

  return (
    <div className="flex gap-4">
      <div className="flex flex-col items-center">
        <div className={`w-4 h-4 rounded-full ${status === "completed" ? "bg-green-500" : status === "in-progress" ? "bg-primary" : "bg-muted-foreground/30"}`} />
        <div className="flex-1 w-0.5 bg-primary/20 mt-2" />
      </div>
      <div className="flex-1 pb-8">
        <div className="flex items-center gap-3 mb-3 flex-wrap">
          <span className="text-sm font-medium text-muted-foreground">{phase}</span>
          <h4 className="font-bold text-lg">{title}</h4>
          <Badge variant="outline" className={statusColors[status]}>
            {statusLabel}
          </Badge>
        </div>
        <ul className="space-y-2">
          {items.map((item, i) => (
            <li key={i} className="flex items-center gap-2 text-muted-foreground">
              <ChevronRight className="w-4 h-4 text-primary" />
              <span>{item}</span>
            </li>
          ))}
        </ul>
      </div>
    </div>
  );
}
