import { useState, useEffect, useRef } from "react";
import { Link, useSearch } from "wouter";
import bscLogo from "@assets/image_1768373046822.png";
import astherfiLogo from "@assets/image_1768380685328.png";
import { motion } from "framer-motion";
import { useLanguage } from "@/lib/i18n";
import { LanguageSwitcher } from "@/components/LanguageSwitcher";
import {
  Search,
  Layers,
  Coins,
  Shield,
  Percent,
  ChevronRight,
  ChevronLeft,
  User,
  TrendingUp,
  Zap,
  Activity,
  BarChart3,
  Home,
  PanelLeftClose,
  PanelLeft,
  Menu,
  Loader2,
  CheckCircle,
  XCircle,
  AlertCircle,
  Globe,
  Twitter,
  MessageCircle,
  ExternalLink,
  Upload,
  Image,
  Wallet,
  Copy,
  Info,
  Lock,
  ArrowDownUp,
  RefreshCw,
  Settings,
  ChevronDown,
  ArrowLeft,
  FileCheck,
  Download,
  FileText,
  AlertTriangle,
  Compass,
  ArrowUpRight,
  ArrowDownLeft,
} from "lucide-react";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Textarea } from "@/components/ui/textarea";
import { jsPDF } from "jspdf";
import { createChart, ColorType, CandlestickData, Time, CandlestickSeries } from "lightweight-charts";

const TOKEN_LABELS = ["Meme", "AI", "DeFi", "Games", "Infra", "De-Sci", "Social", "DePIN", "Charity", "Others"];

type MenuItem = {
  id: string;
  label: string;
  icon: React.ReactNode;
  description?: string;
  badge?: string;
};

const menuItems: MenuItem[] = [
  {
    id: "dashboard",
    label: "Dashboard",
    icon: <Home className="w-5 h-5" />,
    description: "Welcome to Astherafi - your blockchain utilities dashboard",
  },
  {
    id: "token-analyzer",
    label: "Token Analyzer",
    icon: <Search className="w-5 h-5" />,
    description: "Analyze any token contract for security risks and metrics",
  },
  {
    id: "token-generator",
    label: "Create Token",
    icon: <Coins className="w-5 h-5" />,
    description: "Create and deploy custom tokens in minutes",
  },
  {
    id: "swap",
    label: "Swap",
    icon: <ArrowDownUp className="w-5 h-5" />,
    description: "Swap tokens on BSC",
  },
  {
    id: "create-website",
    label: "Create Website",
    icon: <Globe className="w-5 h-5" />,
    description: "Create a simple landing page for your project",
  },
  {
    id: "bundle-checker",
    label: "Bundle Checker",
    icon: <Layers className="w-5 h-5" />,
    description: "Detect bundle signals in Solana and BSC token launches",
  },
  {
    id: "password-score",
    label: "Password Score",
    icon: <Shield className="w-5 h-5" />,
    description: "Check the strength of your wallet passwords",
  },
  {
    id: "staking",
    label: "Staking",
    icon: <Percent className="w-5 h-5" />,
    description: "Stake tokens and earn passive rewards",
  },
  {
    id: "block-explorer",
    label: "Block Explorer",
    icon: <Compass className="w-5 h-5" />,
    description: "Explore BSC addresses, balances, and transactions",
  },
];

const CHAINS = [
  { id: "bsc", name: "BSC", icon: "", image: bscLogo },
  { id: "ethereum", name: "Ethereum", icon: "", image: "https://cryptologos.cc/logos/ethereum-eth-logo.svg" },
  { id: "solana", name: "Solana", icon: "", image: "https://cryptologos.cc/logos/solana-sol-logo.svg" },
];

export default function Dashboard() {
  const searchString = useSearch();
  const [activeItem, setActiveItem] = useState("dashboard");
  const [mobileMenuOpen, setMobileMenuOpen] = useState(false);
  const [sidebarCollapsed, setSidebarCollapsed] = useState(false);
  const [isDesktop, setIsDesktop] = useState(false);
  const [tokenAddress, setTokenAddress] = useState("");
  const [autoAnalyzed, setAutoAnalyzed] = useState(false);
  const canvasRef = useRef<HTMLCanvasElement>(null);
  const { t } = useLanguage();

  // Translated menu items
  const translatedMenuItems: MenuItem[] = [
    { id: "dashboard", label: t("dashboard.title"), icon: <Home className="w-5 h-5" /> },
    { id: "token-analyzer", label: t("dashboard.tokenAnalyzer"), icon: <Search className="w-5 h-5" /> },
    { id: "token-generator", label: t("dashboard.tokenGenerator"), icon: <Coins className="w-5 h-5" /> },
    { id: "website-builder", label: t("dashboard.websiteBuilder"), icon: <Globe className="w-5 h-5" /> },
    { id: "bundle-checker", label: t("dashboard.bundleChecker"), icon: <Layers className="w-5 h-5" /> },
    { id: "password-score", label: t("dashboard.passwordScore"), icon: <Shield className="w-5 h-5" /> },
    { id: "staking", label: t("dashboard.staking"), icon: <Percent className="w-5 h-5" />, badge: "Soon" },
    { id: "token-swap", label: t("dashboard.tokenSwap"), icon: <ArrowDownUp className="w-5 h-5" /> },
    { id: "futures", label: "Futures", icon: <TrendingUp className="w-5 h-5" />, badge: "Soon" },
    { id: "free-audit", label: t("dashboard.freeAudit"), icon: <FileCheck className="w-5 h-5" /> },
    { id: "block-explorer", label: t("dashboard.blockExplorer"), icon: <Compass className="w-5 h-5" />, badge: "Soon" },
  ];

  // Detect screen size with JavaScript for reliable responsive behavior
  useEffect(() => {
    const checkScreenSize = () => {
      setIsDesktop(window.innerWidth >= 1024);
    };
    checkScreenSize();
    window.addEventListener('resize', checkScreenSize);
    return () => window.removeEventListener('resize', checkScreenSize);
  }, []);

  // Handle URL token parameter from landing page search
  useEffect(() => {
    if (autoAnalyzed) return;
    const params = new URLSearchParams(searchString);
    const tokenParam = params.get('token');
    if (tokenParam) {
      setTokenAddress(tokenParam);
      setActiveItem('token-analyzer');
      setAutoAnalyzed(true);
      // Trigger analysis after a short delay to let state update
      setTimeout(() => {
        const analyzeBtn = document.querySelector('[data-testid="button-analyze-token"]') as HTMLButtonElement;
        if (analyzeBtn) analyzeBtn.click();
      }, 100);
    }
  }, [searchString, autoAnalyzed]);

  // Particle background effect
  useEffect(() => {
    const canvas = canvasRef.current;
    if (!canvas) return;

    const ctx = canvas.getContext("2d");
    if (!ctx) return;

    const resizeCanvas = () => {
      canvas.width = window.innerWidth;
      canvas.height = window.innerHeight;
    };
    resizeCanvas();
    window.addEventListener("resize", resizeCanvas);

    interface Particle {
      x: number;
      y: number;
      size: number;
      speedX: number;
      speedY: number;
      opacity: number;
    }

    const particles: Particle[] = [];
    const particleCount = 60;

    for (let i = 0; i < particleCount; i++) {
      particles.push({
        x: Math.random() * canvas.width,
        y: Math.random() * canvas.height,
        size: Math.random() * 1.5 + 0.5,
        speedX: (Math.random() - 0.5) * 0.2,
        speedY: (Math.random() - 0.5) * 0.2,
        opacity: Math.random() * 0.4 + 0.1,
      });
    }

    const animate = () => {
      ctx.clearRect(0, 0, canvas.width, canvas.height);

      particles.forEach((particle) => {
        particle.x += particle.speedX;
        particle.y += particle.speedY;

        if (particle.x < 0) particle.x = canvas.width;
        if (particle.x > canvas.width) particle.x = 0;
        if (particle.y < 0) particle.y = canvas.height;
        if (particle.y > canvas.height) particle.y = 0;

        ctx.beginPath();
        ctx.arc(particle.x, particle.y, particle.size, 0, Math.PI * 2);
        ctx.fillStyle = `rgba(255, 255, 255, ${particle.opacity})`;
        ctx.fill();
      });

      particles.forEach((p1, i) => {
        particles.slice(i + 1).forEach((p2) => {
          const dx = p1.x - p2.x;
          const dy = p1.y - p2.y;
          const distance = Math.sqrt(dx * dx + dy * dy);

          if (distance < 100) {
            ctx.beginPath();
            ctx.moveTo(p1.x, p1.y);
            ctx.lineTo(p2.x, p2.y);
            ctx.strokeStyle = `rgba(48, 208, 242, ${0.08 * (1 - distance / 100)})`;
            ctx.lineWidth = 0.5;
            ctx.stroke();
          }
        });
      });

      requestAnimationFrame(animate);
    };

    animate();

    return () => {
      window.removeEventListener("resize", resizeCanvas);
    };
  }, []);
  const [selectedChain, setSelectedChain] = useState("bsc");
  const [tokenAnalysis, setTokenAnalysis] = useState<any>(null);
  const [isAnalyzing, setIsAnalyzing] = useState(false);
  const [analysisError, setAnalysisError] = useState<string | null>(null);
  const [homeSearchAddress, setHomeSearchAddress] = useState("");
  
  // AI Analysis State
  const [aiAnalysis, setAiAnalysis] = useState<any>(null);
  const [isAiAnalyzing, setIsAiAnalyzing] = useState(false);
  const [aiAnalysisError, setAiAnalysisError] = useState<string | null>(null);

  // Token Generator State
  const [tokenName, setTokenName] = useState("");
  const [tokenSymbol, setTokenSymbol] = useState("");
  const [tokenDescription, setTokenDescription] = useState("");
  const [tokenLabel, setTokenLabel] = useState("Meme");
  const [tokenWebsite, setTokenWebsite] = useState("");
  const [tokenTwitter, setTokenTwitter] = useState("");
  const [tokenTelegram, setTokenTelegram] = useState("");
  const [tokenImage, setTokenImage] = useState<File | null>(null);
  const [tokenImagePreview, setTokenImagePreview] = useState<string | null>(null);
  const [preSaleAmount, setPreSaleAmount] = useState("");
  const [isCreatingToken, setIsCreatingToken] = useState(false);
  const [tokenCreationError, setTokenCreationError] = useState<string | null>(null);
  const [tokenCreationSuccess, setTokenCreationSuccess] = useState<any>(null);
  const [walletConnected, setWalletConnected] = useState(false);
  const [walletAddress, setWalletAddress] = useState("");

  // Password Score State
  const [passwordInput, setPasswordInput] = useState("");
  const [passwordScore, setPasswordScore] = useState(0);
  const [passwordFeedback, setPasswordFeedback] = useState<string[]>([]);

  // Bundle Checker State
  const [bundleMintAddress, setBundleMintAddress] = useState("");
  const [bundleResult, setBundleResult] = useState<any>(null);
  const [isCheckingBundle, setIsCheckingBundle] = useState(false);
  const [bundleError, setBundleError] = useState<string | null>(null);
  const [bundleChain, setBundleChain] = useState<"solana" | "bsc">("solana");

  // Website Builder State
  const [websiteProjectName, setWebsiteProjectName] = useState("");
  const [websiteTagline, setWebsiteTagline] = useState("");
  const [websiteDescription, setWebsiteDescription] = useState("");
  const [websiteContractAddress, setWebsiteContractAddress] = useState("");
  const [websiteTwitter, setWebsiteTwitter] = useState("");
  const [websiteTelegram, setWebsiteTelegram] = useState("");
  const [websiteTheme, setWebsiteTheme] = useState<"dark" | "light" | "cyber">("cyber");
  const [generatedWebsite, setGeneratedWebsite] = useState<string | null>(null);
  const [websiteLogo, setWebsiteLogo] = useState<string | null>(null);

  // Free Audit State
  const [auditContractAddress, setAuditContractAddress] = useState("");
  const [isAuditing, setIsAuditing] = useState(false);
  const [auditResult, setAuditResult] = useState<any>(null);
  const [auditError, setAuditError] = useState<string | null>(null);
  const [auditProgress, setAuditProgress] = useState<string[]>([]);

  // Block Explorer State
  const [explorerAddress, setExplorerAddress] = useState("");
  const [isExploring, setIsExploring] = useState(false);
  const [explorerResult, setExplorerResult] = useState<any>(null);
  const [explorerError, setExplorerError] = useState<string | null>(null);
  const [explorerData, setExplorerData] = useState<any>(null);
  const [isLoadingExplorer, setIsLoadingExplorer] = useState(false);

  // Futures Trading Live Data State
  const [futuresPair, setFuturesPair] = useState("BTCUSDT");
  const [futuresPrice, setFuturesPrice] = useState<any>({
    price: 0,
    markPrice: 0,
    indexPrice: 0,
    fundingRate: 0,
    volume24h: 0,
    openInterest: 0,
    priceChange: 0,
    priceChangePercent: 0,
  });
  const [futuresOrderBook, setFuturesOrderBook] = useState<any>({ bids: [], asks: [] });
  const [futuresKlines, setFuturesKlines] = useState<any[]>([]);
  const [futuresInterval, setFuturesInterval] = useState("1h");
  const chartContainerRef = useRef<HTMLDivElement>(null);
  const chartRef = useRef<any>(null);
  const candlestickSeriesRef = useRef<any>(null);

  // Crypto Prices Marquee State
  const [cryptoPrices, setCryptoPrices] = useState<Array<{
    id: string;
    symbol: string;
    name: string;
    current_price: number;
    price_change_percentage_24h: number;
    image: string;
  }>>([]);

  // Fetch top crypto prices
  useEffect(() => {
    const fetchPrices = async () => {
      try {
        const response = await fetch(
          'https://api.coingecko.com/api/v3/coins/markets?vs_currency=usd&order=market_cap_desc&per_page=10&page=1&sparkline=false&price_change_percentage=24h'
        );
        if (response.ok) {
          const data = await response.json();
          setCryptoPrices(data);
        }
      } catch (error) {
        console.log('Failed to fetch crypto prices');
      }
    };
    fetchPrices();
    // Refresh every 60 seconds
    const interval = setInterval(fetchPrices, 60000);
    return () => clearInterval(interval);
  }, []);

  // Fetch Futures Live Data from Binance
  useEffect(() => {
    if (activeItem !== "futures") return;
    
    const fetchFuturesData = async () => {
      try {
        // Fetch ticker data
        const tickerRes = await fetch(`https://fapi.binance.com/fapi/v1/ticker/24hr?symbol=${futuresPair}`);
        const ticker = await tickerRes.json();
        
        // Fetch mark price and funding rate
        const markRes = await fetch(`https://fapi.binance.com/fapi/v1/premiumIndex?symbol=${futuresPair}`);
        const markData = await markRes.json();
        
        // Fetch order book
        const bookRes = await fetch(`https://fapi.binance.com/fapi/v1/depth?symbol=${futuresPair}&limit=10`);
        const bookData = await bookRes.json();
        
        // Fetch klines (candlestick data)
        const klinesRes = await fetch(`https://fapi.binance.com/fapi/v1/klines?symbol=${futuresPair}&interval=${futuresInterval}&limit=100`);
        const klinesData = await klinesRes.json();
        
        setFuturesPrice({
          price: parseFloat(ticker.lastPrice || 0),
          markPrice: parseFloat(markData.markPrice || 0),
          indexPrice: parseFloat(markData.indexPrice || 0),
          fundingRate: parseFloat(markData.lastFundingRate || 0) * 100,
          volume24h: parseFloat(ticker.quoteVolume || 0),
          openInterest: parseFloat(ticker.openInterest || 0) * parseFloat(ticker.lastPrice || 0),
          priceChange: parseFloat(ticker.priceChange || 0),
          priceChangePercent: parseFloat(ticker.priceChangePercent || 0),
        });
        
        if (bookData.bids && bookData.asks) {
          setFuturesOrderBook({
            bids: bookData.bids.slice(0, 8).map((b: any) => ({ price: parseFloat(b[0]), size: parseFloat(b[1]) })),
            asks: bookData.asks.slice(0, 8).reverse().map((a: any) => ({ price: parseFloat(a[0]), size: parseFloat(a[1]) })),
          });
        }
        
        if (Array.isArray(klinesData)) {
          setFuturesKlines(klinesData.map((k: any) => ({
            time: k[0],
            open: parseFloat(k[1]),
            high: parseFloat(k[2]),
            low: parseFloat(k[3]),
            close: parseFloat(k[4]),
            volume: parseFloat(k[5]),
          })));
        }
      } catch (error) {
        console.log('Failed to fetch futures data');
      }
    };
    
    fetchFuturesData();
    const interval = setInterval(fetchFuturesData, 2000); // Update every 2 seconds
    return () => clearInterval(interval);
  }, [activeItem, futuresPair, futuresInterval]);

  // Initialize TradingView Lightweight Chart
  useEffect(() => {
    if (activeItem !== "futures" || !chartContainerRef.current) return;
    
    // Clean up previous chart if it exists
    if (chartRef.current) {
      chartRef.current.remove();
      chartRef.current = null;
      candlestickSeriesRef.current = null;
    }
    
    // Create new chart
    const chart = createChart(chartContainerRef.current, {
      layout: {
        background: { type: ColorType.Solid, color: 'transparent' },
        textColor: '#9ca3af',
      },
      grid: {
        vertLines: { color: 'rgba(42, 46, 57, 0.3)' },
        horzLines: { color: 'rgba(42, 46, 57, 0.3)' },
      },
      width: chartContainerRef.current.clientWidth,
      height: 300,
      timeScale: {
        timeVisible: true,
        secondsVisible: false,
        borderColor: 'rgba(42, 46, 57, 0.5)',
      },
      rightPriceScale: {
        borderColor: 'rgba(42, 46, 57, 0.5)',
      },
      crosshair: {
        mode: 1,
        vertLine: {
          width: 1,
          color: 'rgba(0, 212, 170, 0.3)',
          style: 2,
        },
        horzLine: {
          width: 1,
          color: 'rgba(0, 212, 170, 0.3)',
          style: 2,
        },
      },
    });
    
    const candlestickSeries = chart.addSeries(CandlestickSeries, {
      upColor: '#22c55e',
      downColor: '#ef4444',
      borderUpColor: '#22c55e',
      borderDownColor: '#ef4444',
      wickUpColor: '#22c55e',
      wickDownColor: '#ef4444',
    });
    
    chartRef.current = chart;
    candlestickSeriesRef.current = candlestickSeries;
    
    // Handle resize
    const handleResize = () => {
      if (chartContainerRef.current && chartRef.current) {
        chartRef.current.applyOptions({ width: chartContainerRef.current.clientWidth });
      }
    };
    
    window.addEventListener('resize', handleResize);
    
    return () => {
      window.removeEventListener('resize', handleResize);
      if (chartRef.current) {
        chartRef.current.remove();
        chartRef.current = null;
        candlestickSeriesRef.current = null;
      }
    };
  }, [activeItem, futuresPair, futuresInterval]);

  // Update chart data when klines change
  useEffect(() => {
    if (!candlestickSeriesRef.current || futuresKlines.length === 0) return;
    
    const chartData: CandlestickData<Time>[] = futuresKlines.map((k) => ({
      time: (k.time / 1000) as Time,
      open: k.open,
      high: k.high,
      low: k.low,
      close: k.close,
    }));
    
    candlestickSeriesRef.current.setData(chartData);
    
    // Auto-fit content
    if (chartRef.current) {
      chartRef.current.timeScale().fitContent();
    }
  }, [futuresKlines]);

  // Swap State
  const [swapFromToken, setSwapFromToken] = useState<string>("BNB");
  const [swapToToken, setSwapToToken] = useState<string>("");
  const [swapFromAmount, setSwapFromAmount] = useState<string>("");
  const [swapToAmount, setSwapToAmount] = useState<string>("");
  const [swapSlippage, setSwapSlippage] = useState<string>("0.5");
  const [swapFromBalance, setSwapFromBalance] = useState<string>("0");
  const [swapToBalance, setSwapToBalance] = useState<string>("0");
  const [isSwapping, setIsSwapping] = useState(false);
  const [swapError, setSwapError] = useState<string | null>(null);
  const [swapSuccess, setSwapSuccess] = useState<any>(null);
  const [swapQuote, setSwapQuote] = useState<any>(null);
  const [isLoadingQuote, setIsLoadingQuote] = useState(false);

  // PancakeSwap Constants
  const PANCAKE_ROUTER_V2 = "0x10ED43C718714eb63d5aA57B78B54704E256024E";
  const WBNB_ADDRESS = "0xbb4CdB9CBd36B01bD1cBaEBF2De08d9173bc095c";
  
  // Popular BSC tokens for swap
  const SWAP_TOKENS = [
    { symbol: "BNB", name: "BNB", address: "native", decimals: 18, logo: bscLogo },
    { symbol: "WBNB", name: "Wrapped BNB", address: WBNB_ADDRESS, decimals: 18, logo: bscLogo },
    { symbol: "USDT", name: "Tether USD", address: "0x55d398326f99059fF775485246999027B3197955", decimals: 18, logo: "https://cryptologos.cc/logos/tether-usdt-logo.svg" },
    { symbol: "USDC", name: "USD Coin", address: "0x8AC76a51cc950d9822D68b83fE1Ad97B32Cd580d", decimals: 18, logo: "https://cryptologos.cc/logos/usd-coin-usdc-logo.svg" },
    { symbol: "BUSD", name: "Binance USD", address: "0xe9e7CEA3DedcA5984780Bafc599bD69ADd087D56", decimals: 18, logo: "https://cryptologos.cc/logos/binance-usd-busd-logo.svg" },
    { symbol: "CAKE", name: "PancakeSwap", address: "0x0E09FaBB73Bd3Ade0a17ECC321fD13a19e81cE82", decimals: 18, logo: "https://cryptologos.cc/logos/pancakeswap-cake-logo.svg" },
  ];

  // Handle website logo upload
  const handleWebsiteLogoUpload = (e: React.ChangeEvent<HTMLInputElement>) => {
    const file = e.target.files?.[0];
    if (file) {
      const reader = new FileReader();
      reader.onloadend = () => {
        setWebsiteLogo(reader.result as string);
      };
      reader.readAsDataURL(file);
    }
  };

  // Free Audit Handler
  const handleStartAudit = async () => {
    if (!auditContractAddress.trim()) return;
    
    setIsAuditing(true);
    setAuditError(null);
    setAuditResult(null);
    setAuditProgress([t("dashboard.auditInProgress")]);
    
    try {
      // Add progress messages
      setTimeout(() => setAuditProgress(prev => [...prev, "Address Queued. Analyzing..."]), 1000);
      
      const response = await fetch(`/api/audit/${auditContractAddress}`);
      const data = await response.json();
      
      if (!response.ok) {
        throw new Error(data.error || "Failed to audit contract");
      }
      
      setAuditProgress(prev => [...prev, t("dashboard.auditComplete")]);
      setAuditResult(data);
    } catch (err: any) {
      setAuditError(err.message || "Failed to audit contract");
    } finally {
      setIsAuditing(false);
    }
  };

  // Generate audit report PDF
  const downloadAuditReport = () => {
    if (!auditResult) return;
    
    const doc = new jsPDF();
    const pageWidth = doc.internal.pageSize.getWidth();
    let y = 20;
    
    // Title
    doc.setFontSize(20);
    doc.setTextColor(0, 200, 255);
    doc.text("ASTHERAFI", pageWidth / 2, y, { align: "center" });
    y += 8;
    doc.setFontSize(14);
    doc.setTextColor(100, 100, 100);
    doc.text("Free Smart Contract Audit Report", pageWidth / 2, y, { align: "center" });
    y += 15;
    
    // Audit Score Box
    const scoreColor = auditResult.auditScore >= 80 ? [0, 180, 0] : 
                       auditResult.auditScore >= 60 ? [255, 193, 7] : 
                       auditResult.auditScore >= 40 ? [255, 152, 0] : [220, 53, 69];
    doc.setFillColor(scoreColor[0], scoreColor[1], scoreColor[2]);
    doc.roundedRect(20, y, 50, 25, 3, 3, 'F');
    doc.setFontSize(24);
    doc.setTextColor(255, 255, 255);
    doc.text(`${auditResult.auditScore}`, 45, y + 14, { align: "center" });
    doc.setFontSize(8);
    doc.text("SCORE", 45, y + 21, { align: "center" });
    
    // Grade Box
    const gradeColor = auditResult.grade?.startsWith("A") ? [0, 180, 0] : 
                       auditResult.grade?.startsWith("B") ? [0, 180, 180] : 
                       auditResult.grade?.startsWith("C") ? [255, 193, 7] : [220, 53, 69];
    doc.setFillColor(gradeColor[0], gradeColor[1], gradeColor[2]);
    doc.roundedRect(75, y, 35, 25, 3, 3, 'F');
    doc.setFontSize(20);
    doc.setTextColor(255, 255, 255);
    doc.text(`${auditResult.grade || "N/A"}`, 92.5, y + 14, { align: "center" });
    doc.setFontSize(8);
    doc.text("GRADE", 92.5, y + 21, { align: "center" });
    
    // Contract Info (right side)
    doc.setFontSize(9);
    doc.setTextColor(60, 60, 60);
    doc.text(`Token: ${auditResult.tokenName || "Unknown"} (${auditResult.tokenSymbol || "???"})`, 120, y + 6);
    doc.text(`Chain: BSC (BEP-20)`, 120, y + 12);
    doc.text(`Holders: ${auditResult.rawData?.holderCount?.toLocaleString() || "N/A"}`, 120, y + 18);
    y += 32;
    
    doc.setFontSize(8);
    doc.setTextColor(100, 100, 100);
    doc.text(`Contract: ${auditContractAddress}`, 20, y);
    y += 5;
    doc.text(`Date: ${new Date().toLocaleDateString()}`, 20, y);
    y += 10;
    
    // Summary Header
    doc.setFontSize(14);
    doc.setTextColor(0, 0, 0);
    doc.text("Security Summary", 20, y);
    y += 8;
    
    // Summary Stats
    const summary = auditResult.summary || {};
    const stats = [
      { label: "Critical", value: summary.critical || 0, color: [220, 53, 69] },
      { label: "Major", value: summary.major || 0, color: [255, 152, 0] },
      { label: "Medium", value: summary.medium || 0, color: [255, 193, 7] },
      { label: "Minor", value: summary.minor || 0, color: [33, 150, 243] },
      { label: "Info", value: summary.info || 0, color: [158, 158, 158] },
    ];
    
    doc.setFontSize(10);
    let xPos = 20;
    stats.forEach((stat) => {
      doc.setTextColor(stat.color[0], stat.color[1], stat.color[2]);
      doc.text(`${stat.label}: ${stat.value}`, xPos, y);
      xPos += 35;
    });
    y += 15;
    
    // Findings Header
    doc.setFontSize(14);
    doc.setTextColor(0, 0, 0);
    doc.text("Detailed Findings", 20, y);
    y += 10;
    
    // Findings
    doc.setFontSize(9);
    if (auditResult.findings && auditResult.findings.length > 0) {
      auditResult.findings.forEach((f: any) => {
        if (y > 270) {
          doc.addPage();
          y = 20;
        }
        
        // Severity badge color
        const severityColors: Record<string, number[]> = {
          critical: [220, 53, 69],
          major: [255, 152, 0],
          medium: [255, 193, 7],
          minor: [33, 150, 243],
          info: [158, 158, 158],
        };
        const color = severityColors[f.severity] || [100, 100, 100];
        
        doc.setTextColor(color[0], color[1], color[2]);
        doc.text(`[${f.severity.toUpperCase()}]`, 20, y);
        doc.setTextColor(0, 0, 0);
        doc.text(f.title, 45, y);
        y += 5;
        
        doc.setTextColor(100, 100, 100);
        const descLines = doc.splitTextToSize(f.description, pageWidth - 50);
        doc.text(descLines, 25, y);
        y += descLines.length * 4 + 2;
        
        doc.setTextColor(80, 80, 80);
        doc.text(`Type: ${f.type} | Confidence: ${f.confidence}`, 25, y);
        y += 8;
      });
    } else {
      doc.setTextColor(0, 150, 0);
      doc.text("No security issues found! This contract passed all security checks.", 20, y);
      y += 10;
    }
    
    // Footer
    y = 285;
    doc.setFontSize(8);
    doc.setTextColor(150, 150, 150);
    doc.text("Report generated by Astherafi Smart Contract Audit Tool", pageWidth / 2, y, { align: "center" });
    y += 4;
    doc.text("https://astherafi.com", pageWidth / 2, y, { align: "center" });
    
    // Save PDF
    doc.save(`audit-report-${auditContractAddress.slice(0, 8)}.pdf`);
  };

  // Block Explorer Handler - Load homepage data
  const loadExplorerData = async () => {
    setIsLoadingExplorer(true);
    try {
      const response = await fetch("/api/explorer");
      const data = await response.json();
      if (response.ok) {
        setExplorerData(data);
      }
    } catch (err) {
      console.error("Failed to load explorer data:", err);
    } finally {
      setIsLoadingExplorer(false);
    }
  };
  
  // Load explorer data when entering block-explorer section
  useEffect(() => {
    if (activeItem === "block-explorer" && !explorerData && !isLoadingExplorer) {
      loadExplorerData();
    }
  }, [activeItem]);
  
  // Block Explorer Handler - Search address
  const handleExploreAddress = async () => {
    if (!explorerAddress.trim()) return;
    
    setIsExploring(true);
    setExplorerError(null);
    setExplorerResult(null);
    
    try {
      const response = await fetch(`/api/explorer/${explorerAddress}`);
      const data = await response.json();
      
      if (!response.ok) {
        throw new Error(data.error || "Failed to fetch address data");
      }
      
      setExplorerResult(data);
    } catch (err: any) {
      setExplorerError(err.message || "Failed to fetch address data");
    } finally {
      setIsExploring(false);
    }
  };

  const activeMenu = translatedMenuItems.find((item) => item.id === activeItem);

  // Generate website HTML with randomized modern template
  const generateWebsite = () => {
    if (!websiteProjectName.trim()) return;

    // Random color palettes
    const colorPalettes = [
      { primary: "#6366f1", secondary: "#8b5cf6", accent: "#ec4899", bg: "#0f0f23", cardBg: "#1a1a2e", text: "#ffffff" },
      { primary: "#00d4ff", secondary: "#00ff88", accent: "#ff00ff", bg: "#0a0a0f", cardBg: "#111122", text: "#ffffff" },
      { primary: "#f59e0b", secondary: "#ef4444", accent: "#f97316", bg: "#1c1917", cardBg: "#292524", text: "#fafaf9" },
      { primary: "#10b981", secondary: "#06b6d4", accent: "#14b8a6", bg: "#0f172a", cardBg: "#1e293b", text: "#f8fafc" },
      { primary: "#e11d48", secondary: "#be123c", accent: "#fb7185", bg: "#18181b", cardBg: "#27272a", text: "#fafafa" },
      { primary: "#8b5cf6", secondary: "#a855f7", accent: "#d946ef", bg: "#0c0a09", cardBg: "#1c1917", text: "#fafaf9" },
      { primary: "#0ea5e9", secondary: "#38bdf8", accent: "#7dd3fc", bg: "#020617", cardBg: "#0f172a", text: "#f8fafc" },
    ];

    // Random gradient directions
    const gradientDirs = ["135deg", "45deg", "180deg", "90deg", "225deg", "315deg"];
    
    // Random animations
    const animations = ["fadeInUp", "fadeInDown", "slideIn", "scaleIn", "bounceIn"];
    
    // Random hero patterns
    const patterns = [
      `radial-gradient(circle at 20% 80%, PRIMARY 0%, transparent 50%), radial-gradient(circle at 80% 20%, SECONDARY 0%, transparent 50%)`,
      `linear-gradient(GRADDIR, PRIMARY 0%, transparent 60%), radial-gradient(ellipse at bottom, SECONDARY 0%, transparent 70%)`,
      `conic-gradient(from 180deg at 50% 50%, PRIMARY, SECONDARY, PRIMARY)`,
      `radial-gradient(ellipse at top left, PRIMARY 0%, transparent 50%), radial-gradient(ellipse at bottom right, SECONDARY 0%, transparent 50%)`,
    ];

    // Random selections
    const palette = colorPalettes[Math.floor(Math.random() * colorPalettes.length)];
    const gradDir = gradientDirs[Math.floor(Math.random() * gradientDirs.length)];
    const anim = animations[Math.floor(Math.random() * animations.length)];
    const patternTemplate = patterns[Math.floor(Math.random() * patterns.length)];
    const heroPattern = patternTemplate
      .replace(/PRIMARY/g, palette.primary + "20")
      .replace(/SECONDARY/g, palette.secondary + "15")
      .replace(/GRADDIR/g, gradDir);

    // Random feature icons
    const featureIcons = ["⚡", "🚀", "💎", "🔒", "🌟", "💫", "🎯", "✨", "🔥", "💰"];
    const shuffledIcons = featureIcons.sort(() => Math.random() - 0.5).slice(0, 4);

    // Random stats
    const stats = [
      { value: Math.floor(Math.random() * 50000) + 10000, label: "Holders" },
      { value: "$" + (Math.random() * 10).toFixed(1) + "M", label: "Market Cap" },
      { value: Math.floor(Math.random() * 100) + 1 + "K", label: "Transactions" },
      { value: "24/7", label: "Support" },
    ];

    const html = `<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8">
  <meta name="viewport" content="width=device-width, initial-scale=1.0">
  <title>${websiteProjectName}</title>
  <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600;700;800;900&display=swap" rel="stylesheet">
  <style>
    * { margin: 0; padding: 0; box-sizing: border-box; }
    :root {
      --primary: ${palette.primary};
      --secondary: ${palette.secondary};
      --accent: ${palette.accent};
      --bg: ${palette.bg};
      --card-bg: ${palette.cardBg};
      --text: ${palette.text};
    }
    body { font-family: 'Inter', sans-serif; background: var(--bg); color: var(--text); min-height: 100vh; overflow-x: hidden; }
    
    @keyframes fadeInUp { from { opacity: 0; transform: translateY(30px); } to { opacity: 1; transform: translateY(0); } }
    @keyframes fadeInDown { from { opacity: 0; transform: translateY(-30px); } to { opacity: 1; transform: translateY(0); } }
    @keyframes slideIn { from { opacity: 0; transform: translateX(-30px); } to { opacity: 1; transform: translateX(0); } }
    @keyframes scaleIn { from { opacity: 0; transform: scale(0.9); } to { opacity: 1; transform: scale(1); } }
    @keyframes bounceIn { 0% { opacity: 0; transform: scale(0.3); } 50% { transform: scale(1.05); } 100% { opacity: 1; transform: scale(1); } }
    @keyframes float { 0%, 100% { transform: translateY(0); } 50% { transform: translateY(-10px); } }
    @keyframes glow { 0%, 100% { box-shadow: 0 0 20px var(--primary); } 50% { box-shadow: 0 0 40px var(--primary), 0 0 60px var(--secondary); } }
    
    .animate { animation: ${anim} 0.8s ease-out forwards; }
    .delay-1 { animation-delay: 0.1s; opacity: 0; }
    .delay-2 { animation-delay: 0.2s; opacity: 0; }
    .delay-3 { animation-delay: 0.3s; opacity: 0; }
    .delay-4 { animation-delay: 0.4s; opacity: 0; }
    
    .container { max-width: 1200px; margin: 0 auto; padding: 0 2rem; }
    
    /* Navigation */
    nav { position: fixed; top: 0; left: 0; right: 0; z-index: 100; padding: 1rem 0; background: ${palette.bg}ee; backdrop-filter: blur(10px); border-bottom: 1px solid ${palette.primary}20; }
    nav .container { display: flex; justify-content: space-between; align-items: center; }
    .logo { font-size: 1.5rem; font-weight: 800; background: linear-gradient(${gradDir}, var(--primary), var(--secondary)); -webkit-background-clip: text; -webkit-text-fill-color: transparent; display: flex; align-items: center; gap: 0.75rem; }
    .logo-img { width: 40px; height: 40px; border-radius: 10px; object-fit: cover; }
    .nav-links { display: flex; gap: 2rem; align-items: center; }
    .nav-links a { color: var(--text); text-decoration: none; opacity: 0.8; transition: all 0.3s; }
    .nav-links a:hover { opacity: 1; color: var(--primary); }
    
    /* Hero */
    .hero { min-height: 100vh; display: flex; align-items: center; justify-content: center; text-align: center; position: relative; padding: 8rem 2rem 4rem; background: ${heroPattern}; }
    .hero::before { content: ''; position: absolute; inset: 0; background: url("data:image/svg+xml,%3Csvg viewBox='0 0 200 200' xmlns='http://www.w3.org/2000/svg'%3E%3Cfilter id='noise'%3E%3CfeTurbulence type='fractalNoise' baseFrequency='0.65' numOctaves='3' stitchTiles='stitch'/%3E%3C/filter%3E%3Crect width='100%' height='100%' filter='url(%23noise)'/%3E%3C/svg%3E"); opacity: 0.03; pointer-events: none; }
    .hero-content { position: relative; z-index: 1; max-width: 800px; }
    .hero-badge { display: inline-block; padding: 0.5rem 1rem; background: var(--primary); background: linear-gradient(${gradDir}, var(--primary)20, var(--secondary)20); border: 1px solid var(--primary)40; border-radius: 100px; font-size: 0.875rem; margin-bottom: 2rem; }
    .hero h1 { font-size: clamp(2.5rem, 8vw, 5rem); font-weight: 900; line-height: 1.1; margin-bottom: 1.5rem; background: linear-gradient(${gradDir}, var(--text), var(--primary)); -webkit-background-clip: text; -webkit-text-fill-color: transparent; }
    .hero p { font-size: 1.25rem; opacity: 0.8; max-width: 600px; margin: 0 auto 2.5rem; line-height: 1.7; }
    .hero-buttons { display: flex; gap: 1rem; justify-content: center; flex-wrap: wrap; }
    
    .btn { display: inline-flex; align-items: center; gap: 0.5rem; padding: 1rem 2rem; border-radius: 12px; font-weight: 600; text-decoration: none; transition: all 0.3s; cursor: pointer; border: none; font-size: 1rem; }
    .btn-primary { background: linear-gradient(${gradDir}, var(--primary), var(--secondary)); color: white; box-shadow: 0 4px 20px ${palette.primary}40; }
    .btn-primary:hover { transform: translateY(-2px); box-shadow: 0 6px 30px ${palette.primary}60; }
    .btn-outline { background: transparent; border: 2px solid var(--primary); color: var(--primary); }
    .btn-outline:hover { background: var(--primary)10; transform: translateY(-2px); }
    
    /* Stats */
    .stats { padding: 4rem 0; border-top: 1px solid ${palette.primary}15; border-bottom: 1px solid ${palette.primary}15; }
    .stats-grid { display: grid; grid-template-columns: repeat(4, 1fr); gap: 2rem; text-align: center; }
    .stat-item h3 { font-size: 2.5rem; font-weight: 800; background: linear-gradient(${gradDir}, var(--primary), var(--secondary)); -webkit-background-clip: text; -webkit-text-fill-color: transparent; }
    .stat-item p { opacity: 0.6; margin-top: 0.5rem; }
    
    /* Features */
    .features { padding: 6rem 0; }
    .section-header { text-align: center; max-width: 600px; margin: 0 auto 4rem; }
    .section-badge { display: inline-block; padding: 0.5rem 1rem; background: var(--primary)15; border-radius: 100px; font-size: 0.875rem; color: var(--primary); margin-bottom: 1rem; }
    .section-header h2 { font-size: 2.5rem; font-weight: 800; margin-bottom: 1rem; }
    .section-header p { opacity: 0.7; }
    .features-grid { display: grid; grid-template-columns: repeat(auto-fit, minmax(250px, 1fr)); gap: 2rem; }
    .feature-card { background: var(--card-bg); border: 1px solid ${palette.primary}20; border-radius: 20px; padding: 2rem; transition: all 0.3s; }
    .feature-card:hover { transform: translateY(-5px); border-color: var(--primary); box-shadow: 0 20px 40px ${palette.primary}20; }
    .feature-icon { width: 60px; height: 60px; background: linear-gradient(${gradDir}, var(--primary)20, var(--secondary)20); border-radius: 16px; display: flex; align-items: center; justify-content: center; font-size: 1.5rem; margin-bottom: 1.5rem; }
    .feature-card h3 { font-size: 1.25rem; margin-bottom: 0.75rem; }
    .feature-card p { opacity: 0.7; font-size: 0.95rem; line-height: 1.6; }
    
    /* About */
    .about { padding: 6rem 0; }
    .about-content { display: grid; grid-template-columns: 1fr 1fr; gap: 4rem; align-items: center; }
    .about-text h2 { font-size: 2.5rem; font-weight: 800; margin-bottom: 1.5rem; }
    .about-text p { opacity: 0.8; line-height: 1.8; margin-bottom: 1.5rem; }
    .about-visual { background: linear-gradient(${gradDir}, var(--primary)10, var(--secondary)10); border: 1px solid var(--primary)20; border-radius: 24px; padding: 3rem; display: flex; align-items: center; justify-content: center; min-height: 300px; animation: float 6s ease-in-out infinite; }
    .about-visual .logo-big { font-size: 4rem; font-weight: 900; background: linear-gradient(${gradDir}, var(--primary), var(--secondary)); -webkit-background-clip: text; -webkit-text-fill-color: transparent; }
    .about-visual .about-logo { width: 150px; height: 150px; border-radius: 24px; object-fit: cover; box-shadow: 0 20px 40px ${palette.primary}30; }
    
    /* Contract */
    .contract-section { padding: 6rem 0; }
    .contract-box { background: var(--card-bg); border: 1px solid var(--primary)30; border-radius: 20px; padding: 3rem; text-align: center; max-width: 700px; margin: 0 auto; }
    .contract-box h2 { font-size: 1.5rem; margin-bottom: 1.5rem; }
    .contract-address { background: var(--bg); border: 1px solid var(--primary)20; border-radius: 12px; padding: 1.5rem; font-family: monospace; word-break: break-all; font-size: 0.9rem; margin-bottom: 1.5rem; }
    .copy-btn { background: var(--primary)20; border: 1px solid var(--primary)40; color: var(--primary); padding: 0.75rem 1.5rem; border-radius: 8px; cursor: pointer; transition: all 0.3s; }
    .copy-btn:hover { background: var(--primary)30; }
    
    /* CTA */
    .cta { padding: 6rem 0; text-align: center; background: linear-gradient(${gradDir}, var(--primary)10, var(--secondary)05); border-radius: 32px; margin: 4rem 2rem; }
    .cta h2 { font-size: 2.5rem; font-weight: 800; margin-bottom: 1rem; }
    .cta p { opacity: 0.8; margin-bottom: 2rem; max-width: 500px; margin-left: auto; margin-right: auto; }
    .cta-buttons { display: flex; gap: 1rem; justify-content: center; flex-wrap: wrap; }
    
    /* Footer */
    footer { padding: 4rem 0 2rem; border-top: 1px solid ${palette.primary}15; }
    .footer-content { display: flex; justify-content: space-between; align-items: center; flex-wrap: wrap; gap: 2rem; }
    .footer-links { display: flex; gap: 2rem; }
    .footer-links a { color: var(--text); opacity: 0.6; text-decoration: none; transition: all 0.3s; }
    .footer-links a:hover { opacity: 1; color: var(--primary); }
    .footer-bottom { text-align: center; padding-top: 2rem; margin-top: 2rem; border-top: 1px solid ${palette.primary}10; opacity: 0.5; font-size: 0.875rem; }
    
    @media (max-width: 768px) {
      .stats-grid { grid-template-columns: repeat(2, 1fr); }
      .about-content { grid-template-columns: 1fr; }
      .nav-links { display: none; }
    }
  </style>
</head>
<body>
  <nav>
    <div class="container">
      <div class="logo">${websiteLogo ? `<img src="${websiteLogo}" alt="${websiteProjectName}" class="logo-img">` : ''}${websiteProjectName}</div>
      <div class="nav-links">
        <a href="#features">Features</a>
        <a href="#about">About</a>
        ${websiteContractAddress ? '<a href="#contract">Contract</a>' : ''}
        ${websiteTwitter ? `<a href="${websiteTwitter}" target="_blank">Twitter</a>` : ''}
        ${websiteTelegram ? `<a href="${websiteTelegram}" target="_blank">Telegram</a>` : ''}
      </div>
    </div>
  </nav>

  <section class="hero">
    <div class="hero-content">
      <div class="hero-badge animate delay-1">${shuffledIcons[0]} New Token Launch</div>
      <h1 class="animate delay-2">${websiteProjectName}</h1>
      <p class="animate delay-3">${websiteTagline || 'The next generation decentralized token built for the future of finance.'}</p>
      <div class="hero-buttons animate delay-4">
        ${websiteTwitter ? `<a href="${websiteTwitter}" class="btn btn-primary" target="_blank">${shuffledIcons[1]} Join Community</a>` : '<a href="#about" class="btn btn-primary">Learn More</a>'}
        ${websiteTelegram ? `<a href="${websiteTelegram}" class="btn btn-outline" target="_blank">Telegram</a>` : ''}
        ${websiteContractAddress ? '<a href="#contract" class="btn btn-outline">View Contract</a>' : ''}
      </div>
    </div>
  </section>

  <section class="stats">
    <div class="container">
      <div class="stats-grid">
        ${stats.map(s => `<div class="stat-item"><h3>${s.value}</h3><p>${s.label}</p></div>`).join('')}
      </div>
    </div>
  </section>

  <section class="features" id="features">
    <div class="container">
      <div class="section-header">
        <div class="section-badge">Why Choose Us</div>
        <h2>Built Different</h2>
        <p>Experience the next level of decentralized finance with cutting-edge technology.</p>
      </div>
      <div class="features-grid">
        <div class="feature-card">
          <div class="feature-icon">${shuffledIcons[0]}</div>
          <h3>Lightning Fast</h3>
          <p>Experience instant transactions with our optimized smart contract architecture.</p>
        </div>
        <div class="feature-card">
          <div class="feature-icon">${shuffledIcons[1]}</div>
          <h3>Ultra Secure</h3>
          <p>Audited and verified contracts ensure your assets are always protected.</p>
        </div>
        <div class="feature-card">
          <div class="feature-icon">${shuffledIcons[2]}</div>
          <h3>Community Driven</h3>
          <p>Join thousands of holders in shaping the future of this project.</p>
        </div>
        <div class="feature-card">
          <div class="feature-icon">${shuffledIcons[3]}</div>
          <h3>Fair Launch</h3>
          <p>No presale, no team tokens. Everyone has an equal opportunity.</p>
        </div>
      </div>
    </div>
  </section>

  ${websiteDescription ? `
  <section class="about" id="about">
    <div class="container">
      <div class="about-content">
        <div class="about-text">
          <h2>About ${websiteProjectName}</h2>
          <p>${websiteDescription}</p>
          <div class="hero-buttons">
            ${websiteTwitter ? `<a href="${websiteTwitter}" class="btn btn-primary" target="_blank">Follow Us</a>` : ''}
            ${websiteTelegram ? `<a href="${websiteTelegram}" class="btn btn-outline" target="_blank">Join Telegram</a>` : ''}
          </div>
        </div>
        <div class="about-visual">
          ${websiteLogo ? `<img src="${websiteLogo}" alt="${websiteProjectName}" class="about-logo">` : `<div class="logo-big">${websiteProjectName.charAt(0)}</div>`}
        </div>
      </div>
    </div>
  </section>` : ''}

  ${websiteContractAddress ? `
  <section class="contract-section" id="contract">
    <div class="container">
      <div class="contract-box">
        <h2>Contract Address</h2>
        <div class="contract-address" id="contractAddr">${websiteContractAddress}</div>
        <button class="copy-btn" onclick="navigator.clipboard.writeText('${websiteContractAddress}');this.textContent='Copied!'">Copy Address</button>
      </div>
    </div>
  </section>` : ''}

  <section class="cta">
    <div class="container">
      <h2>Ready to Join?</h2>
      <p>Be part of the ${websiteProjectName} community and help shape the future of decentralized finance.</p>
      <div class="cta-buttons">
        ${websiteTwitter ? `<a href="${websiteTwitter}" class="btn btn-primary" target="_blank">Twitter</a>` : ''}
        ${websiteTelegram ? `<a href="${websiteTelegram}" class="btn btn-outline" target="_blank">Telegram</a>` : ''}
      </div>
    </div>
  </section>

  <footer>
    <div class="container">
      <div class="footer-content">
        <div class="logo">${websiteLogo ? `<img src="${websiteLogo}" alt="${websiteProjectName}" class="logo-img">` : ''}${websiteProjectName}</div>
        <div class="footer-links">
          ${websiteTwitter ? `<a href="${websiteTwitter}" target="_blank">Twitter</a>` : ''}
          ${websiteTelegram ? `<a href="${websiteTelegram}" target="_blank">Telegram</a>` : ''}
          ${websiteContractAddress ? '<a href="#contract">Contract</a>' : ''}
        </div>
      </div>
      <div class="footer-bottom">Built with Astherafi</div>
    </div>
  </footer>
</body>
</html>`;

    setGeneratedWebsite(html);
  };

  const downloadWebsite = () => {
    if (!generatedWebsite) return;
    const blob = new Blob([generatedWebsite], { type: "text/html" });
    const url = URL.createObjectURL(blob);
    const a = document.createElement("a");
    a.href = url;
    a.download = `${websiteProjectName.toLowerCase().replace(/\s+/g, "-")}-landing.html`;
    a.click();
    URL.revokeObjectURL(url);
  };

  // Bundle checker function
  const checkBundle = async () => {
    if (!bundleMintAddress.trim()) {
      setBundleError("Please enter a mint address");
      return;
    }

    setIsCheckingBundle(true);
    setBundleError(null);
    setBundleResult(null);

    try {
      const response = await fetch("/api/bundle-checker", {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({ mintAddress: bundleMintAddress.trim(), chain: bundleChain }),
      });

      const data = await response.json();

      if (!response.ok) {
        throw new Error(data.error || "Failed to check bundle");
      }

      setBundleResult(data);
    } catch (error: any) {
      setBundleError(error.message || "Failed to check bundle");
    } finally {
      setIsCheckingBundle(false);
    }
  };

  // Password strength calculator
  const calculatePasswordScore = (password: string) => {
    let score = 0;
    const feedback: string[] = [];

    if (password.length === 0) {
      setPasswordScore(0);
      setPasswordFeedback([]);
      return;
    }

    // Length checks
    if (password.length >= 8) { score += 20; } else { feedback.push("Use at least 8 characters"); }
    if (password.length >= 12) { score += 10; }
    if (password.length >= 16) { score += 10; }

    // Character variety
    if (/[a-z]/.test(password)) { score += 10; } else { feedback.push("Add lowercase letters"); }
    if (/[A-Z]/.test(password)) { score += 15; } else { feedback.push("Add uppercase letters"); }
    if (/[0-9]/.test(password)) { score += 15; } else { feedback.push("Add numbers"); }
    if (/[^a-zA-Z0-9]/.test(password)) { score += 20; } else { feedback.push("Add special characters (!@#$%^&*)"); }

    // Bonus for mixed patterns
    if (/(?=.*[a-z])(?=.*[A-Z])(?=.*[0-9])(?=.*[^a-zA-Z0-9])/.test(password)) {
      score = Math.min(100, score);
    }

    setPasswordScore(Math.min(100, score));
    setPasswordFeedback(feedback);
  };

  const getScoreColor = () => {
    if (passwordScore >= 80) return "text-green-400";
    if (passwordScore >= 60) return "text-yellow-400";
    if (passwordScore >= 40) return "text-orange-400";
    return "text-red-400";
  };

  const getScoreLabel = () => {
    if (passwordScore >= 80) return "Strong";
    if (passwordScore >= 60) return "Good";
    if (passwordScore >= 40) return "Fair";
    if (passwordScore > 0) return "Weak";
    return "Enter password";
  };

  const getProgressColor = () => {
    if (passwordScore >= 80) return "bg-green-500";
    if (passwordScore >= 60) return "bg-yellow-500";
    if (passwordScore >= 40) return "bg-orange-500";
    return "bg-red-500";
  };

  const handleImageUpload = (e: React.ChangeEvent<HTMLInputElement>) => {
    const file = e.target.files?.[0];
    if (file) {
      if (file.size > 5 * 1024 * 1024) {
        setTokenCreationError("Image must be less than 5MB");
        return;
      }
      setTokenImage(file);
      const reader = new FileReader();
      reader.onloadend = () => {
        setTokenImagePreview(reader.result as string);
      };
      reader.readAsDataURL(file);
    }
  };

  const BSC_CHAIN_ID = "0x38"; // BSC Mainnet

  const switchToBSC = async () => {
    try {
      await (window as any).ethereum.request({
        method: "wallet_switchEthereumChain",
        params: [{ chainId: BSC_CHAIN_ID }],
      });
      return true;
    } catch (switchError: any) {
      if (switchError.code === 4902) {
        try {
          await (window as any).ethereum.request({
            method: "wallet_addEthereumChain",
            params: [{
              chainId: BSC_CHAIN_ID,
              chainName: "BNB Smart Chain",
              nativeCurrency: { name: "BNB", symbol: "BNB", decimals: 18 },
              rpcUrls: ["https://bsc-dataseed.binance.org/"],
              blockExplorerUrls: ["https://bscscan.com"],
            }],
          });
          return true;
        } catch (addError) {
          return false;
        }
      }
      return false;
    }
  };

  const connectWallet = async () => {
    if (typeof window !== "undefined" && (window as any).ethereum) {
      try {
        // First request accounts
        const accounts = await (window as any).ethereum.request({
          method: "eth_requestAccounts",
        });
        
        if (accounts && accounts.length > 0) {
          // Check and switch to BSC network
          const chainId = await (window as any).ethereum.request({ method: "eth_chainId" });
          
          if (chainId !== BSC_CHAIN_ID) {
            setTokenCreationError("Switching to BSC network...");
            const switched = await switchToBSC();
            if (!switched) {
              setTokenCreationError("Please switch to BSC (BNB Smart Chain) network in your wallet");
              return;
            }
            setTokenCreationError(null);
          }
          
          setWalletAddress(accounts[0]);
          setWalletConnected(true);
        }
      } catch (error: any) {
        setTokenCreationError(error.message || "Failed to connect wallet");
      }
    } else {
      setTokenCreationError("Please install MetaMask or another Web3 wallet");
    }
  };

  const copyToClipboard = (text: string) => {
    navigator.clipboard.writeText(text);
  };

  const createToken = async () => {
    if (!tokenName || !tokenSymbol || !tokenDescription) {
      setTokenCreationError("Please fill in all required fields");
      return;
    }

    if (!walletConnected) {
      setTokenCreationError("Please connect your wallet first");
      return;
    }

    // Verify BSC network before proceeding
    const chainId = await (window as any).ethereum.request({ method: "eth_chainId" });
    if (chainId !== BSC_CHAIN_ID) {
      setTokenCreationError("Please switch to BSC (BNB Smart Chain) network");
      const switched = await switchToBSC();
      if (!switched) return;
    }

    setIsCreatingToken(true);
    setTokenCreationError(null);
    setTokenCreationSuccess(null);

    try {
      // Step 1: Get nonce
      const nonceRes = await fetch("/api/token-generator/nonce", {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({ walletAddress }),
      });
      
      if (!nonceRes.ok) {
        const errorData = await nonceRes.json();
        throw new Error(errorData.error || "Failed to generate authentication nonce");
      }
      
      const { nonce } = await nonceRes.json();

      // Step 2: Sign message with wallet
      const message = `You are sign in Meme ${nonce}`;
      const signature = await (window as any).ethereum.request({
        method: "personal_sign",
        params: [message, walletAddress],
      });

      // Step 3: Login to get access token
      const loginRes = await fetch("/api/token-generator/login", {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({ walletAddress, signature, nonce }),
      });

      if (!loginRes.ok) {
        const errorData = await loginRes.json();
        throw new Error(errorData.error || "Authentication failed");
      }

      const { accessToken } = await loginRes.json();

      // Step 4: Create token via API
      const createRes = await fetch("/api/token-generator/create", {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({
          accessToken,
          tokenData: {
            name: tokenName,
            symbol: tokenSymbol,
            description: tokenDescription,
            label: tokenLabel,
            website: tokenWebsite,
            twitter: tokenTwitter,
            telegram: tokenTelegram,
            preSale: preSaleAmount || "0",
            imageUrl: tokenImagePreview || "",
          },
        }),
      });

      const createData = await createRes.json();

      if (!createRes.ok) {
        throw new Error(createData.error || "Token creation failed");
      }

      // Step 5: Submit actual transaction to BSC mainnet via TokenManager2 contract
      if (createData.createArg && createData.signature) {
        const TOKEN_MANAGER_CONTRACT = "0x5c952063c7fc8610FFDB798152D69F0B9550762b";
        const DEPLOY_COST_WEI = "0x2386F26FC10000"; // 0.01 BNB in hex (10000000000000000 wei)
        
        // Build transaction data for TokenManager2.create() function
        // The createArg contains the encoded parameters, signature is the auth
        const txData = createData.createArg;
        
        // Send transaction to BSC mainnet
        const txHash = await (window as any).ethereum.request({
          method: "eth_sendTransaction",
          params: [{
            from: walletAddress,
            to: TOKEN_MANAGER_CONTRACT,
            value: DEPLOY_COST_WEI,
            data: txData,
          }],
        });

        setTokenCreationSuccess({
          message: "Token deployed to BSC mainnet!",
          txHash: txHash,
          contractAddress: createData.contractAddress || "Pending confirmation...",
          explorerUrl: `https://bscscan.com/tx/${txHash}`,
        });
      } else if (createData.txHash) {
        // Token was created directly via API
        setTokenCreationSuccess({
          message: "Token created successfully!",
          txHash: createData.txHash,
          contractAddress: createData.contractAddress,
          explorerUrl: `https://bscscan.com/tx/${createData.txHash}`,
        });
      } else {
        throw new Error("No transaction data received from API");
      }
    } catch (error: any) {
      if (error.code === 4001) {
        setTokenCreationError("Transaction cancelled by user");
      } else if (error.code === -32000 || error.message?.includes("insufficient funds")) {
        setTokenCreationError("Insufficient BNB balance. You need at least 0.01 BNB plus gas fees.");
      } else {
        setTokenCreationError(error.message || "Token creation failed");
      }
    } finally {
      setIsCreatingToken(false);
    }
  };

  const analyzeToken = async () => {
    if (!tokenAddress.trim()) {
      setAnalysisError("Please enter a token address");
      return;
    }

    setIsAnalyzing(true);
    setAnalysisError(null);
    setTokenAnalysis(null);

    try {
      const response = await fetch("/api/analyze-token", {
        method: "POST",
        headers: {
          "Content-Type": "application/json",
        },
        body: JSON.stringify({ 
          tokenAddress: tokenAddress.trim(),
          chain: selectedChain,
        }),
      });

      if (!response.ok) {
        const errorData = await response.json();
        throw new Error(errorData.error || "Failed to analyze token");
      }

      const data = await response.json();
      setTokenAnalysis(data);
      setAiAnalysis(null); // Reset AI analysis when new token is analyzed
    } catch (error) {
      setAnalysisError(error instanceof Error ? error.message : "An error occurred");
    } finally {
      setIsAnalyzing(false);
    }
  };

  // AI-Powered Analysis
  const runAiAnalysis = async () => {
    if (!tokenAnalysis) {
      setAiAnalysisError("Please analyze a token first");
      return;
    }

    setIsAiAnalyzing(true);
    setAiAnalysisError(null);

    try {
      const response = await fetch("/api/ai-analyze", {
        method: "POST",
        headers: {
          "Content-Type": "application/json",
        },
        body: JSON.stringify({
          tokenAddress: tokenAddress.trim(),
          chain: selectedChain,
          tokenData: tokenAnalysis,
        }),
      });

      if (!response.ok) {
        const errorData = await response.json();
        throw new Error(errorData.error || "AI analysis failed");
      }

      const data = await response.json();
      setAiAnalysis(data.analysis);
    } catch (error) {
      setAiAnalysisError(error instanceof Error ? error.message : "AI analysis failed");
    } finally {
      setIsAiAnalyzing(false);
    }
  };

  // Swap Functions
  const getSwapTokenBySymbol = (symbol: string) => {
    return SWAP_TOKENS.find(t => t.symbol === symbol);
  };

  const connectSwapWallet = async () => {
    setSwapError(null);
    
    if (typeof window === "undefined" || !(window as any).ethereum) {
      setSwapError("Please install MetaMask or another Web3 wallet to swap tokens");
      return;
    }

    try {
      const accounts = await (window as any).ethereum.request({
        method: "eth_requestAccounts",
      });
      
      if (!accounts || accounts.length === 0) {
        setSwapError("No accounts found. Please unlock your wallet.");
        return;
      }

      const chainId = await (window as any).ethereum.request({ method: "eth_chainId" });
      
      if (chainId !== BSC_CHAIN_ID) {
        setSwapError("Switching to BSC network...");
        const switched = await switchToBSC();
        if (!switched) {
          setSwapError("Please switch to BSC (BNB Smart Chain) network in your wallet");
          return;
        }
        setSwapError(null);
      }
      
      setWalletAddress(accounts[0]);
      setWalletConnected(true);
      await fetchSwapBalances(accounts[0]);
    } catch (error: any) {
      if (error.code === 4001) {
        setSwapError("Connection rejected. Please approve the connection in your wallet.");
      } else if (error.code === -32002) {
        setSwapError("Connection pending. Please check your wallet for a pending request.");
      } else {
        setSwapError(error.message || "Failed to connect wallet. Please try again.");
      }
    }
  };

  const fetchSwapBalances = async (address: string) => {
    try {
      // Fetch BNB balance
      const bnbBalance = await (window as any).ethereum.request({
        method: "eth_getBalance",
        params: [address, "latest"],
      });
      const bnbInEther = parseInt(bnbBalance, 16) / 1e18;
      setSwapFromBalance(bnbInEther.toFixed(4));
    } catch (error) {
      console.error("Failed to fetch balances:", error);
    }
  };

  const getSwapQuote = async () => {
    if (!swapFromAmount || parseFloat(swapFromAmount) <= 0 || !swapToToken) {
      return;
    }

    setIsLoadingQuote(true);
    try {
      const fromToken = getSwapTokenBySymbol(swapFromToken);
      const toToken = getSwapTokenBySymbol(swapToToken);
      
      if (!fromToken || !toToken) return;

      const amountIn = BigInt(Math.floor(parseFloat(swapFromAmount) * 10 ** fromToken.decimals));
      
      // Build path for getAmountsOut
      const path = fromToken.address === "native" 
        ? [WBNB_ADDRESS, toToken.address]
        : toToken.address === "native"
          ? [fromToken.address, WBNB_ADDRESS]
          : [fromToken.address, WBNB_ADDRESS, toToken.address];

      // ABI for getAmountsOut
      const getAmountsOutData = "0xd06ca61f" + 
        amountIn.toString(16).padStart(64, '0') +
        "0000000000000000000000000000000000000000000000000000000000000040" +
        path.length.toString(16).padStart(64, '0') +
        path.map(addr => addr.toLowerCase().replace('0x', '').padStart(64, '0')).join('');

      const result = await (window as any).ethereum.request({
        method: "eth_call",
        params: [{
          to: PANCAKE_ROUTER_V2,
          data: getAmountsOutData,
        }, "latest"],
      });

      // Parse result - amounts array
      const amountsOut = result.slice(2);
      const lastAmount = amountsOut.slice(-64);
      const amountOut = parseInt(lastAmount, 16) / (10 ** toToken.decimals);
      
      setSwapToAmount(amountOut.toFixed(6));
      setSwapQuote({
        amountIn: swapFromAmount,
        amountOut: amountOut.toFixed(6),
        path,
        priceImpact: "< 0.1%",
      });
    } catch (error) {
      console.error("Failed to get quote:", error);
      setSwapToAmount("0");
    } finally {
      setIsLoadingQuote(false);
    }
  };

  const executeSwap = async () => {
    if (!walletConnected) {
      setSwapError("Please connect your wallet first");
      return;
    }

    if (!swapFromAmount || parseFloat(swapFromAmount) <= 0) {
      setSwapError("Please enter an amount to swap");
      return;
    }

    if (!swapToToken) {
      setSwapError("Please select a token to swap to");
      return;
    }

    setIsSwapping(true);
    setSwapError(null);
    setSwapSuccess(null);

    try {
      const fromToken = getSwapTokenBySymbol(swapFromToken);
      const toToken = getSwapTokenBySymbol(swapToToken);
      
      if (!fromToken || !toToken) throw new Error("Invalid token selection");

      const amountIn = BigInt(Math.floor(parseFloat(swapFromAmount) * 10 ** fromToken.decimals));
      const slippageMultiplier = 1 - (parseFloat(swapSlippage) / 100);
      const amountOutMin = BigInt(Math.floor(parseFloat(swapToAmount) * slippageMultiplier * 10 ** toToken.decimals));
      const deadline = Math.floor(Date.now() / 1000) + 1200; // 20 minutes

      let txHash: string;

      if (fromToken.address === "native") {
        // swapExactETHForTokens
        const path = [WBNB_ADDRESS, toToken.address];
        
        // Encode function call for swapExactETHForTokens
        const functionSelector = "0x7ff36ab5"; // swapExactETHForTokens
        const encodedData = functionSelector +
          amountOutMin.toString(16).padStart(64, '0') +
          "0000000000000000000000000000000000000000000000000000000000000080" +
          walletAddress.toLowerCase().replace('0x', '').padStart(64, '0') +
          deadline.toString(16).padStart(64, '0') +
          path.length.toString(16).padStart(64, '0') +
          path.map(addr => addr.toLowerCase().replace('0x', '').padStart(64, '0')).join('');

        txHash = await (window as any).ethereum.request({
          method: "eth_sendTransaction",
          params: [{
            from: walletAddress,
            to: PANCAKE_ROUTER_V2,
            value: "0x" + amountIn.toString(16),
            data: encodedData,
          }],
        });
      } else if (toToken.address === "native") {
        // First approve the router to spend tokens
        const approveData = "0x095ea7b3" +
          PANCAKE_ROUTER_V2.toLowerCase().replace('0x', '').padStart(64, '0') +
          "ffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff";

        await (window as any).ethereum.request({
          method: "eth_sendTransaction",
          params: [{
            from: walletAddress,
            to: fromToken.address,
            data: approveData,
          }],
        });

        // Wait a bit for approval
        await new Promise(resolve => setTimeout(resolve, 3000));

        // swapExactTokensForETH
        const path = [fromToken.address, WBNB_ADDRESS];
        const functionSelector = "0x18cbafe5"; // swapExactTokensForETH
        const encodedData = functionSelector +
          amountIn.toString(16).padStart(64, '0') +
          amountOutMin.toString(16).padStart(64, '0') +
          "00000000000000000000000000000000000000000000000000000000000000a0" +
          walletAddress.toLowerCase().replace('0x', '').padStart(64, '0') +
          deadline.toString(16).padStart(64, '0') +
          path.length.toString(16).padStart(64, '0') +
          path.map(addr => addr.toLowerCase().replace('0x', '').padStart(64, '0')).join('');

        txHash = await (window as any).ethereum.request({
          method: "eth_sendTransaction",
          params: [{
            from: walletAddress,
            to: PANCAKE_ROUTER_V2,
            data: encodedData,
          }],
        });
      } else {
        // First approve the router to spend tokens
        const approveData = "0x095ea7b3" +
          PANCAKE_ROUTER_V2.toLowerCase().replace('0x', '').padStart(64, '0') +
          "ffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff";

        await (window as any).ethereum.request({
          method: "eth_sendTransaction",
          params: [{
            from: walletAddress,
            to: fromToken.address,
            data: approveData,
          }],
        });

        // Wait a bit for approval
        await new Promise(resolve => setTimeout(resolve, 3000));

        // swapExactTokensForTokens
        const path = [fromToken.address, WBNB_ADDRESS, toToken.address];
        const functionSelector = "0x38ed1739"; // swapExactTokensForTokens
        const encodedData = functionSelector +
          amountIn.toString(16).padStart(64, '0') +
          amountOutMin.toString(16).padStart(64, '0') +
          "00000000000000000000000000000000000000000000000000000000000000a0" +
          walletAddress.toLowerCase().replace('0x', '').padStart(64, '0') +
          deadline.toString(16).padStart(64, '0') +
          path.length.toString(16).padStart(64, '0') +
          path.map(addr => addr.toLowerCase().replace('0x', '').padStart(64, '0')).join('');

        txHash = await (window as any).ethereum.request({
          method: "eth_sendTransaction",
          params: [{
            from: walletAddress,
            to: PANCAKE_ROUTER_V2,
            data: encodedData,
          }],
        });
      }

      setSwapSuccess({
        message: "Swap executed successfully!",
        txHash,
        explorerUrl: `https://bscscan.com/tx/${txHash}`,
      });

      // Refresh balances
      await fetchSwapBalances(walletAddress);
      setSwapFromAmount("");
      setSwapToAmount("");
      setSwapQuote(null);

    } catch (error: any) {
      if (error.code === 4001) {
        setSwapError("Transaction cancelled by user");
      } else if (error.code === -32000 || error.message?.includes("insufficient funds")) {
        setSwapError("Insufficient balance for this swap");
      } else {
        setSwapError(error.message || "Swap failed");
      }
    } finally {
      setIsSwapping(false);
    }
  };

  const switchSwapTokens = () => {
    const tempFrom = swapFromToken;
    const tempFromAmount = swapFromAmount;
    setSwapFromToken(swapToToken);
    setSwapToToken(tempFrom);
    setSwapFromAmount(swapToAmount);
    setSwapToAmount(tempFromAmount);
  };

  return (
    <div className="h-screen w-full flex overflow-hidden relative" style={{ background: "linear-gradient(180deg, #0a0a0f 0%, #0d1117 30%, #0a0a0f 100%)" }}>
      {/* Particle Canvas Background */}
      <canvas
        ref={canvasRef}
        className="fixed inset-0 pointer-events-none z-0"
      />

      {/* Aurora Background Effects */}
      <div className="fixed inset-0 overflow-hidden pointer-events-none z-0">
        <div 
          className="absolute w-[600px] h-[600px] bg-gradient-to-br from-cyan-500/15 via-cyan-400/5 to-transparent rounded-full blur-3xl"
          style={{ 
            top: "-10%", 
            left: "-5%",
            animation: "dashAurora1 20s ease-in-out infinite"
          }} 
        />
        <div 
          className="absolute w-[500px] h-[500px] bg-gradient-to-bl from-purple-600/10 via-purple-500/5 to-transparent rounded-full blur-3xl"
          style={{ 
            top: "20%", 
            right: "-10%",
            animation: "dashAurora2 25s ease-in-out infinite"
          }} 
        />
        <div 
          className="absolute w-[400px] h-[400px] bg-gradient-to-tl from-cyan-400/10 via-purple-500/5 to-transparent rounded-full blur-3xl"
          style={{ 
            bottom: "10%", 
            right: "20%",
            animation: "dashAurora3 18s ease-in-out infinite"
          }} 
        />
      </div>

      <style>{`
        @keyframes dashAurora1 {
          0%, 100% { transform: translate(0, 0) scale(1); opacity: 0.5; }
          50% { transform: translate(5%, 10%) scale(1.1); opacity: 0.7; }
        }
        @keyframes dashAurora2 {
          0%, 100% { transform: translate(0, 0) scale(1); opacity: 0.4; }
          50% { transform: translate(-10%, -5%) scale(1.05); opacity: 0.6; }
        }
        @keyframes dashAurora3 {
          0%, 100% { transform: translate(0, 0) scale(1); opacity: 0.5; }
          50% { transform: translate(5%, -10%) scale(0.95); opacity: 0.7; }
        }
      `}</style>

      {/* Mobile Overlay - visible when mobile menu is open */}
      {!isDesktop && mobileMenuOpen && (
        <div 
          className="fixed inset-0 bg-black/60 z-[1040]" 
          onClick={() => setMobileMenuOpen(false)} 
        />
      )}
      
      {/* Mobile Sidebar - fixed overlay, slides in from left */}
      {!isDesktop && (
        <aside
          className={`
            fixed left-0 top-0 h-full w-72 border-r border-primary/10 bg-card/80 backdrop-blur-md flex-col z-[1045]
            transition-transform duration-300 ease-in-out flex
            ${mobileMenuOpen ? 'translate-x-0 shadow-2xl' : '-translate-x-full'}
          `}
        >
          <div className="h-16 px-4 flex items-center justify-between border-b border-primary/10 shrink-0">
            <div className="flex items-center gap-3">
              <Link href="/">
                <img src={astherfiLogo} alt="Astherafi" className="w-10 h-10 rounded-xl shrink-0 cursor-pointer hover:opacity-80 transition-opacity" />
              </Link>
              <div>
                <h1 className="font-display font-bold text-lg text-foreground">Astherafi</h1>
                <p className="text-xs text-muted-foreground">{t("dashboard.version")}</p>
              </div>
            </div>
            <div className="flex items-center gap-1">
              <Link href="/">
                <Button
                  variant="ghost"
                  size="icon"
                  className="text-muted-foreground hover:text-foreground"
                  data-testid="button-back-home-mobile"
                >
                  <ArrowLeft className="w-5 h-5" />
                </Button>
              </Link>
              <Button
                variant="ghost"
                size="icon"
                onClick={() => setMobileMenuOpen(false)}
                className="text-muted-foreground hover:text-foreground"
                data-testid="button-close-mobile-menu"
              >
                <ChevronLeft className="w-5 h-5" />
              </Button>
            </div>
          </div>
          <nav className="flex-1 p-3 space-y-1 overflow-y-auto">
            {translatedMenuItems.map((item) => (
              <button
                key={item.id}
                onClick={() => {
                  setActiveItem(item.id);
                  setMobileMenuOpen(false);
                }}
                data-testid={`nav-mobile-${item.id}`}
                className={`
                  w-full flex items-center gap-3 px-4 py-3 rounded-lg text-left transition-all
                  ${activeItem === item.id 
                    ? "bg-primary/15 text-primary border border-primary/30" 
                    : "text-foreground/70 hover:text-foreground hover:bg-muted/50 border border-transparent"
                  }
                `}
              >
                <span className={activeItem === item.id ? "text-primary" : "text-foreground/60"}>{item.icon}</span>
                <span className="font-medium text-sm">{item.label}</span>
                {item.badge && (
                  <span className="ml-auto px-1.5 py-0.5 text-[10px] font-semibold rounded bg-primary/20 text-primary border border-primary/30">
                    {item.badge}
                  </span>
                )}
                {activeItem === item.id && !item.badge && <ChevronRight className="w-4 h-4 ml-auto text-primary" />}
              </button>
            ))}
          </nav>
        </aside>
      )}
      
      {/* Desktop Sidebar - toggles between full and collapsed */}
      {isDesktop && (
        <aside 
          className={`
            relative h-full border-r border-primary/10 bg-card/80 backdrop-blur-md flex flex-col shrink-0
            transition-all duration-300 ease-in-out
            ${sidebarCollapsed ? 'w-[70px]' : 'w-64'}
          `}
        >
          <div className={`h-16 flex items-center justify-between border-b border-primary/10 shrink-0 ${sidebarCollapsed ? 'px-3' : 'px-4'}`}>
            <div className={`flex items-center ${sidebarCollapsed ? '' : 'gap-3'}`}>
              <Link href="/">
                <img src={astherfiLogo} alt="Astherafi" className="w-10 h-10 rounded-xl shrink-0 cursor-pointer hover:opacity-80 transition-opacity" />
              </Link>
              {!sidebarCollapsed && (
                <div>
                  <h1 className="font-display font-bold text-lg text-foreground">Astherafi</h1>
                  <p className="text-xs text-muted-foreground">{t("dashboard.version")}</p>
                </div>
              )}
            </div>
            {!sidebarCollapsed && (
              <Link href="/">
                <Button
                  variant="ghost"
                  size="icon"
                  className="text-muted-foreground hover:text-foreground"
                  data-testid="button-back-home-desktop"
                  title="Back to Home"
                >
                  <ArrowLeft className="w-5 h-5" />
                </Button>
              </Link>
            )}
          </div>
          <nav className={`flex-1 space-y-1 overflow-y-auto ${sidebarCollapsed ? 'p-2' : 'p-3'}`}>
            {translatedMenuItems.map((item) => (
              <button
                key={item.id}
                onClick={() => setActiveItem(item.id)}
                data-testid={`nav-${item.id}`}
                title={sidebarCollapsed ? item.label : undefined}
                className={`
                  w-full flex items-center rounded-lg text-left transition-all
                  ${sidebarCollapsed ? 'justify-center p-3' : 'gap-3 px-4 py-3'}
                  ${activeItem === item.id 
                    ? "bg-primary/15 text-primary border border-primary/30" 
                    : "text-foreground/70 hover:text-foreground hover:bg-muted/50 border border-transparent"
                  }
                `}
              >
                <span className={activeItem === item.id ? "text-primary" : "text-foreground/60"}>{item.icon}</span>
                {!sidebarCollapsed && (
                  <>
                    <span className="font-medium text-sm">{item.label}</span>
                    {item.badge && (
                      <span className="ml-auto px-1.5 py-0.5 text-[10px] font-semibold rounded bg-primary/20 text-primary border border-primary/30">
                        {item.badge}
                      </span>
                    )}
                    {activeItem === item.id && !item.badge && <ChevronRight className="w-4 h-4 ml-auto text-primary" />}
                  </>
                )}
              </button>
            ))}
          </nav>
        </aside>
      )}

      {/* Main Content Area */}
      <div className="flex-1 flex flex-col h-full w-full min-w-0 overflow-hidden">
        {/* Header with Marquee */}
        <header className={`border-b border-primary/10 flex items-center justify-between bg-card/70 backdrop-blur-md shrink-0 ${isDesktop ? 'h-16' : 'h-14'}`}>
          <div className="flex items-center gap-3 px-4 shrink-0">
            {/* Mobile menu button */}
            {!isDesktop && (
              <Button
                variant="ghost"
                size="icon"
                className="text-muted-foreground hover:text-foreground"
                onClick={() => setMobileMenuOpen(true)}
                data-testid="button-mobile-menu"
              >
                <Menu className="w-5 h-5" />
              </Button>
            )}
            {/* Desktop sidebar toggle */}
            {isDesktop && (
              <Button
                variant="ghost"
                size="icon"
                className="text-muted-foreground hover:text-foreground"
                onClick={() => setSidebarCollapsed(!sidebarCollapsed)}
                data-testid="button-toggle-sidebar"
              >
                {sidebarCollapsed ? <PanelLeft className="w-5 h-5" /> : <PanelLeftClose className="w-5 h-5" />}
              </Button>
            )}
            {/* Title on mobile/tablet */}
            {!isDesktop && <h1 className="font-display font-bold text-lg">Astherafi</h1>}
          </div>
          
          {/* Crypto Prices Marquee - inside header */}
          {cryptoPrices.length > 0 && (
            <div className="flex-1 overflow-hidden mx-4">
              <div className="flex animate-marquee whitespace-nowrap h-full items-center">
                {[...cryptoPrices, ...cryptoPrices].map((coin, index) => (
                  <div key={`${coin.id}-${index}`} className="inline-flex items-center gap-2 mx-4">
                    <img src={coin.image} alt={coin.name} className="w-4 h-4 rounded-full" />
                    <span className="text-xs font-medium text-foreground/80">{coin.symbol.toUpperCase()}</span>
                    <span className="text-xs text-foreground">${coin.current_price.toLocaleString(undefined, { minimumFractionDigits: 2, maximumFractionDigits: coin.current_price < 1 ? 6 : 2 })}</span>
                    <span className={`text-xs font-medium ${coin.price_change_percentage_24h >= 0 ? 'text-green-400' : 'text-red-400'}`}>
                      {coin.price_change_percentage_24h >= 0 ? '+' : ''}{coin.price_change_percentage_24h.toFixed(2)}%
                    </span>
                  </div>
                ))}
              </div>
            </div>
          )}
          
          {/* Language Switcher */}
          <div className="px-4 shrink-0">
            <LanguageSwitcher />
          </div>
        </header>

        <main className="flex-1 p-3 sm:p-4 md:p-6 overflow-y-auto">
          {activeItem !== "dashboard" && (
            <motion.div initial={{ y: 10, opacity: 0 }} animate={{ y: 0, opacity: 1 }} className="mb-4 sm:mb-8">
              <h2 className="text-xl sm:text-2xl md:text-3xl font-display font-bold text-glow-cyan">{activeMenu?.label}</h2>
              <p className="text-muted-foreground mt-1 text-sm sm:text-base">{activeMenu?.description}</p>
            </motion.div>
          )}

          <motion.div
            key={`content-${activeItem}`}
            initial={{ opacity: 0, scale: 0.98 }}
            animate={{ opacity: 1, scale: 1 }}
            className="rounded-xl sm:rounded-2xl bg-card/60 backdrop-blur-sm border border-border/50 p-4 sm:p-6 md:p-8 min-h-[300px] sm:min-h-[400px] relative overflow-hidden shadow-sm"
          >
            <div className="absolute top-0 right-0 w-48 sm:w-96 h-48 sm:h-96 bg-gradient-to-bl from-primary/5 to-transparent rounded-full blur-3xl pointer-events-none" />
            <div className="relative z-10">
              {activeItem === "dashboard" && (
                <div className="space-y-6 sm:space-y-8">
                  <div className="flex flex-col sm:flex-row items-start sm:items-center gap-3 sm:gap-4">
                    <div className="w-12 h-12 sm:w-16 sm:h-16 rounded-xl sm:rounded-2xl cyber-gradient flex items-center justify-center glow-cyan shrink-0">
                      <TrendingUp className="w-6 h-6 sm:w-8 sm:h-8 text-background" />
                    </div>
                    <div>
                      <h2 className="text-xl sm:text-2xl md:text-3xl font-display font-bold">{t("dashboard.overview")}</h2>
                      <p className="text-muted-foreground text-sm sm:text-base">{t("dashboard.selectTool")}</p>
                    </div>
                  </div>
                  <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-3 sm:gap-4 md:gap-6">
                    {translatedMenuItems.filter(i => i.id !== 'dashboard').map((item) => (
                      <motion.button
                        key={item.id}
                        whileHover={{ scale: 1.02 }}
                        whileTap={{ scale: 0.98 }}
                        onClick={() => setActiveItem(item.id)}
                        className="p-4 sm:p-6 rounded-lg sm:rounded-xl bg-muted/20 border border-border/50 text-left hover:bg-muted/30 transition-all group"
                      >
                        <div className="w-10 h-10 sm:w-12 sm:h-12 rounded-lg bg-primary/10 text-primary flex items-center justify-center mb-3 sm:mb-4 group-hover:bg-primary/20 transition-colors">
                          {item.icon}
                        </div>
                        <h4 className="font-display font-semibold text-base sm:text-lg mb-1 sm:mb-2">{item.label}</h4>
                        <p className="text-xs sm:text-sm text-muted-foreground line-clamp-2">{item.description}</p>
                      </motion.button>
                    ))}
                  </div>
                </div>
              )}

              {activeItem === "token-analyzer" && (
                <div className="space-y-4">
                  {/* Chain Selection */}
                  <div className="flex flex-wrap gap-2 mb-2">
                    {CHAINS.map((chain) => (
                      <Button
                        key={chain.id}
                        variant={selectedChain === chain.id ? "default" : "outline"}
                        onClick={() => setSelectedChain(chain.id)}
                        className={`${selectedChain === chain.id ? "cyber-gradient text-background" : ""} flex items-center gap-2`}
                        data-testid={`button-chain-${chain.id}`}
                      >
                        {(chain as any).image ? (
                          <img src={(chain as any).image} alt={chain.name} className="w-5 h-5" />
                        ) : (
                          <span>{chain.icon}</span>
                        )}
                        {chain.name}
                      </Button>
                    ))}
                  </div>
                  <div className="flex flex-col sm:flex-row gap-3">
                    <div className="flex-1 flex gap-2">
                      <Input
                        placeholder="Search for tokens, pairs, or addresses..."
                        className="flex-1 bg-muted/30 border-border/50 h-12 font-mono text-sm"
                        value={tokenAddress}
                        onChange={(e) => setTokenAddress(e.target.value)}
                        onKeyDown={(e) => e.key === "Enter" && analyzeToken()}
                      />
                      <Button 
                        className="cyber-gradient text-background font-semibold px-6 h-12 glow-cyan shrink-0"
                        onClick={analyzeToken}
                        disabled={isAnalyzing}
                        data-testid="button-analyze-token"
                      >
                        {isAnalyzing ? <Loader2 className="w-4 h-4 animate-spin" /> : t("dashboard.analyze")}
                      </Button>
                    </div>
                  </div>

                  {analysisError && (
                    <div className="p-4 rounded-lg bg-red-500/10 border border-red-500/30 flex items-center gap-3">
                      <XCircle className="w-5 h-5 text-red-400" />
                      <p className="text-sm text-red-300">{analysisError}</p>
                    </div>
                  )}

                  {tokenAnalysis && (
                    <div className="space-y-6 mt-6">
                      {/* Token Header with Logo */}
                      <div className="p-4 rounded-lg bg-gradient-to-r from-primary/10 to-secondary/10 border border-primary/20">
                        <div className="flex flex-col sm:flex-row items-start sm:items-center justify-between gap-4">
                          <div className="flex items-center gap-3">
                            {tokenAnalysis.logo ? (
                              <img 
                                src={tokenAnalysis.logo} 
                                alt={tokenAnalysis.name}
                                className="w-12 h-12 rounded-full object-cover bg-muted"
                                onError={(e) => {
                                  (e.target as HTMLImageElement).style.display = 'none';
                                  (e.target as HTMLImageElement).nextElementSibling?.classList.remove('hidden');
                                }}
                              />
                            ) : null}
                            <div className={`w-12 h-12 rounded-full cyber-gradient flex items-center justify-center text-background font-bold ${tokenAnalysis.logo ? 'hidden' : ''}`}>
                              {tokenAnalysis.symbol?.[0] || "?"}
                            </div>
                            <div>
                              <h4 className="font-display font-semibold text-lg">{tokenAnalysis.name}</h4>
                              <p className="text-xs text-muted-foreground">{tokenAnalysis.symbol} • {tokenAnalysis.chain}</p>
                              <p className="text-xs text-muted-foreground font-mono mt-1 break-all">{tokenAnalysis.address}</p>
                            </div>
                          </div>
                          <div className="flex items-center gap-2">
                            <div className={`px-3 py-1 rounded-full text-xs font-medium ${tokenAnalysis.simulation?.isHoneypot ? 'bg-red-500/20 text-red-400' : 'bg-green-500/20 text-green-400'}`}>
                              {tokenAnalysis.simulation?.isHoneypot ? 'HONEYPOT' : 'SAFE'}
                            </div>
                          </div>
                        </div>
                        
                        {/* Social Links */}
                        {tokenAnalysis.socials && Object.keys(tokenAnalysis.socials).length > 0 && (
                          <div className="flex flex-wrap items-center gap-2 mt-4 pt-4 border-t border-border/30">
                            <span className="text-xs text-muted-foreground mr-2">Links:</span>
                            {tokenAnalysis.socials.website && (
                              <a 
                                href={tokenAnalysis.socials.website} 
                                target="_blank" 
                                rel="noopener noreferrer"
                                className="flex items-center gap-1 px-3 py-1.5 rounded-full bg-muted/30 hover:bg-muted/50 text-xs transition-colors"
                              >
                                <Globe className="w-3 h-3" />
                                Website
                                <ExternalLink className="w-3 h-3 opacity-50" />
                              </a>
                            )}
                            {tokenAnalysis.socials.twitter && (
                              <a 
                                href={tokenAnalysis.socials.twitter} 
                                target="_blank" 
                                rel="noopener noreferrer"
                                className="flex items-center gap-1 px-3 py-1.5 rounded-full bg-muted/30 hover:bg-muted/50 text-xs transition-colors"
                              >
                                <svg className="w-3 h-3" viewBox="0 0 24 24" fill="currentColor">
                                  <path d="M18.244 2.25h3.308l-7.227 8.26 8.502 11.24H16.17l-5.214-6.817L4.99 21.75H1.68l7.73-8.835L1.254 2.25H8.08l4.713 6.231zm-1.161 17.52h1.833L7.084 4.126H5.117z"/>
                                </svg>
                                Twitter
                                <ExternalLink className="w-3 h-3 opacity-50" />
                              </a>
                            )}
                            {tokenAnalysis.socials.telegram && (
                              <a 
                                href={tokenAnalysis.socials.telegram} 
                                target="_blank" 
                                rel="noopener noreferrer"
                                className="flex items-center gap-1 px-3 py-1.5 rounded-full bg-muted/30 hover:bg-muted/50 text-xs transition-colors"
                              >
                                <MessageCircle className="w-3 h-3" />
                                Telegram
                                <ExternalLink className="w-3 h-3 opacity-50" />
                              </a>
                            )}
                            {tokenAnalysis.socials.discord && (
                              <a 
                                href={tokenAnalysis.socials.discord} 
                                target="_blank" 
                                rel="noopener noreferrer"
                                className="flex items-center gap-1 px-3 py-1.5 rounded-full bg-muted/30 hover:bg-muted/50 text-xs transition-colors"
                              >
                                <MessageCircle className="w-3 h-3" />
                                Discord
                                <ExternalLink className="w-3 h-3 opacity-50" />
                              </a>
                            )}
                          </div>
                        )}
                      </div>
                      
                      {/* Market Data */}
                      <div className="p-5 rounded-lg bg-muted/20 border border-border/50">
                        <h5 className="text-primary font-display font-semibold text-sm uppercase tracking-wider mb-4">Market Data</h5>
                        <div className="grid grid-cols-2 sm:grid-cols-4 gap-4">
                          <div>
                            <p className="text-xs text-muted-foreground uppercase mb-1">Price</p>
                            <p className="text-primary font-semibold">${tokenAnalysis.technical?.priceUsd || "0.00"}</p>
                          </div>
                          <div>
                            <p className="text-xs text-muted-foreground uppercase mb-1">Market Cap</p>
                            <p className="text-primary font-semibold">${Number(tokenAnalysis.technical?.marketCap || 0).toLocaleString()}</p>
                          </div>
                          <div>
                            <p className="text-xs text-muted-foreground uppercase mb-1">24h Volume</p>
                            <p className="text-primary font-semibold">${Number(tokenAnalysis.technical?.volume24h || 0).toLocaleString()}</p>
                          </div>
                          <div>
                            <p className="text-xs text-muted-foreground uppercase mb-1">Liquidity</p>
                            <p className="text-primary font-semibold">${Number(tokenAnalysis.technical?.liquidity || 0).toLocaleString()}</p>
                          </div>
                        </div>
                        {tokenAnalysis.technical && (
                          <div className="grid grid-cols-4 gap-4 mt-4 pt-4 border-t border-border/30">
                            <div className="text-center">
                              <p className="text-xs text-muted-foreground mb-1">5m</p>
                              <p className={`text-sm font-medium ${(tokenAnalysis.technical.priceChange5m || 0) >= 0 ? 'text-green-400' : 'text-red-400'}`}>
                                {(tokenAnalysis.technical.priceChange5m || 0) >= 0 ? '+' : ''}{tokenAnalysis.technical.priceChange5m || 0}%
                              </p>
                            </div>
                            <div className="text-center">
                              <p className="text-xs text-muted-foreground mb-1">1h</p>
                              <p className={`text-sm font-medium ${(tokenAnalysis.technical.priceChange1h || 0) >= 0 ? 'text-green-400' : 'text-red-400'}`}>
                                {(tokenAnalysis.technical.priceChange1h || 0) >= 0 ? '+' : ''}{tokenAnalysis.technical.priceChange1h || 0}%
                              </p>
                            </div>
                            <div className="text-center">
                              <p className="text-xs text-muted-foreground mb-1">6h</p>
                              <p className={`text-sm font-medium ${(tokenAnalysis.technical.priceChange6h || 0) >= 0 ? 'text-green-400' : 'text-red-400'}`}>
                                {(tokenAnalysis.technical.priceChange6h || 0) >= 0 ? '+' : ''}{tokenAnalysis.technical.priceChange6h || 0}%
                              </p>
                            </div>
                            <div className="text-center">
                              <p className="text-xs text-muted-foreground mb-1">24h</p>
                              <p className={`text-sm font-medium ${(tokenAnalysis.technical.priceChange24h || 0) >= 0 ? 'text-green-400' : 'text-red-400'}`}>
                                {(tokenAnalysis.technical.priceChange24h || 0) >= 0 ? '+' : ''}{tokenAnalysis.technical.priceChange24h || 0}%
                              </p>
                            </div>
                          </div>
                        )}
                      </div>

                      {/* Simulation Results */}
                      <div className="p-5 rounded-lg bg-muted/20 border border-border/50">
                        <h5 className="text-primary font-display font-semibold text-sm uppercase tracking-wider mb-4">Simulation Results</h5>
                        <div className="grid grid-cols-2 sm:grid-cols-3 gap-x-6 gap-y-4">
                          <div>
                            <p className="text-xs text-muted-foreground uppercase mb-1">Buy Tax</p>
                            <p className="text-primary font-semibold">{tokenAnalysis.simulation?.buyTax || "0.0%"}</p>
                          </div>
                          <div>
                            <p className="text-xs text-muted-foreground uppercase mb-1">Buy Gas</p>
                            <p className="text-primary font-semibold">{Number(tokenAnalysis.simulation?.buyGas || 0).toLocaleString()}</p>
                          </div>
                          <div>
                            <p className="text-xs text-muted-foreground uppercase mb-1">Buy Limit</p>
                            <p className={`font-semibold ${tokenAnalysis.simulation?.buyLimit === "BLOCKED" ? 'text-red-400' : 'text-green-400'}`}>
                              {tokenAnalysis.simulation?.buyLimit || "NONE DETECTED"}
                            </p>
                          </div>
                          <div>
                            <p className="text-xs text-muted-foreground uppercase mb-1">Sell Tax</p>
                            <p className="text-primary font-semibold">{tokenAnalysis.simulation?.sellTax || "0.0%"}</p>
                          </div>
                          <div>
                            <p className="text-xs text-muted-foreground uppercase mb-1">Sell Gas</p>
                            <p className="text-primary font-semibold">{Number(tokenAnalysis.simulation?.sellGas || 0).toLocaleString()}</p>
                          </div>
                          <div>
                            <p className="text-xs text-muted-foreground uppercase mb-1">Sell Limit</p>
                            <p className={`font-semibold ${tokenAnalysis.simulation?.sellLimit === "RESTRICTED" ? 'text-red-400' : 'text-green-400'}`}>
                              {tokenAnalysis.simulation?.sellLimit || "NONE DETECTED"}
                            </p>
                          </div>
                          <div>
                            <p className="text-xs text-muted-foreground uppercase mb-1">Transfer Tax</p>
                            <p className="text-primary font-semibold">{tokenAnalysis.simulation?.transferTax || "0.0%"}</p>
                          </div>
                          <div>
                            <p className="text-xs text-muted-foreground uppercase mb-1">Source Code</p>
                            <p className={`font-semibold flex items-center gap-1 ${tokenAnalysis.simulation?.sourceCode === "OPEN SOURCE" ? 'text-green-400' : 'text-yellow-400'}`}>
                              {tokenAnalysis.simulation?.sourceCode === "OPEN SOURCE" ? <CheckCircle className="w-4 h-4" /> : <AlertCircle className="w-4 h-4" />}
                              {tokenAnalysis.simulation?.sourceCode || "NOT VERIFIED"}
                            </p>
                          </div>
                        </div>
                      </div>

                      {/* Recent Holder Analysis */}
                      <div className="p-5 rounded-lg bg-muted/20 border border-border/50">
                        <h5 className="text-primary font-display font-semibold text-sm uppercase tracking-wider mb-4">Recent Holder Analysis</h5>
                        <div className="grid grid-cols-2 sm:grid-cols-3 gap-x-6 gap-y-4">
                          <div>
                            <p className="text-xs text-muted-foreground uppercase mb-1">Holders Analysed</p>
                            <p className="text-primary font-semibold">{Number(tokenAnalysis.holders?.total || 0).toLocaleString()}</p>
                          </div>
                          <div>
                            <p className="text-xs text-muted-foreground uppercase mb-1">Can Sell</p>
                            <p className="text-green-400 font-semibold">{Number(tokenAnalysis.holders?.canSell || 0).toLocaleString()}</p>
                          </div>
                          <div>
                            <p className="text-xs text-muted-foreground uppercase mb-1">Can't Sell</p>
                            <p className={`font-semibold ${(tokenAnalysis.holders?.cantSell || 0) > 0 ? 'text-red-400' : 'text-green-400'}`}>
                              {Number(tokenAnalysis.holders?.cantSell || 0).toLocaleString()}
                            </p>
                          </div>
                          <div>
                            <p className="text-xs text-muted-foreground uppercase mb-1">Siphoned</p>
                            <p className={`font-semibold ${(tokenAnalysis.holders?.siphoned || 0) > 0 ? 'text-red-400' : 'text-primary'}`}>
                              {Number(tokenAnalysis.holders?.siphoned || 0).toLocaleString()}
                            </p>
                          </div>
                          <div>
                            <p className="text-xs text-muted-foreground uppercase mb-1">Average Tax</p>
                            <p className="text-primary font-semibold">{tokenAnalysis.holders?.averageTax || "0.0%"}</p>
                          </div>
                          <div>
                            <p className="text-xs text-muted-foreground uppercase mb-1">Highest Tax</p>
                            <p className="text-primary font-semibold">{tokenAnalysis.holders?.highestTax || "0.0%"}</p>
                          </div>
                          <div>
                            <p className="text-xs text-muted-foreground uppercase mb-1">Average Gas</p>
                            <p className="text-primary font-semibold">{Number(tokenAnalysis.holders?.averageGas || 0).toLocaleString()}</p>
                          </div>
                        </div>
                      </div>

                      {/* Security Flags */}
                      <div className="p-5 rounded-lg bg-muted/20 border border-border/50">
                        <h5 className="text-primary font-display font-semibold text-sm uppercase tracking-wider mb-4">Security Flags</h5>
                        <div className="grid grid-cols-2 sm:grid-cols-4 gap-3">
                          <div className={`p-3 rounded-lg ${tokenAnalysis.security?.isProxy ? 'bg-yellow-500/10 border border-yellow-500/30' : 'bg-green-500/10 border border-green-500/30'}`}>
                            <p className="text-xs text-muted-foreground mb-1">Proxy Contract</p>
                            <p className={`font-semibold text-sm ${tokenAnalysis.security?.isProxy ? 'text-yellow-400' : 'text-green-400'}`}>
                              {tokenAnalysis.security?.isProxy ? 'YES' : 'NO'}
                            </p>
                          </div>
                          <div className={`p-3 rounded-lg ${tokenAnalysis.security?.isMintable ? 'bg-yellow-500/10 border border-yellow-500/30' : 'bg-green-500/10 border border-green-500/30'}`}>
                            <p className="text-xs text-muted-foreground mb-1">Mintable</p>
                            <p className={`font-semibold text-sm ${tokenAnalysis.security?.isMintable ? 'text-yellow-400' : 'text-green-400'}`}>
                              {tokenAnalysis.security?.isMintable ? 'YES' : 'NO'}
                            </p>
                          </div>
                          <div className={`p-3 rounded-lg ${tokenAnalysis.security?.hiddenOwner ? 'bg-red-500/10 border border-red-500/30' : 'bg-green-500/10 border border-green-500/30'}`}>
                            <p className="text-xs text-muted-foreground mb-1">Hidden Owner</p>
                            <p className={`font-semibold text-sm ${tokenAnalysis.security?.hiddenOwner ? 'text-red-400' : 'text-green-400'}`}>
                              {tokenAnalysis.security?.hiddenOwner ? 'YES' : 'NO'}
                            </p>
                          </div>
                          <div className={`p-3 rounded-lg ${tokenAnalysis.security?.canTakeBackOwnership ? 'bg-red-500/10 border border-red-500/30' : 'bg-green-500/10 border border-green-500/30'}`}>
                            <p className="text-xs text-muted-foreground mb-1">Take Back Ownership</p>
                            <p className={`font-semibold text-sm ${tokenAnalysis.security?.canTakeBackOwnership ? 'text-red-400' : 'text-green-400'}`}>
                              {tokenAnalysis.security?.canTakeBackOwnership ? 'YES' : 'NO'}
                            </p>
                          </div>
                          <div className={`p-3 rounded-lg ${tokenAnalysis.security?.ownerCanChangeBalance ? 'bg-red-500/10 border border-red-500/30' : 'bg-green-500/10 border border-green-500/30'}`}>
                            <p className="text-xs text-muted-foreground mb-1">Owner Change Balance</p>
                            <p className={`font-semibold text-sm ${tokenAnalysis.security?.ownerCanChangeBalance ? 'text-red-400' : 'text-green-400'}`}>
                              {tokenAnalysis.security?.ownerCanChangeBalance ? 'YES' : 'NO'}
                            </p>
                          </div>
                          <div className={`p-3 rounded-lg ${tokenAnalysis.security?.isBlacklisted ? 'bg-yellow-500/10 border border-yellow-500/30' : 'bg-green-500/10 border border-green-500/30'}`}>
                            <p className="text-xs text-muted-foreground mb-1">Blacklist</p>
                            <p className={`font-semibold text-sm ${tokenAnalysis.security?.isBlacklisted ? 'text-yellow-400' : 'text-green-400'}`}>
                              {tokenAnalysis.security?.isBlacklisted ? 'YES' : 'NO'}
                            </p>
                          </div>
                          <div className={`p-3 rounded-lg ${tokenAnalysis.security?.hasTradingCooldown ? 'bg-yellow-500/10 border border-yellow-500/30' : 'bg-green-500/10 border border-green-500/30'}`}>
                            <p className="text-xs text-muted-foreground mb-1">Trading Cooldown</p>
                            <p className={`font-semibold text-sm ${tokenAnalysis.security?.hasTradingCooldown ? 'text-yellow-400' : 'text-green-400'}`}>
                              {tokenAnalysis.security?.hasTradingCooldown ? 'YES' : 'NO'}
                            </p>
                          </div>
                          <div className={`p-3 rounded-lg ${tokenAnalysis.security?.lpLocked ? 'bg-green-500/10 border border-green-500/30' : 'bg-yellow-500/10 border border-yellow-500/30'}`}>
                            <p className="text-xs text-muted-foreground mb-1">LP Locked</p>
                            <p className={`font-semibold text-sm ${tokenAnalysis.security?.lpLocked ? 'text-green-400' : 'text-yellow-400'}`}>
                              {tokenAnalysis.security?.lpLocked ? 'YES' : 'NO'}
                            </p>
                          </div>
                        </div>
                      </div>

                      {/* Sentiment Data */}
                      {tokenAnalysis.sentiment && (
                        <div className="p-5 rounded-lg bg-muted/20 border border-border/50">
                          <h5 className="text-primary font-display font-semibold text-sm uppercase tracking-wider mb-4">Trading Sentiment (24h)</h5>
                          <div className="grid grid-cols-2 sm:grid-cols-4 gap-4">
                            <div>
                              <p className="text-xs text-muted-foreground uppercase mb-1">Overall</p>
                              <p className={`font-semibold ${tokenAnalysis.sentiment.overall === 'Bullish' ? 'text-green-400' : tokenAnalysis.sentiment.overall === 'Bearish' ? 'text-red-400' : 'text-yellow-400'}`}>
                                {tokenAnalysis.sentiment.overall}
                              </p>
                            </div>
                            <div>
                              <p className="text-xs text-muted-foreground uppercase mb-1">Buy Pressure</p>
                              <p className="text-green-400 font-semibold">{tokenAnalysis.sentiment.buyPressure}</p>
                            </div>
                            <div>
                              <p className="text-xs text-muted-foreground uppercase mb-1">Buys (24h)</p>
                              <p className="text-green-400 font-semibold">{Number(tokenAnalysis.sentiment.buys24h || 0).toLocaleString()}</p>
                            </div>
                            <div>
                              <p className="text-xs text-muted-foreground uppercase mb-1">Sells (24h)</p>
                              <p className="text-red-400 font-semibold">{Number(tokenAnalysis.sentiment.sells24h || 0).toLocaleString()}</p>
                            </div>
                          </div>
                        </div>
                      )}

                      {/* AI Analysis Button */}
                      <div className="mt-6">
                        <Button 
                          onClick={runAiAnalysis}
                          disabled={isAiAnalyzing}
                          className="w-full bg-gradient-to-r from-purple-600 to-cyan-500 text-white font-semibold h-12"
                          data-testid="button-ai-analyze"
                        >
                          {isAiAnalyzing ? (
                            <>
                              <Loader2 className="w-4 h-4 animate-spin mr-2" />
                              Analyzing with AI...
                            </>
                          ) : (
                            <>
                              <Zap className="w-4 h-4 mr-2" />
                              Get AI-Powered Analysis
                            </>
                          )}
                        </Button>
                      </div>

                      {aiAnalysisError && (
                        <div className="p-4 rounded-lg bg-red-500/10 border border-red-500/30 flex items-center gap-3 mt-4">
                          <XCircle className="w-5 h-5 text-red-400" />
                          <p className="text-sm text-red-300">{aiAnalysisError}</p>
                        </div>
                      )}

                      {/* AI Analysis Results */}
                      {aiAnalysis && (
                        <div className="mt-6 space-y-4">
                          <div className="flex items-center gap-2 mb-4">
                            <div className="w-8 h-8 rounded-lg bg-gradient-to-r from-purple-600 to-cyan-500 flex items-center justify-center">
                              <Zap className="w-4 h-4 text-white" />
                            </div>
                            <h4 className="font-display font-semibold text-lg">AI Analysis</h4>
                            <span className="px-2 py-0.5 rounded-full text-xs bg-purple-500/20 text-purple-300">DeepSeek</span>
                          </div>

                          {/* Score Cards */}
                          <div className="grid grid-cols-1 sm:grid-cols-3 gap-3">
                            <div className="p-4 rounded-lg bg-gradient-to-br from-green-500/10 to-green-600/5 border border-green-500/20">
                              <div className="flex items-center justify-between mb-2">
                                <span className="text-xs text-muted-foreground uppercase tracking-wider">Sentiment</span>
                                <span className={`text-lg font-bold ${aiAnalysis.sentiment?.score >= 70 ? 'text-green-400' : aiAnalysis.sentiment?.score >= 40 ? 'text-yellow-400' : 'text-red-400'}`}>
                                  {aiAnalysis.sentiment?.score || 0}/100
                                </span>
                              </div>
                              <p className="text-xs text-muted-foreground line-clamp-3">{aiAnalysis.sentiment?.analysis || "N/A"}</p>
                            </div>

                            <div className="p-4 rounded-lg bg-gradient-to-br from-blue-500/10 to-blue-600/5 border border-blue-500/20">
                              <div className="flex items-center justify-between mb-2">
                                <span className="text-xs text-muted-foreground uppercase tracking-wider">Technical</span>
                                <span className={`text-lg font-bold ${aiAnalysis.technical?.score >= 70 ? 'text-green-400' : aiAnalysis.technical?.score >= 40 ? 'text-yellow-400' : 'text-red-400'}`}>
                                  {aiAnalysis.technical?.score || 0}/100
                                </span>
                              </div>
                              <p className="text-xs text-muted-foreground line-clamp-3">{aiAnalysis.technical?.analysis || "N/A"}</p>
                            </div>

                            <div className="p-4 rounded-lg bg-gradient-to-br from-purple-500/10 to-purple-600/5 border border-purple-500/20">
                              <div className="flex items-center justify-between mb-2">
                                <span className="text-xs text-muted-foreground uppercase tracking-wider">On-Chain</span>
                                <span className={`text-lg font-bold ${aiAnalysis.onChain?.score >= 70 ? 'text-green-400' : aiAnalysis.onChain?.score >= 40 ? 'text-yellow-400' : 'text-red-400'}`}>
                                  {aiAnalysis.onChain?.score || 0}/100
                                </span>
                              </div>
                              <p className="text-xs text-muted-foreground line-clamp-3">{aiAnalysis.onChain?.analysis || "N/A"}</p>
                            </div>
                          </div>

                          {/* Overall Recommendation */}
                          <div className="p-5 rounded-lg bg-gradient-to-r from-primary/10 to-secondary/10 border border-primary/20">
                            <div className="flex items-center justify-between mb-3">
                              <span className="text-sm font-medium">Overall Score</span>
                              <div className="flex items-center gap-2">
                                <span className={`text-2xl font-bold ${aiAnalysis.overall?.score >= 70 ? 'text-green-400' : aiAnalysis.overall?.score >= 40 ? 'text-yellow-400' : 'text-red-400'}`}>
                                  {aiAnalysis.overall?.score || 0}/100
                                </span>
                              </div>
                            </div>
                            <div className="p-3 rounded-lg bg-background/50">
                              <p className="text-sm">{aiAnalysis.overall?.recommendation || "No recommendation available"}</p>
                            </div>
                          </div>
                        </div>
                      )}
                    </div>
                  )}
                </div>
              )}

              {activeItem === "token-generator" && (
                <div className="space-y-6">
                  {/* Header Info */}
                  <div className="flex items-center gap-4 p-4 rounded-lg bg-gradient-to-r from-primary/10 to-secondary/10 border border-primary/20">
                    <div className="w-12 h-12 rounded-xl cyber-gradient flex items-center justify-center glow-cyan">
                      <Coins className="w-6 h-6 text-background" />
                    </div>
                    <div className="flex-1">
                      <h3 className="font-display font-semibold">Launch Your Token on BSC</h3>
                    </div>
                    <div className="text-right">
                      <p className="text-xs text-muted-foreground">Deploy Cost</p>
                      <p className="font-display font-bold text-primary">0.01 BNB</p>
                    </div>
                  </div>

                  {/* Wallet Connection */}
                  {!walletConnected && (
                    <Button onClick={connectWallet} className="w-full cyber-gradient text-background font-semibold h-12 glow-cyan">
                      <Wallet className="w-5 h-5 mr-2" />
                      Connect Wallet
                    </Button>
                  )}

                  <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
                    {/* Left Column - Token Details */}
                    <div className="space-y-4">
                      {/* Image Upload */}
                      <div>
                        <label className="text-sm font-medium text-muted-foreground mb-2 block">Token Image</label>
                        <div className="flex items-center gap-4">
                          <label className="w-24 h-24 rounded-xl border-2 border-dashed border-border/50 flex items-center justify-center cursor-pointer hover:border-primary/50 transition-colors overflow-hidden">
                            {tokenImagePreview ? (
                              <img src={tokenImagePreview} alt="Token" className="w-full h-full object-cover" />
                            ) : (
                              <div className="text-center">
                                <Upload className="w-6 h-6 text-muted-foreground mx-auto mb-1" />
                                <span className="text-xs text-muted-foreground">Upload</span>
                              </div>
                            )}
                            <input type="file" className="hidden" accept="image/*" onChange={handleImageUpload} />
                          </label>
                          <div className="text-xs text-muted-foreground">
                            <p>PNG, JPEG, WEBP, GIF</p>
                            <p>Max Size: 5MB</p>
                          </div>
                        </div>
                      </div>

                      {/* Token Name */}
                      <div>
                        <label className="text-sm font-medium text-muted-foreground mb-2 block">Token Name *</label>
                        <Input
                          placeholder="e.g. My Awesome Token"
                          className="bg-muted/30 border-border/50 h-12"
                          value={tokenName}
                          onChange={(e) => setTokenName(e.target.value)}
                        />
                      </div>

                      {/* Ticker Symbol */}
                      <div>
                        <label className="text-sm font-medium text-muted-foreground mb-2 block">Ticker Symbol *</label>
                        <Input
                          placeholder="e.g. MAT"
                          className="bg-muted/30 border-border/50 h-12"
                          value={tokenSymbol}
                          onChange={(e) => setTokenSymbol(e.target.value.toUpperCase())}
                          maxLength={10}
                        />
                      </div>

                      {/* Description */}
                      <div>
                        <label className="text-sm font-medium text-muted-foreground mb-2 block">Description *</label>
                        <Textarea
                          placeholder="Describe your token..."
                          className="bg-muted/30 border-border/50 min-h-[100px] resize-none"
                          value={tokenDescription}
                          onChange={(e) => setTokenDescription(e.target.value)}
                        />
                      </div>

                      {/* Label/Category */}
                      <div>
                        <label className="text-sm font-medium text-muted-foreground mb-2 block">Category</label>
                        <select
                          value={tokenLabel}
                          onChange={(e) => setTokenLabel(e.target.value)}
                          className="w-full bg-muted/30 border border-border/50 rounded-lg px-4 h-12 text-sm font-medium focus:outline-none focus:ring-2 focus:ring-primary/50"
                        >
                          {TOKEN_LABELS.map((label) => (
                            <option key={label} value={label} className="bg-card text-foreground">{label}</option>
                          ))}
                        </select>
                      </div>
                    </div>

                    {/* Right Column - Social Links & Settings */}
                    <div className="space-y-4">
                      {/* Social Links */}
                      <div>
                        <label className="text-sm font-medium text-muted-foreground mb-2 block">Website (Optional)</label>
                        <div className="relative">
                          <Globe className="absolute left-3 top-1/2 -translate-y-1/2 w-4 h-4 text-muted-foreground" />
                          <Input
                            placeholder="https://yourwebsite.com"
                            className="bg-muted/30 border-border/50 h-12 pl-10"
                            value={tokenWebsite}
                            onChange={(e) => setTokenWebsite(e.target.value)}
                          />
                        </div>
                      </div>

                      <div>
                        <label className="text-sm font-medium text-muted-foreground mb-2 block">Twitter (Optional)</label>
                        <div className="relative">
                          <Twitter className="absolute left-3 top-1/2 -translate-y-1/2 w-4 h-4 text-muted-foreground" />
                          <Input
                            placeholder="https://twitter.com/yourtoken"
                            className="bg-muted/30 border-border/50 h-12 pl-10"
                            value={tokenTwitter}
                            onChange={(e) => setTokenTwitter(e.target.value)}
                          />
                        </div>
                      </div>

                      <div>
                        <label className="text-sm font-medium text-muted-foreground mb-2 block">Telegram (Optional)</label>
                        <div className="relative">
                          <MessageCircle className="absolute left-3 top-1/2 -translate-y-1/2 w-4 h-4 text-muted-foreground" />
                          <Input
                            placeholder="https://t.me/yourgroup"
                            className="bg-muted/30 border-border/50 h-12 pl-10"
                            value={tokenTelegram}
                            onChange={(e) => setTokenTelegram(e.target.value)}
                          />
                        </div>
                      </div>

                      {/* Pre-sale Amount */}
                      <div>
                        <label className="text-sm font-medium text-muted-foreground mb-2 block">Pre-buy Amount (Optional)</label>
                        <div className="relative">
                          <Input
                            type="number"
                            placeholder="0"
                            className="bg-muted/30 border-border/50 h-12 pr-16"
                            value={preSaleAmount}
                            onChange={(e) => setPreSaleAmount(e.target.value)}
                            step="0.1"
                            min="0"
                          />
                          <span className="absolute right-3 top-1/2 -translate-y-1/2 text-sm text-muted-foreground">BNB</span>
                        </div>
                        <p className="text-xs text-muted-foreground mt-1">Amount of BNB to pre-purchase your token</p>
                      </div>


                    </div>
                  </div>

                  {/* Error Message */}
                  {tokenCreationError && (
                    <div className="p-4 rounded-lg bg-red-500/10 border border-red-500/30 flex items-center gap-3">
                      <XCircle className="w-5 h-5 text-red-400 flex-shrink-0" />
                      <p className="text-sm text-red-300">{tokenCreationError}</p>
                    </div>
                  )}

                  {/* Success Message */}
                  {tokenCreationSuccess && (
                    <div className="p-4 rounded-lg bg-green-500/10 border border-green-500/30 space-y-3">
                      <div className="flex items-center gap-3">
                        <CheckCircle className="w-5 h-5 text-green-400" />
                        <p className="text-sm text-green-300">{tokenCreationSuccess.message}</p>
                      </div>
                      <div className="text-xs text-muted-foreground space-y-1">
                        {tokenCreationSuccess.contractAddress && (
                          <p>Contract: <span className="font-mono">{tokenCreationSuccess.contractAddress}</span></p>
                        )}
                        {tokenCreationSuccess.txHash && (
                          <p>Tx Hash: <span className="font-mono">{tokenCreationSuccess.txHash.slice(0, 20)}...</span></p>
                        )}
                        {tokenCreationSuccess.explorerUrl && (
                          <a 
                            href={tokenCreationSuccess.explorerUrl} 
                            target="_blank" 
                            rel="noopener noreferrer"
                            className="inline-flex items-center gap-1 text-primary hover:underline mt-2"
                            data-testid="link-bscscan"
                          >
                            <ExternalLink className="w-3 h-3" />
                            View on BSCScan
                          </a>
                        )}
                      </div>
                    </div>
                  )}

                  {/* Create Button */}
                  <Button
                    onClick={createToken}
                    disabled={isCreatingToken || !walletConnected}
                    className="w-full cyber-gradient text-background font-bold h-14 text-lg glow-cyan"
                  >
                    {isCreatingToken ? (
                      <>
                        <Loader2 className="w-5 h-5 mr-2 animate-spin" />
                        Creating Token...
                      </>
                    ) : (
                      <>
                        <Coins className="w-5 h-5 mr-2" />
                        Create Token (0.01 BNB)
                      </>
                    )}
                  </Button>
                </div>
              )}

              {/* Password Score Checker */}
              {activeItem === "password-score" && (
                <div className="space-y-6">
                  <div className="flex items-center gap-4 p-4 rounded-lg bg-gradient-to-r from-primary/10 to-secondary/10 border border-primary/20">
                    <div className="w-12 h-12 rounded-xl cyber-gradient flex items-center justify-center glow-cyan">
                      <Lock className="w-6 h-6 text-background" />
                    </div>
                    <div className="flex-1">
                      <h3 className="font-display font-semibold">Password Strength Checker</h3>
                      <p className="text-sm text-muted-foreground">Test how secure your password is</p>
                    </div>
                  </div>

                  <div className="space-y-4">
                    <div>
                      <label className="text-sm font-medium text-muted-foreground mb-2 block">Enter Password</label>
                      <Input
                        type="password"
                        placeholder="Type your password to check strength..."
                        className="bg-muted/30 border-border/50 h-12"
                        value={passwordInput}
                        onChange={(e) => {
                          setPasswordInput(e.target.value);
                          calculatePasswordScore(e.target.value);
                        }}
                        data-testid="input-password-check"
                      />
                    </div>

                    {/* Score Display */}
                    <div className="p-6 rounded-lg bg-muted/20 border border-border/50 space-y-4">
                      <div className="flex items-center justify-between">
                        <span className="text-sm text-muted-foreground">Password Strength</span>
                        <span className={`text-2xl font-bold font-display ${getScoreColor()}`}>
                          {passwordScore}%
                        </span>
                      </div>

                      {/* Progress Bar */}
                      <div className="h-3 bg-muted rounded-full overflow-hidden">
                        <div 
                          className={`h-full transition-all duration-300 ${getProgressColor()}`}
                          style={{ width: `${passwordScore}%` }}
                        />
                      </div>

                      <div className="flex items-center justify-between">
                        <span className={`text-lg font-semibold ${getScoreColor()}`}>
                          {getScoreLabel()}
                        </span>
                        {passwordScore >= 80 && (
                          <CheckCircle className="w-5 h-5 text-green-400" />
                        )}
                      </div>
                    </div>

                    {/* Feedback */}
                    {passwordFeedback.length > 0 && (
                      <div className="p-4 rounded-lg bg-muted/10 border border-border/30 space-y-2">
                        <p className="text-sm font-medium text-muted-foreground">Tips to improve:</p>
                        <ul className="space-y-1">
                          {passwordFeedback.map((tip, index) => (
                            <li key={index} className="flex items-center gap-2 text-sm text-muted-foreground">
                              <XCircle className="w-4 h-4 text-orange-400 flex-shrink-0" />
                              {tip}
                            </li>
                          ))}
                        </ul>
                      </div>
                    )}

                    {passwordScore >= 80 && (
                      <div className="p-4 rounded-lg bg-green-500/10 border border-green-500/30 flex items-center gap-3">
                        <CheckCircle className="w-5 h-5 text-green-400" />
                        <p className="text-sm text-green-300">Great job! This is a strong password.</p>
                      </div>
                    )}
                  </div>
                </div>
              )}

              {/* Bundle Checker */}
              {activeItem === "bundle-checker" && (
                <div className="space-y-6">
                  <div className="flex items-center gap-4 p-4 rounded-lg bg-gradient-to-r from-primary/10 to-secondary/10 border border-primary/20">
                    <div className="w-12 h-12 rounded-xl cyber-gradient flex items-center justify-center glow-cyan">
                      <Layers className="w-6 h-6 text-background" />
                    </div>
                    <div className="flex-1">
                      <h3 className="font-display font-semibold">Bundle Checker</h3>
                      <p className="text-sm text-muted-foreground">Detect bundle signals in Solana and BSC token launches</p>
                    </div>
                  </div>

                  <div className="space-y-4">
                    {/* Chain Selection */}
                    <div className="flex gap-2">
                      <Button
                        variant={bundleChain === "solana" ? "default" : "outline"}
                        onClick={() => { setBundleChain("solana"); setBundleResult(null); setBundleError(null); }}
                        className={bundleChain === "solana" ? "cyber-gradient text-background" : ""}
                        data-testid="button-chain-solana"
                      >
                        Solana
                      </Button>
                      <Button
                        variant={bundleChain === "bsc" ? "default" : "outline"}
                        onClick={() => { setBundleChain("bsc"); setBundleResult(null); setBundleError(null); }}
                        className={bundleChain === "bsc" ? "cyber-gradient text-background" : ""}
                        data-testid="button-chain-bsc"
                      >
                        BSC
                      </Button>
                    </div>

                    <div>
                      <label className="text-sm font-medium text-muted-foreground mb-2 block">
                        {bundleChain === "solana" ? "Solana Mint Address" : "BSC Token Contract Address"}
                      </label>
                      <div className="flex gap-2">
                        <Input
                          placeholder={bundleChain === "solana" ? "Enter Solana token mint address..." : "Enter BSC token contract address..."}
                          className="bg-muted/30 border-border/50 h-12 flex-1"
                          value={bundleMintAddress}
                          onChange={(e) => setBundleMintAddress(e.target.value)}
                          data-testid="input-bundle-mint"
                        />
                        <Button
                          onClick={checkBundle}
                          disabled={isCheckingBundle}
                          className="cyber-gradient text-background font-bold h-12 px-6"
                          data-testid="button-check-bundle"
                        >
                          {isCheckingBundle ? (
                            <Loader2 className="w-5 h-5 animate-spin" />
                          ) : (
                            <Search className="w-5 h-5" />
                          )}
                        </Button>
                      </div>
                    </div>

                    {/* Error */}
                    {bundleError && (
                      <div className="p-4 rounded-lg bg-red-500/10 border border-red-500/30 flex items-center gap-3">
                        <XCircle className="w-5 h-5 text-red-400" />
                        <p className="text-sm text-red-300">{bundleError}</p>
                      </div>
                    )}

                    {/* Results */}
                    {bundleResult && (
                      <div className="space-y-4">
                        {/* Token Info */}
                        {bundleResult.tokenInfo && (
                          <div className="p-4 rounded-lg bg-muted/20 border border-border/50">
                            <div className="flex items-center justify-between">
                              <div>
                                <p className="font-semibold">{bundleResult.tokenInfo.name}</p>
                                <p className="text-sm text-muted-foreground">{bundleResult.tokenInfo.symbol}</p>
                              </div>
                              <div className="text-right text-xs text-muted-foreground">
                                <p>{bundleResult.transactionCount} transactions analyzed</p>
                              </div>
                            </div>
                          </div>
                        )}

                        {/* Bundle Analysis */}
                        <div className={`p-6 rounded-lg border ${
                          bundleResult.bundleAnalysis.isBundled 
                            ? "bg-red-500/10 border-red-500/30" 
                            : "bg-green-500/10 border-green-500/30"
                        }`}>
                          <div className="flex items-center gap-3 mb-4">
                            {bundleResult.bundleAnalysis.isBundled ? (
                              <AlertCircle className="w-6 h-6 text-red-400" />
                            ) : (
                              <CheckCircle className="w-6 h-6 text-green-400" />
                            )}
                            <div>
                              <p className={`font-semibold ${
                                bundleResult.bundleAnalysis.isBundled ? "text-red-300" : "text-green-300"
                              }`}>
                                {bundleResult.bundleAnalysis.isBundled ? "Bundle Detected" : "No Bundle Detected"}
                              </p>
                              <p className="text-sm text-muted-foreground">
                                Confidence: {bundleResult.bundleAnalysis.confidence}% | Risk: {bundleResult.bundleAnalysis.riskLevel}
                              </p>
                            </div>
                          </div>

                          <p className="text-sm text-muted-foreground mb-4">{bundleResult.bundleAnalysis.details}</p>

                          {/* Signals */}
                          {bundleResult.bundleAnalysis.signals && bundleResult.bundleAnalysis.signals.length > 0 && (
                            <div className="space-y-2">
                              <p className="text-xs font-medium text-muted-foreground">Detected Signals:</p>
                              <ul className="space-y-1">
                                {bundleResult.bundleAnalysis.signals.map((signal: string, idx: number) => (
                                  <li key={idx} className="flex items-center gap-2 text-sm">
                                    <Zap className="w-3 h-3 text-yellow-400" />
                                    {signal}
                                  </li>
                                ))}
                              </ul>
                            </div>
                          )}

                          {/* Stats */}
                          <div className="grid grid-cols-2 gap-4 mt-4 pt-4 border-t border-border/30">
                            <div>
                              <p className="text-xs text-muted-foreground">Unique Wallets</p>
                              <p className="font-semibold">{bundleResult.bundleAnalysis.uniqueWallets || 0}</p>
                            </div>
                            <div>
                              <p className="text-xs text-muted-foreground">Bundle Slots</p>
                              <p className="font-semibold">{bundleResult.bundleAnalysis.bundleCount || 0}</p>
                            </div>
                          </div>
                        </div>
                      </div>
                    )}
                  </div>
                </div>
              )}

              {/* Website Builder */}
              {activeItem === "website-builder" && (
                <div className="space-y-6">
                  <div className="flex items-center gap-4 p-4 rounded-lg bg-gradient-to-r from-primary/10 to-secondary/10 border border-primary/20">
                    <div className="w-12 h-12 rounded-xl cyber-gradient flex items-center justify-center glow-cyan">
                      <Globe className="w-6 h-6 text-background" />
                    </div>
                    <div className="flex-1">
                      <h3 className="font-display font-semibold">Create Landing Page</h3>
                      <p className="text-sm text-muted-foreground">Build a simple website for your crypto project</p>
                    </div>
                  </div>

                  <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
                    {/* Left Column - Form */}
                    <div className="space-y-4">
                      {/* Logo Upload */}
                      <div>
                        <label className="text-sm font-medium text-muted-foreground mb-2 block">Project Logo</label>
                        <div className="flex items-center gap-4">
                          <label className="w-20 h-20 rounded-xl border-2 border-dashed border-border/50 flex items-center justify-center cursor-pointer hover:border-primary/50 transition-colors overflow-hidden">
                            {websiteLogo ? (
                              <img src={websiteLogo} alt="Logo" className="w-full h-full object-cover" />
                            ) : (
                              <div className="text-center">
                                <Upload className="w-5 h-5 text-muted-foreground mx-auto mb-1" />
                                <span className="text-xs text-muted-foreground">Upload</span>
                              </div>
                            )}
                            <input 
                              type="file" 
                              className="hidden" 
                              accept="image/*" 
                              onChange={handleWebsiteLogoUpload}
                              data-testid="input-website-logo"
                            />
                          </label>
                          <div className="text-xs text-muted-foreground">
                            <p>PNG, JPEG, SVG</p>
                            <p>Recommended: 200x200px</p>
                            {websiteLogo && (
                              <button 
                                onClick={() => setWebsiteLogo(null)} 
                                className="text-red-400 hover:text-red-300 mt-1"
                              >
                                Remove
                              </button>
                            )}
                          </div>
                        </div>
                      </div>

                      <div>
                        <label className="text-sm font-medium text-muted-foreground mb-2 block">Project Name *</label>
                        <Input
                          placeholder="e.g. MoonCoin"
                          className="bg-muted/30 border-border/50 h-12"
                          value={websiteProjectName}
                          onChange={(e) => setWebsiteProjectName(e.target.value)}
                          data-testid="input-website-name"
                        />
                      </div>

                      <div>
                        <label className="text-sm font-medium text-muted-foreground mb-2 block">Tagline</label>
                        <Input
                          placeholder="e.g. To the moon and beyond"
                          className="bg-muted/30 border-border/50 h-12"
                          value={websiteTagline}
                          onChange={(e) => setWebsiteTagline(e.target.value)}
                          data-testid="input-website-tagline"
                        />
                      </div>

                      <div>
                        <label className="text-sm font-medium text-muted-foreground mb-2 block">Description</label>
                        <Textarea
                          placeholder="Describe your project..."
                          className="bg-muted/30 border-border/50 min-h-[100px]"
                          value={websiteDescription}
                          onChange={(e) => setWebsiteDescription(e.target.value)}
                          data-testid="input-website-description"
                        />
                      </div>

                      <div>
                        <label className="text-sm font-medium text-muted-foreground mb-2 block">Contract Address</label>
                        <Input
                          placeholder="0x..."
                          className="bg-muted/30 border-border/50 h-12"
                          value={websiteContractAddress}
                          onChange={(e) => setWebsiteContractAddress(e.target.value)}
                          data-testid="input-website-contract"
                        />
                      </div>

                      <div className="grid grid-cols-2 gap-4">
                        <div>
                          <label className="text-sm font-medium text-muted-foreground mb-2 block">Twitter URL</label>
                          <Input
                            placeholder="https://twitter.com/..."
                            className="bg-muted/30 border-border/50 h-12"
                            value={websiteTwitter}
                            onChange={(e) => setWebsiteTwitter(e.target.value)}
                            data-testid="input-website-twitter"
                          />
                        </div>
                        <div>
                          <label className="text-sm font-medium text-muted-foreground mb-2 block">Telegram URL</label>
                          <Input
                            placeholder="https://t.me/..."
                            className="bg-muted/30 border-border/50 h-12"
                            value={websiteTelegram}
                            onChange={(e) => setWebsiteTelegram(e.target.value)}
                            data-testid="input-website-telegram"
                          />
                        </div>
                      </div>

                      <div>
                        <label className="text-sm font-medium text-muted-foreground mb-2 block">Theme</label>
                        <div className="flex gap-2">
                          {(["cyber", "dark", "light"] as const).map((t) => (
                            <Button
                              key={t}
                              variant={websiteTheme === t ? "default" : "outline"}
                              onClick={() => setWebsiteTheme(t)}
                              className={websiteTheme === t ? "cyber-gradient text-background" : ""}
                              data-testid={`button-theme-${t}`}
                            >
                              {t.charAt(0).toUpperCase() + t.slice(1)}
                            </Button>
                          ))}
                        </div>
                      </div>

                      <Button
                        onClick={generateWebsite}
                        disabled={!websiteProjectName.trim()}
                        className="w-full cyber-gradient text-background font-semibold h-12 glow-cyan"
                        data-testid="button-generate-website"
                      >
                        Generate Website
                      </Button>
                    </div>

                    {/* Right Column - Preview */}
                    <div className="space-y-4">
                      <div className="flex items-center justify-between">
                        <label className="text-sm font-medium text-muted-foreground">Preview</label>
                        {generatedWebsite && (
                          <Button
                            onClick={downloadWebsite}
                            variant="outline"
                            size="sm"
                            data-testid="button-download-website"
                          >
                            <ExternalLink className="w-4 h-4 mr-2" />
                            Download HTML
                          </Button>
                        )}
                      </div>
                      <div className="rounded-lg border border-border/50 overflow-hidden bg-muted/20 min-h-[400px]">
                        {generatedWebsite ? (
                          <iframe
                            srcDoc={generatedWebsite}
                            className="w-full h-[500px] border-0"
                            title="Website Preview"
                            data-testid="iframe-website-preview"
                          />
                        ) : (
                          <div className="flex flex-col items-center justify-center h-[400px] text-muted-foreground">
                            <Globe className="w-12 h-12 mb-4 opacity-30" />
                            <p className="text-sm">Fill in the form and click Generate to preview</p>
                          </div>
                        )}
                      </div>
                      <p className="text-sm text-muted-foreground text-center">
                        Get domain and hosting <a href="https://t.me/CryptoSafeDev" target="_blank" rel="noopener noreferrer" className="text-primary hover:underline">here</a>
                      </p>
                    </div>
                  </div>
                </div>
              )}

              {activeItem === "staking" && (
                <div className="flex flex-col items-center justify-center py-12 text-center space-y-4">
                  <div className="w-20 h-20 rounded-2xl bg-primary/10 flex items-center justify-center">
                    {activeMenu?.icon}
                  </div>
                  <h3 className="text-2xl font-display font-bold">{activeMenu?.label}</h3>
                  <p className="text-muted-foreground max-w-md">{activeMenu?.description}</p>
                  <Button className="cyber-gradient text-background font-bold px-8 h-12 mt-4">Coming Soon</Button>
                </div>
              )}

              {/* Block Explorer Section */}
              {activeItem === "block-explorer" && (
                <div className="space-y-6">
                  {/* Header */}
                  <div className="text-center space-y-2">
                    <h2 className="text-3xl font-display font-bold text-glow-cyan">BNB Smart Chain Explorer</h2>
                    <p className="text-muted-foreground">A blockchain explorer and analytics platform for BNB Smart Chain</p>
                  </div>

                  {/* Search Bar */}
                  <div className="max-w-2xl mx-auto">
                    <div className="flex gap-2">
                      <div className="relative flex-1">
                        <Search className="absolute left-4 top-1/2 -translate-y-1/2 w-5 h-5 text-muted-foreground" />
                        <Input
                          placeholder="Search by Block / Txn / Address / Token"
                          value={explorerAddress}
                          onChange={(e) => setExplorerAddress(e.target.value)}
                          onKeyDown={(e) => e.key === "Enter" && handleExploreAddress()}
                          className="pl-12 h-12 bg-card border-border text-sm"
                          data-testid="input-explorer-address"
                        />
                      </div>
                      <Button
                        onClick={handleExploreAddress}
                        disabled={isExploring || !explorerAddress.trim()}
                        className="h-12 px-6 cyber-gradient text-background font-semibold"
                        data-testid="button-explore-address"
                      >
                        {isExploring ? <Loader2 className="w-5 h-5 animate-spin" /> : <Search className="w-5 h-5" />}
                      </Button>
                    </div>
                  </div>

                  {/* Stats Cards */}
                  <div className="grid grid-cols-3 gap-4">
                    <div className="p-4 rounded-xl bg-card border border-border">
                      <p className="text-sm text-muted-foreground mb-1">Latest Block</p>
                      <p className="text-2xl font-bold text-primary font-mono">
                        {isLoadingExplorer ? "..." : explorerData?.latestBlock?.toLocaleString() || "---"}
                        <span className="text-xs text-muted-foreground ml-1">(0.4s)</span>
                      </p>
                    </div>
                    <div className="p-4 rounded-xl bg-card border border-border">
                      <p className="text-sm text-muted-foreground mb-1">Active Validators</p>
                      <p className="text-2xl font-bold">{explorerData?.activeValidators || 45}</p>
                    </div>
                    <div className="p-4 rounded-xl bg-card border border-border">
                      <p className="text-sm text-muted-foreground mb-1">BNB Price</p>
                      <p className="text-2xl font-bold text-green-400">
                        ${explorerData?.bnbPrice?.toFixed(2) || "---"}
                      </p>
                    </div>
                  </div>

                  {/* Loading State */}
                  {isLoadingExplorer && (
                    <div className="flex items-center justify-center py-12">
                      <Loader2 className="w-8 h-8 animate-spin text-primary" />
                    </div>
                  )}

                  {/* Two Column Layout: Blocks and Transactions */}
                  {explorerData && !explorerResult && (
                    <div className="grid md:grid-cols-2 gap-6">
                      {/* Latest 10 Blocks */}
                      <div className="p-4 rounded-xl bg-card border border-border">
                        <h3 className="font-display font-semibold text-lg mb-4 flex items-center gap-2">
                          <Layers className="w-5 h-5 text-primary" />
                          Latest 10 Blocks
                        </h3>
                        <div className="overflow-x-auto">
                          <table className="w-full text-sm">
                            <thead>
                              <tr className="border-b border-border text-muted-foreground text-xs">
                                <th className="text-left py-2">Block</th>
                                <th className="text-left py-2">Validated</th>
                                <th className="text-right py-2">Block Reward</th>
                              </tr>
                            </thead>
                            <tbody className="divide-y divide-border/30">
                              {explorerData.blocks?.map((block: any, i: number) => (
                                <tr key={i} className="hover:bg-muted/20">
                                  <td className="py-2">
                                    <div className="flex items-center gap-2">
                                      <div className="w-6 h-6 rounded bg-primary/20 flex items-center justify-center">
                                        <Layers className="w-3 h-3 text-primary" />
                                      </div>
                                      <div>
                                        <a href={`https://bscscan.com/block/${block.number}`} target="_blank" rel="noopener noreferrer" className="text-primary hover:underline font-mono text-xs">
                                          {block.number}
                                        </a>
                                        <p className="text-xs text-muted-foreground">{block.timeAgo}</p>
                                      </div>
                                    </div>
                                  </td>
                                  <td className="py-2">
                                    <p className="text-xs text-primary">Validator</p>
                                    <p className="text-xs text-muted-foreground">{block.txCount} transactions</p>
                                  </td>
                                  <td className="py-2 text-right font-mono text-xs">{block.reward} BNB</td>
                                </tr>
                              ))}
                            </tbody>
                          </table>
                        </div>
                      </div>

                      {/* Latest 10 Transactions */}
                      <div className="p-4 rounded-xl bg-card border border-border">
                        <h3 className="font-display font-semibold text-lg mb-4 flex items-center gap-2">
                          <Activity className="w-5 h-5 text-primary" />
                          Latest 10 Transactions
                        </h3>
                        <div className="overflow-x-auto">
                          <table className="w-full text-sm">
                            <thead>
                              <tr className="border-b border-border text-muted-foreground text-xs">
                                <th className="text-left py-2">Txn Hash</th>
                                <th className="text-left py-2">From / To</th>
                                <th className="text-right py-2">Value</th>
                              </tr>
                            </thead>
                            <tbody className="divide-y divide-border/30">
                              {explorerData.transactions?.map((tx: any, i: number) => (
                                <tr key={i} className="hover:bg-muted/20">
                                  <td className="py-2">
                                    <div className="flex items-center gap-2">
                                      <div className="w-6 h-6 rounded bg-cyan-500/20 flex items-center justify-center">
                                        <FileText className="w-3 h-3 text-cyan-400" />
                                      </div>
                                      <div>
                                        <a href={`https://bscscan.com/tx/${tx.hash}`} target="_blank" rel="noopener noreferrer" className="text-primary hover:underline font-mono text-xs">
                                          {tx.hash?.slice(0, 10)}...{tx.hash?.slice(-4)}
                                        </a>
                                        <p className="text-xs text-muted-foreground">{tx.timeAgo}</p>
                                      </div>
                                    </div>
                                  </td>
                                  <td className="py-2">
                                    <p className="text-xs">
                                      <span className="text-muted-foreground">From </span>
                                      <a href={`https://bscscan.com/address/${tx.from}`} target="_blank" rel="noopener noreferrer" className="text-primary hover:underline font-mono">
                                        {tx.from?.slice(0, 6)}...{tx.from?.slice(-4)}
                                      </a>
                                    </p>
                                    <p className="text-xs">
                                      <span className="text-muted-foreground">To </span>
                                      <a href={`https://bscscan.com/address/${tx.to}`} target="_blank" rel="noopener noreferrer" className="text-green-400 hover:underline font-mono">
                                        {tx.to?.slice(0, 6)}...{tx.to?.slice(-4)}
                                      </a>
                                    </p>
                                  </td>
                                  <td className="py-2 text-right font-mono text-xs">{tx.value} BNB</td>
                                </tr>
                              ))}
                            </tbody>
                          </table>
                        </div>
                      </div>
                    </div>
                  )}

                  {/* Error Display */}
                  {explorerError && (
                    <div className="p-4 rounded-lg bg-red-500/10 border border-red-500/30">
                      <div className="flex items-center gap-2 text-red-400">
                        <XCircle className="w-5 h-5" />
                        <p className="text-sm">{explorerError}</p>
                      </div>
                    </div>
                  )}

                  {/* Address Search Results - bsctrace.com style */}
                  {explorerResult && (
                    <div className="space-y-4">
                      {/* Back Button */}
                      <Button variant="ghost" size="sm" onClick={() => setExplorerResult(null)} className="text-muted-foreground">
                        <ArrowLeft className="w-4 h-4 mr-2" /> Back to Explorer
                      </Button>
                      
                      {/* Address Header */}
                      <div className="flex items-center gap-3 flex-wrap">
                        <h2 className="text-2xl font-display font-bold">Address</h2>
                        <span className={`px-2 py-0.5 rounded text-xs font-medium border ${explorerResult.isContract ? 'bg-purple-500/10 text-purple-400 border-purple-500/30' : 'bg-cyan-500/10 text-cyan-400 border-cyan-500/30'}`}>
                          {explorerResult.isContract ? 'Contract' : 'Wallet'}
                        </span>
                        <div className="flex items-center gap-2 flex-1 min-w-0">
                          <p className="font-mono text-sm text-muted-foreground truncate">{explorerResult.address}</p>
                          <button onClick={() => navigator.clipboard.writeText(explorerResult.address)} className="text-muted-foreground hover:text-primary">
                            <Copy className="w-4 h-4" />
                          </button>
                        </div>
                      </div>

                      {/* Info Boxes - bsctrace style */}
                      <div className="grid grid-cols-2 md:grid-cols-4 gap-4 p-4 rounded-xl bg-card border border-border">
                        <div className="border-r border-border pr-4">
                          <p className="text-xs text-muted-foreground mb-1">Balance</p>
                          <p className="font-bold text-primary">{explorerResult.bnbBalance} BNB</p>
                        </div>
                        <div className="border-r border-border pr-4">
                          <p className="text-xs text-muted-foreground mb-1">Token</p>
                          <p className="font-bold">{explorerResult.tokenHoldings?.length || 0} Token & 0 NFT</p>
                        </div>
                        <div className="border-r border-border pr-4">
                          <p className="text-xs text-muted-foreground mb-1">Contract Creator</p>
                          {explorerResult.contractCreator ? (
                            <a href={`https://bscscan.com/address/${explorerResult.contractCreator}`} target="_blank" rel="noopener noreferrer" className="text-primary hover:underline text-sm font-mono">
                              {explorerResult.contractCreator?.slice(0, 6)}...{explorerResult.contractCreator?.slice(-4)}
                            </a>
                          ) : <p className="text-muted-foreground text-sm">-</p>}
                        </div>
                        <div>
                          <p className="text-xs text-muted-foreground mb-1">Token Tracker</p>
                          <a href={explorerResult.bscscanUrl} target="_blank" rel="noopener noreferrer" className="text-primary hover:underline text-sm flex items-center gap-1">
                            View on BscScan <ExternalLink className="w-3 h-3" />
                          </a>
                        </div>
                      </div>

                      {/* Transactions Tab */}
                      <div className="rounded-xl bg-card border border-border overflow-hidden">
                        <div className="border-b border-border px-4 py-3 flex items-center gap-4 overflow-x-auto">
                          <button className="text-sm font-medium text-primary border-b-2 border-primary pb-2 whitespace-nowrap">Transactions</button>
                          <button className="text-sm text-muted-foreground hover:text-foreground pb-2 whitespace-nowrap">Internal Txns</button>
                          <button className="text-sm text-muted-foreground hover:text-foreground pb-2 whitespace-nowrap">BEP-20 Token Txns</button>
                          <button className="text-sm text-muted-foreground hover:text-foreground pb-2 whitespace-nowrap">BEP-721 Token Txns</button>
                          <button className="text-sm text-muted-foreground hover:text-foreground pb-2 whitespace-nowrap">Contract</button>
                        </div>

                        {/* Transactions Table */}
                        <div className="overflow-x-auto">
                          <table className="w-full text-sm">
                            <thead>
                              <tr className="border-b border-border text-muted-foreground text-xs bg-muted/20">
                                <th className="text-left py-3 px-4">Txn Hash</th>
                                <th className="text-left py-3 px-2">Method</th>
                                <th className="text-left py-3 px-2">Block</th>
                                <th className="text-left py-3 px-2">Age</th>
                                <th className="text-left py-3 px-2">From</th>
                                <th className="text-left py-3 px-2"></th>
                                <th className="text-left py-3 px-2">To</th>
                                <th className="text-right py-3 px-2">Value</th>
                                <th className="text-right py-3 px-4">Txn Fee</th>
                              </tr>
                            </thead>
                            <tbody className="divide-y divide-border/30">
                              {explorerResult.transactions?.length > 0 ? (
                                explorerResult.transactions.map((tx: any, i: number) => (
                                  <tr key={i} className="hover:bg-muted/20">
                                    <td className="py-3 px-4">
                                      <a href={`https://bscscan.com/tx/${tx.hash}`} target="_blank" rel="noopener noreferrer" className="text-primary hover:underline font-mono text-xs">
                                        {tx.hash?.slice(0, 8)}...{tx.hash?.slice(-6)}
                                      </a>
                                    </td>
                                    <td className="py-3 px-2">
                                      <span className="px-2 py-1 rounded bg-muted text-xs font-mono">{tx.method || "Transfer"}</span>
                                    </td>
                                    <td className="py-3 px-2">
                                      <a href={`https://bscscan.com/block/${tx.blockNumber}`} target="_blank" rel="noopener noreferrer" className="text-primary hover:underline font-mono text-xs">
                                        {tx.blockNumber}
                                      </a>
                                    </td>
                                    <td className="py-3 px-2 text-xs text-muted-foreground whitespace-nowrap">{tx.timeAgo || tx.age || "Recent"}</td>
                                    <td className="py-3 px-2">
                                      <a href={`https://bscscan.com/address/${tx.from}`} target="_blank" rel="noopener noreferrer" className="text-primary hover:underline font-mono text-xs">
                                        {tx.from?.slice(0, 6)}...{tx.from?.slice(-4)}
                                      </a>
                                    </td>
                                    <td className="py-3 px-2">
                                      <span className={`px-1.5 py-0.5 rounded text-[10px] font-medium ${tx.type === 'in' ? 'bg-green-500/20 text-green-400' : 'bg-yellow-500/20 text-yellow-400'}`}>
                                        {tx.type === 'in' ? 'IN' : 'OUT'}
                                      </span>
                                    </td>
                                    <td className="py-3 px-2">
                                      <a href={`https://bscscan.com/address/${tx.to}`} target="_blank" rel="noopener noreferrer" className="text-primary hover:underline font-mono text-xs">
                                        {tx.to?.slice(0, 6)}...{tx.to?.slice(-4)}
                                      </a>
                                    </td>
                                    <td className="py-3 px-2 text-right font-mono text-xs whitespace-nowrap">
                                      {tx.value} BNB
                                    </td>
                                    <td className="py-3 px-4 text-right font-mono text-xs text-muted-foreground whitespace-nowrap">
                                      {tx.txnFee || "0.00000000"} BNB
                                    </td>
                                  </tr>
                                ))
                              ) : (
                                <tr>
                                  <td colSpan={9} className="py-8 text-center">
                                    <div className="space-y-2">
                                      <p className="text-muted-foreground">No recent transactions found</p>
                                      <a 
                                        href={`https://bscscan.com/txs?a=${explorerResult.address}`} 
                                        target="_blank" 
                                        rel="noopener noreferrer"
                                        className="text-primary hover:underline text-sm inline-flex items-center gap-1"
                                      >
                                        View full history on BscScan <ExternalLink className="w-3 h-3" />
                                      </a>
                                    </div>
                                  </td>
                                </tr>
                              )}
                            </tbody>
                          </table>
                        </div>
                      </div>

                      {/* Token Holdings - collapsible */}
                      {explorerResult.tokenHoldings?.length > 0 && (
                        <div className="p-4 rounded-xl bg-card border border-border">
                          <h3 className="font-display font-semibold text-sm mb-3 flex items-center gap-2">
                            <Coins className="w-4 h-4 text-primary" /> Token Holdings ({explorerResult.tokenHoldings.length})
                          </h3>
                          <div className="grid grid-cols-2 md:grid-cols-4 gap-2">
                            {explorerResult.tokenHoldings.slice(0, 8).map((token: any, i: number) => (
                              <a key={i} href={`https://bscscan.com/token/${token.contract}`} target="_blank" rel="noopener noreferrer" className="p-2 rounded bg-muted/30 hover:bg-muted/50 text-xs">
                                <p className="font-medium truncate">{token.symbol}</p>
                                <p className="text-muted-foreground font-mono truncate">{token.balance}</p>
                              </a>
                            ))}
                          </div>
                        </div>
                      )}
                    </div>
                  )}
                </div>
              )}

              {/* Swap Section */}
              {activeItem === "token-swap" && (
                <div className="space-y-6 max-w-lg mx-auto">
                  {/* Header */}
                  <div className="flex items-center justify-between">
                    <div className="flex items-center gap-3">
                      <div className="w-12 h-12 rounded-xl cyber-gradient flex items-center justify-center glow-cyan">
                        <ArrowDownUp className="w-6 h-6 text-background" />
                      </div>
                      <div>
                        <h3 className="font-display font-semibold text-lg">Swap Tokens</h3>
                        <p className="text-xs text-muted-foreground">Trade tokens on BSC</p>
                      </div>
                    </div>
                    <div className="flex items-center gap-2">
                      <Button
                        variant="ghost"
                        size="icon"
                        onClick={() => getSwapQuote()}
                        disabled={isLoadingQuote}
                        data-testid="button-refresh-quote"
                      >
                        <RefreshCw className={`w-4 h-4 ${isLoadingQuote ? 'animate-spin' : ''}`} />
                      </Button>
                    </div>
                  </div>

                  {/* Wallet Status */}
                  {walletConnected ? (
                    <div className="p-3 rounded-lg bg-green-500/10 border border-green-500/30 flex items-center gap-3">
                      <CheckCircle className="w-5 h-5 text-green-400" />
                      <div className="flex-1">
                        <p className="text-sm text-green-400 font-medium">Wallet Connected</p>
                        <p className="text-xs text-muted-foreground font-mono">{walletAddress.slice(0, 6)}...{walletAddress.slice(-4)}</p>
                      </div>
                      <p className="text-sm font-medium">{swapFromBalance} BNB</p>
                    </div>
                  ) : (
                    <div className="p-3 rounded-lg bg-muted/30 border border-border/50 flex items-center gap-3">
                      <Wallet className="w-5 h-5 text-muted-foreground" />
                      <p className="text-sm text-muted-foreground flex-1">Connect wallet to swap</p>
                      <Button 
                        onClick={connectSwapWallet}
                        size="sm"
                        className="cyber-gradient text-background font-semibold"
                        data-testid="button-connect-swap-wallet"
                      >
                        Connect
                      </Button>
                    </div>
                  )}

                  {/* From Token */}
                  <div className="p-4 rounded-xl bg-card border border-border/50">
                    <div className="flex items-center justify-between mb-2">
                      <span className="text-sm text-muted-foreground">From</span>
                      <span className="text-xs text-muted-foreground">Balance: {walletConnected ? swapFromBalance : "0.0000"}</span>
                    </div>
                    <div className="flex items-center gap-3">
                      <Input
                        type="number"
                        placeholder="0.0"
                        className="flex-1 bg-transparent border-0 text-2xl font-bold h-12 p-0 focus-visible:ring-0"
                        value={swapFromAmount}
                        onChange={(e) => {
                          setSwapFromAmount(e.target.value);
                          if (e.target.value && swapToToken && walletConnected) {
                            setTimeout(getSwapQuote, 500);
                          }
                        }}
                        data-testid="input-swap-from-amount"
                      />
                      <select
                        value={swapFromToken}
                        onChange={(e) => setSwapFromToken(e.target.value)}
                        className="bg-muted/50 border border-border/50 rounded-lg px-3 py-2 text-sm font-medium"
                        data-testid="select-swap-from-token"
                      >
                        {SWAP_TOKENS.map((token) => (
                          <option key={token.symbol} value={token.symbol} disabled={token.symbol === swapToToken}>
                            {token.symbol}
                          </option>
                        ))}
                      </select>
                    </div>
                    <Button
                      variant="ghost"
                      size="sm"
                      onClick={() => setSwapFromAmount(swapFromBalance)}
                      className="text-xs text-primary mt-2"
                      disabled={!walletConnected}
                      data-testid="button-max-amount"
                    >
                      MAX
                    </Button>
                  </div>

                  {/* Switch Button */}
                  <div className="flex justify-center -my-2 relative z-10">
                    <Button
                      variant="outline"
                      size="icon"
                      onClick={switchSwapTokens}
                      className="rounded-full bg-background border-2 border-primary/30 hover:border-primary"
                      data-testid="button-switch-tokens"
                    >
                      <ArrowDownUp className="w-4 h-4" />
                    </Button>
                  </div>

                  {/* To Token */}
                  <div className="p-4 rounded-xl bg-card border border-border/50">
                    <div className="flex items-center justify-between mb-2">
                      <span className="text-sm text-muted-foreground">To</span>
                      <span className="text-xs text-muted-foreground">Balance: {swapToBalance}</span>
                    </div>
                    <div className="flex items-center gap-3">
                      <Input
                        type="number"
                        placeholder="0.0"
                        className="flex-1 bg-transparent border-0 text-2xl font-bold h-12 p-0 focus-visible:ring-0"
                        value={swapToAmount}
                        readOnly
                        data-testid="input-swap-to-amount"
                      />
                      <select
                        value={swapToToken}
                        onChange={(e) => {
                          setSwapToToken(e.target.value);
                          if (swapFromAmount && e.target.value && walletConnected) {
                            setTimeout(getSwapQuote, 500);
                          }
                        }}
                        className="bg-muted/50 border border-border/50 rounded-lg px-3 py-2 text-sm font-medium"
                        data-testid="select-swap-to-token"
                      >
                        <option value="">Select token</option>
                        {SWAP_TOKENS.map((token) => (
                          <option key={token.symbol} value={token.symbol} disabled={token.symbol === swapFromToken}>
                            {token.symbol}
                          </option>
                        ))}
                      </select>
                    </div>
                  </div>

                  {/* Slippage Settings */}
                  <div className="p-4 rounded-xl bg-muted/20 border border-border/30">
                    <div className="flex items-center justify-between flex-wrap gap-2">
                      <div className="flex items-center gap-2">
                        <Settings className="w-4 h-4 text-muted-foreground" />
                        <span className="text-sm text-muted-foreground">Slippage Tolerance</span>
                      </div>
                      <div className="flex items-center gap-2">
                        {["0.5", "1", "3"].map((s) => (
                          <Button
                            key={s}
                            variant={swapSlippage === s ? "default" : "ghost"}
                            size="sm"
                            onClick={() => setSwapSlippage(s)}
                            className={swapSlippage === s ? "cyber-gradient text-background" : ""}
                            data-testid={`button-slippage-${s}`}
                          >
                            {s}%
                          </Button>
                        ))}
                        <Input
                          type="number"
                          placeholder="Custom"
                          className="w-16 h-8 text-sm text-center"
                          value={!["0.5", "1", "3"].includes(swapSlippage) ? swapSlippage : ""}
                          onChange={(e) => setSwapSlippage(e.target.value)}
                          data-testid="input-custom-slippage"
                        />
                      </div>
                    </div>
                  </div>

                  {/* Quote Info */}
                  {swapQuote && (
                    <div className="p-4 rounded-xl bg-primary/5 border border-primary/20 space-y-2">
                      <div className="flex justify-between text-sm">
                        <span className="text-muted-foreground">Rate</span>
                        <span>1 {swapFromToken} = {(parseFloat(swapToAmount) / parseFloat(swapFromAmount)).toFixed(6)} {swapToToken}</span>
                      </div>
                      <div className="flex justify-between text-sm">
                        <span className="text-muted-foreground">Min. Received</span>
                        <span>{(parseFloat(swapToAmount) * (1 - parseFloat(swapSlippage) / 100)).toFixed(6)} {swapToToken}</span>
                      </div>
                      <div className="flex justify-between text-sm">
                        <span className="text-muted-foreground">Price Impact</span>
                        <span className="text-green-400">{swapQuote.priceImpact}</span>
                      </div>
                    </div>
                  )}

                  {/* Error/Success Messages */}
                  {swapError && (
                    <div className="p-4 rounded-lg bg-red-500/10 border border-red-500/30 flex items-center gap-3">
                      <XCircle className="w-5 h-5 text-red-400" />
                      <p className="text-sm text-red-300">{swapError}</p>
                    </div>
                  )}

                  {swapSuccess && (
                    <div className="p-4 rounded-lg bg-green-500/10 border border-green-500/30">
                      <div className="flex items-center gap-3 mb-2">
                        <CheckCircle className="w-5 h-5 text-green-400" />
                        <p className="text-sm text-green-400 font-medium">{swapSuccess.message}</p>
                      </div>
                      <a 
                        href={swapSuccess.explorerUrl}
                        target="_blank"
                        rel="noopener noreferrer"
                        className="text-xs text-primary hover:underline flex items-center gap-1"
                      >
                        View on BscScan <ExternalLink className="w-3 h-3" />
                      </a>
                    </div>
                  )}

                  {/* Swap Button */}
                  <Button
                    onClick={() => {
                      if (!walletConnected) {
                        connectSwapWallet();
                      } else {
                        executeSwap();
                      }
                    }}
                    disabled={isSwapping || (walletConnected && (!swapFromAmount || !swapToToken || parseFloat(swapFromAmount) <= 0))}
                    className="w-full cyber-gradient text-background font-semibold h-14 text-lg glow-cyan"
                    data-testid="button-execute-swap"
                  >
                    {isSwapping ? (
                      <>
                        <Loader2 className="w-5 h-5 mr-2 animate-spin" />
                        Swapping...
                      </>
                    ) : !walletConnected ? (
                      "Connect Wallet to Swap"
                    ) : !swapToToken ? (
                      "Select a token"
                    ) : !swapFromAmount || parseFloat(swapFromAmount) <= 0 ? (
                      "Enter an amount"
                    ) : (
                      `Swap ${swapFromToken} for ${swapToToken}`
                    )}
                  </Button>

                </div>
              )}

              {/* Futures Trading Section */}
              {activeItem === "futures" && (
                <div className="space-y-4">
                  {/* Trading Pair Header */}
                  <div className="flex items-center justify-between flex-wrap gap-4 p-3 rounded-xl bg-card border border-border">
                    <div className="flex items-center gap-4">
                      <div className="flex items-center gap-2">
                        <div className={`w-8 h-8 rounded-full flex items-center justify-center text-xs font-bold text-black ${
                          futuresPair === "BTCUSDT" ? "bg-yellow-500" : 
                          futuresPair === "ETHUSDT" ? "bg-blue-500" : 
                          futuresPair === "BNBUSDT" ? "bg-yellow-400" : 
                          futuresPair === "SOLUSDT" ? "bg-purple-500" : "bg-gray-500"
                        }`}>
                          {futuresPair.replace("USDT", "").slice(0, 3)}
                        </div>
                        <div>
                          <select 
                            value={futuresPair}
                            onChange={(e) => setFuturesPair(e.target.value)}
                            className="bg-transparent text-lg font-bold border-none focus:outline-none cursor-pointer" 
                            data-testid="select-trading-pair"
                          >
                            <option value="BTCUSDT">BTC/USDT Perp</option>
                            <option value="ETHUSDT">ETH/USDT Perp</option>
                            <option value="BNBUSDT">BNB/USDT Perp</option>
                            <option value="SOLUSDT">SOL/USDT Perp</option>
                            <option value="XRPUSDT">XRP/USDT Perp</option>
                          </select>
                          <p className={`text-xs ${futuresPrice.priceChangePercent >= 0 ? 'text-green-400' : 'text-red-400'}`}>
                            {futuresPrice.priceChangePercent >= 0 ? '+' : ''}{futuresPrice.priceChangePercent.toFixed(2)}%
                          </p>
                        </div>
                      </div>
                      <div className="hidden md:flex items-center gap-6 text-sm">
                        <div>
                          <p className="text-muted-foreground text-xs">Mark</p>
                          <p className="font-mono text-primary">{futuresPrice.markPrice.toLocaleString(undefined, { minimumFractionDigits: 1, maximumFractionDigits: 1 })}</p>
                        </div>
                        <div>
                          <p className="text-muted-foreground text-xs">Index</p>
                          <p className="font-mono">{futuresPrice.indexPrice.toLocaleString(undefined, { minimumFractionDigits: 1, maximumFractionDigits: 1 })}</p>
                        </div>
                        <div>
                          <p className="text-muted-foreground text-xs">Funding</p>
                          <p className={`font-mono ${futuresPrice.fundingRate >= 0 ? 'text-green-400' : 'text-red-400'}`}>{futuresPrice.fundingRate.toFixed(4)}%</p>
                        </div>
                        <div>
                          <p className="text-muted-foreground text-xs">24h Volume</p>
                          <p className="font-mono">{futuresPrice.volume24h.toLocaleString(undefined, { maximumFractionDigits: 0 })}</p>
                        </div>
                        <div>
                          <p className="text-muted-foreground text-xs">Open Interest</p>
                          <p className="font-mono">{futuresPrice.openInterest.toLocaleString(undefined, { maximumFractionDigits: 0 })}</p>
                        </div>
                      </div>
                    </div>
                    <Button variant="outline" size="sm" className="gap-2">
                      <Shield className="w-4 h-4" /> Shield
                    </Button>
                  </div>

                  {/* Main Trading Interface */}
                  <div className="grid lg:grid-cols-4 gap-4">
                    {/* Chart Area */}
                    <div className="lg:col-span-2 rounded-xl bg-card border border-border p-4 min-h-[400px]">
                      <div className="flex items-center justify-between mb-4 flex-wrap gap-2">
                        <div className="flex items-center gap-1 text-xs">
                          {[
                            { label: "5m", value: "5m" },
                            { label: "15m", value: "15m" },
                            { label: "1H", value: "1h" },
                            { label: "4H", value: "4h" },
                            { label: "1D", value: "1d" },
                            { label: "1W", value: "1w" },
                          ].map((interval) => (
                            <button
                              key={interval.value}
                              onClick={() => setFuturesInterval(interval.value)}
                              className={`px-2 py-1 rounded transition-colors ${
                                futuresInterval === interval.value
                                  ? "bg-primary/20 text-primary"
                                  : "bg-muted hover:bg-muted/80"
                              }`}
                              data-testid={`btn-interval-${interval.value}`}
                            >
                              {interval.label}
                            </button>
                          ))}
                        </div>
                        <div className="flex items-center gap-3 text-xs">
                          <span className={`font-bold text-lg ${futuresPrice.priceChangePercent >= 0 ? 'text-green-400' : 'text-red-400'}`}>
                            ${futuresPrice.price.toLocaleString(undefined, { minimumFractionDigits: 2, maximumFractionDigits: 2 })}
                          </span>
                          <span className="text-muted-foreground">
                            Vol: {(futuresPrice.volume24h / 1000000000).toFixed(2)}B
                          </span>
                        </div>
                      </div>
                      
                      {/* TradingView Lightweight Chart */}
                      <div 
                        ref={chartContainerRef} 
                        className="w-full h-[300px] rounded-lg"
                        data-testid="futures-chart-container"
                      >
                        {futuresKlines.length === 0 && (
                          <div className="flex items-center justify-center h-full">
                            <Loader2 className="w-8 h-8 animate-spin text-primary" />
                          </div>
                        )}
                      </div>
                    </div>

                    {/* Order Book */}
                    <div className="rounded-xl bg-card border border-border p-4">
                      <div className="flex items-center justify-between mb-3">
                        <h4 className="font-semibold text-sm">Order book</h4>
                        <span className="text-xs text-muted-foreground">Trades</span>
                      </div>
                      <div className="text-xs space-y-1">
                        <div className="grid grid-cols-3 text-muted-foreground mb-2">
                          <span>Price (USDT)</span>
                          <span className="text-right">Size</span>
                          <span className="text-right">Total</span>
                        </div>
                        {/* Asks (Red) - Live from Binance */}
                        {futuresOrderBook.asks.length > 0 ? (
                          (() => {
                            const maxSize = Math.max(...futuresOrderBook.asks.map((o: any) => o.size));
                            let cumulative = 0;
                            return futuresOrderBook.asks.map((order: any, i: number) => {
                              cumulative += order.size * order.price;
                              return (
                                <div key={i} className="grid grid-cols-3 text-red-400 relative">
                                  <div className="absolute right-0 top-0 bottom-0 bg-red-500/10" style={{ width: `${(order.size / maxSize) * 100}%` }}></div>
                                  <span className="font-mono relative z-10">{order.price.toLocaleString(undefined, { minimumFractionDigits: 1, maximumFractionDigits: 1 })}</span>
                                  <span className="text-right font-mono relative z-10">{order.size.toFixed(3)}</span>
                                  <span className="text-right font-mono relative z-10">{(cumulative / 1000).toFixed(1)}K</span>
                                </div>
                              );
                            });
                          })()
                        ) : (
                          <div className="py-2 text-center text-muted-foreground">Loading...</div>
                        )}
                        
                        {/* Current Price */}
                        <div className="py-2 text-center">
                          <span className={`text-lg font-bold ${futuresPrice.priceChangePercent >= 0 ? 'text-green-400' : 'text-red-400'}`}>
                            {futuresPrice.price.toLocaleString(undefined, { minimumFractionDigits: 1, maximumFractionDigits: 1 })}
                          </span>
                          <span className="text-xs text-muted-foreground ml-2">
                            {futuresPrice.markPrice.toLocaleString(undefined, { minimumFractionDigits: 1, maximumFractionDigits: 1 })}
                          </span>
                        </div>
                        
                        {/* Bids (Green) - Live from Binance */}
                        {futuresOrderBook.bids.length > 0 ? (
                          (() => {
                            const maxSize = Math.max(...futuresOrderBook.bids.map((o: any) => o.size));
                            let cumulative = 0;
                            return futuresOrderBook.bids.map((order: any, i: number) => {
                              cumulative += order.size * order.price;
                              return (
                                <div key={i} className="grid grid-cols-3 text-green-400 relative">
                                  <div className="absolute right-0 top-0 bottom-0 bg-green-500/10" style={{ width: `${(order.size / maxSize) * 100}%` }}></div>
                                  <span className="font-mono relative z-10">{order.price.toLocaleString(undefined, { minimumFractionDigits: 1, maximumFractionDigits: 1 })}</span>
                                  <span className="text-right font-mono relative z-10">{order.size.toFixed(3)}</span>
                                  <span className="text-right font-mono relative z-10">{(cumulative / 1000).toFixed(1)}K</span>
                                </div>
                              );
                            });
                          })()
                        ) : (
                          <div className="py-2 text-center text-muted-foreground">Loading...</div>
                        )}
                      </div>
                    </div>

                    {/* Trading Panel */}
                    <div className="rounded-xl bg-card border border-border p-4 space-y-4">
                      {/* Leverage & Order Type */}
                      <div className="flex items-center justify-between">
                        <div className="flex items-center gap-2">
                          <span className="text-xs text-muted-foreground">Cross</span>
                          <select className="bg-muted text-xs rounded px-2 py-1 border-none">
                            <option>20x</option>
                            <option>10x</option>
                            <option>5x</option>
                            <option>50x</option>
                            <option>100x</option>
                          </select>
                          <span className="text-xs text-muted-foreground">M</span>
                        </div>
                      </div>

                      {/* Order Type Tabs */}
                      <div className="flex gap-2 text-xs border-b border-border pb-2">
                        <button className="text-primary border-b-2 border-primary pb-1">Market</button>
                        <button className="text-muted-foreground hover:text-foreground pb-1">Limit</button>
                        <button className="text-muted-foreground hover:text-foreground pb-1">Stop Limit</button>
                      </div>

                      {/* Order Inputs */}
                      <div className="space-y-3">
                        <div>
                          <label className="text-xs text-muted-foreground">Avbl</label>
                          <div className="flex items-center justify-between text-sm">
                            <span>0.00 USDT</span>
                          </div>
                        </div>
                        
                        <div className="relative">
                          <Input 
                            placeholder="89257.7" 
                            className="pr-16 text-right font-mono" 
                            data-testid="input-futures-price"
                          />
                          <span className="absolute right-3 top-1/2 -translate-y-1/2 text-xs text-muted-foreground">USDT</span>
                        </div>
                        
                        <div className="relative">
                          <Input 
                            placeholder="Size" 
                            className="pr-16 text-right font-mono" 
                            data-testid="input-futures-size"
                          />
                          <select className="absolute right-2 top-1/2 -translate-y-1/2 bg-transparent text-xs border-none">
                            <option>USDT</option>
                            <option>BTC</option>
                          </select>
                        </div>

                        {/* Size Slider */}
                        <div className="flex items-center gap-2">
                          <input type="range" min="0" max="100" defaultValue="0" className="flex-1 h-1 bg-muted rounded-lg appearance-none cursor-pointer" />
                          <span className="text-xs text-muted-foreground">0%</span>
                        </div>

                        <div className="flex items-center gap-2 text-xs">
                          <input type="checkbox" id="tp-sl" className="rounded" />
                          <label htmlFor="tp-sl" className="text-muted-foreground">TP/SL</label>
                          <input type="checkbox" id="hidden" className="rounded ml-2" />
                          <label htmlFor="hidden" className="text-muted-foreground">Hidden Order</label>
                        </div>

                        {/* Margin Info */}
                        <div className="text-xs space-y-1 text-muted-foreground">
                          <div className="flex justify-between">
                            <span>Liq. Price</span>
                            <span>--</span>
                          </div>
                          <div className="flex justify-between">
                            <span>Margin</span>
                            <span>0.00 USDT</span>
                          </div>
                          <div className="flex justify-between">
                            <span>Max</span>
                            <span>0.00 USDT</span>
                          </div>
                        </div>

                        {/* Buy/Sell Buttons */}
                        <div className="grid grid-cols-2 gap-2">
                          <Button className="bg-green-600 hover:bg-green-700 text-white font-semibold" data-testid="button-buy-long">
                            Buy / Long
                          </Button>
                          <Button className="bg-red-600 hover:bg-red-700 text-white font-semibold" data-testid="button-sell-short">
                            Sell / Short
                          </Button>
                        </div>

                        {/* Enable Trading Button */}
                        <Button className="w-full cyber-gradient text-background font-semibold" data-testid="button-enable-trading">
                          Enable Trading
                        </Button>
                      </div>
                    </div>
                  </div>

                  {/* Bottom Section - Orders & Account */}
                  <div className="grid lg:grid-cols-3 gap-4">
                    {/* Orders Tabs */}
                    <div className="lg:col-span-2 rounded-xl bg-card border border-border p-4">
                      <div className="flex items-center gap-4 border-b border-border pb-3 mb-4 overflow-x-auto">
                        <button className="text-sm font-medium text-primary whitespace-nowrap">Open orders</button>
                        <button className="text-sm text-muted-foreground hover:text-foreground whitespace-nowrap">Positions</button>
                        <button className="text-sm text-muted-foreground hover:text-foreground whitespace-nowrap">Assets</button>
                        <button className="text-sm text-muted-foreground hover:text-foreground whitespace-nowrap">TWAP</button>
                        <button className="text-sm text-muted-foreground hover:text-foreground whitespace-nowrap">Order history</button>
                        <button className="text-sm text-muted-foreground hover:text-foreground whitespace-nowrap">Position History</button>
                        <button className="text-sm text-muted-foreground hover:text-foreground whitespace-nowrap">Trade history</button>
                        <button className="text-sm text-muted-foreground hover:text-foreground whitespace-nowrap">Transaction history</button>
                      </div>
                      <div className="flex flex-col items-center justify-center py-12 text-muted-foreground">
                        <Wallet className="w-12 h-12 mb-4 opacity-50" />
                        <p>Please connect a wallet first</p>
                      </div>
                    </div>

                    {/* Account Info */}
                    <div className="rounded-xl bg-card border border-border p-4 space-y-4">
                      <div className="flex items-center justify-between">
                        <h4 className="font-semibold">Account</h4>
                        <div className="flex gap-2">
                          <Button size="sm" className="bg-yellow-500 hover:bg-yellow-600 text-black text-xs">Deposit</Button>
                          <Button size="sm" variant="outline" className="text-xs">Withdraw</Button>
                          <Button size="sm" variant="outline" className="text-xs">Transfer</Button>
                        </div>
                      </div>
                      
                      <div className="space-y-3 text-sm">
                        <div>
                          <p className="text-muted-foreground text-xs">Account Equity</p>
                          <p className="font-bold">-- USDT</p>
                        </div>
                        <div className="grid grid-cols-2 gap-2 text-xs">
                          <div>
                            <p className="text-muted-foreground">Spot total value</p>
                            <p>--</p>
                          </div>
                          <div>
                            <p className="text-muted-foreground">Perp total value</p>
                            <p>--</p>
                          </div>
                          <div>
                            <p className="text-muted-foreground">Perpetuals unrealized Pnl</p>
                            <p>--</p>
                          </div>
                          <div>
                            <p className="text-muted-foreground">Shield unrealized Pnl</p>
                            <p>--</p>
                          </div>
                        </div>
                        
                        <div className="pt-2 border-t border-border">
                          <p className="text-muted-foreground text-xs mb-2">Margin</p>
                          <div className="grid grid-cols-2 gap-2 text-xs">
                            <div>
                              <p className="text-muted-foreground">Account Margin Ratio</p>
                              <p>--</p>
                            </div>
                            <div>
                              <p className="text-muted-foreground">Account Maintenance Margin</p>
                              <p className="text-primary">0.00%</p>
                            </div>
                            <div>
                              <p className="text-muted-foreground">Account Equity</p>
                              <p>--</p>
                            </div>
                          </div>
                        </div>
                      </div>
                    </div>
                  </div>

                  {/* Coming Soon Banner */}
                  <div className="p-4 rounded-xl bg-gradient-to-r from-primary/10 to-secondary/10 border border-primary/20 text-center">
                    <p className="text-lg font-semibold mb-2">Futures Trading Coming Soon</p>
                    <p className="text-sm text-muted-foreground">Trade perpetual contracts with up to 100x leverage. Stay tuned for the launch!</p>
                  </div>
                </div>
              )}

              {/* Free Audit Section */}
              {activeItem === "free-audit" && (
                <div className="space-y-6 max-w-3xl mx-auto">
                  {/* Header */}
                  <div className="text-center space-y-4">
                    <h2 className="text-3xl font-display font-bold text-glow-cyan">Free Smart Contract Audit</h2>
                    <p className="text-muted-foreground">{t("dashboard.freeAuditDesc")}</p>
                  </div>

                  {/* Contract Input */}
                  <div className="p-6 rounded-xl border-2 border-dashed border-primary/30 bg-primary/5 space-y-4">
                    <div className="flex justify-center">
                      <div className="w-16 h-16 rounded-xl cyber-gradient flex items-center justify-center glow-cyan">
                        <FileText className="w-8 h-8 text-background" />
                      </div>
                    </div>
                    
                    <div className="text-center">
                      <p className="font-medium mb-2">{t("dashboard.contractAddress")}</p>
                    </div>
                    
                    <Input
                      placeholder="0x..."
                      value={auditContractAddress}
                      onChange={(e) => setAuditContractAddress(e.target.value)}
                      className="text-center text-sm h-12 bg-background/50 border-primary/30"
                      data-testid="input-audit-contract"
                    />
                    
                    <div className="flex items-center justify-center gap-2 text-yellow-500 text-sm">
                      <AlertTriangle className="w-4 h-4" />
                      <span>{t("dashboard.contractVerified")}</span>
                    </div>
                  </div>

                  {/* Start Audit Button */}
                  <Button
                    onClick={handleStartAudit}
                    disabled={isAuditing || !auditContractAddress.trim()}
                    className="w-full h-14 text-lg font-semibold"
                    style={{ backgroundColor: "#c6ff00", color: "#000" }}
                    data-testid="button-start-audit"
                  >
                    {isAuditing ? (
                      <>
                        <Loader2 className="w-5 h-5 mr-2 animate-spin" />
                        {t("dashboard.auditInProgress")}
                      </>
                    ) : (
                      t("dashboard.startAudit")
                    )}
                  </Button>

                  {/* Progress Messages */}
                  {auditProgress.length > 0 && (
                    <div className="space-y-2">
                      {auditProgress.map((msg, i) => (
                        <div 
                          key={i} 
                          className={`flex items-center gap-3 p-3 rounded-lg ${
                            i === auditProgress.length - 1 && msg.includes("success") 
                              ? "bg-green-500/10 border border-green-500/30" 
                              : "bg-primary/5 border border-primary/20"
                          }`}
                        >
                          {msg.includes("success") || msg.includes("完成") || msg.includes("بنجاح") ? (
                            <CheckCircle className="w-5 h-5 text-green-400" />
                          ) : (
                            <Info className="w-5 h-5 text-primary" />
                          )}
                          <span className="text-sm">{msg}</span>
                        </div>
                      ))}
                    </div>
                  )}

                  {/* Error */}
                  {auditError && (
                    <div className="p-4 rounded-lg bg-red-500/10 border border-red-500/30 flex items-center gap-3">
                      <XCircle className="w-5 h-5 text-red-400" />
                      <p className="text-sm text-red-300">{auditError}</p>
                    </div>
                  )}

                  {/* Audit Results */}
                  {auditResult && (
                    <div className="space-y-6">
                      {/* Audit Score */}
                      <div className="flex items-center justify-center gap-8 p-6 rounded-xl bg-gradient-to-r from-primary/10 to-secondary/10 border border-primary/20">
                        <div className="text-center">
                          <div className={`text-6xl font-bold font-display ${
                            auditResult.auditScore >= 80 ? "text-green-400" :
                            auditResult.auditScore >= 60 ? "text-yellow-400" :
                            auditResult.auditScore >= 40 ? "text-orange-400" :
                            "text-red-400"
                          }`}>
                            {auditResult.auditScore}
                          </div>
                          <p className="text-sm text-muted-foreground mt-1">Audit Score</p>
                        </div>
                        <div className="text-center">
                          <div className={`text-6xl font-bold font-display ${
                            auditResult.grade?.startsWith("A") ? "text-green-400" :
                            auditResult.grade?.startsWith("B") ? "text-cyan-400" :
                            auditResult.grade?.startsWith("C") ? "text-yellow-400" :
                            "text-red-400"
                          }`}>
                            {auditResult.grade}
                          </div>
                          <p className="text-sm text-muted-foreground mt-1">Grade</p>
                        </div>
                        <div className="text-left space-y-1 text-sm text-muted-foreground">
                          <p><span className="text-foreground font-medium">{auditResult.tokenName}</span> ({auditResult.tokenSymbol})</p>
                          <p>Chain: BSC (BEP-20)</p>
                          <p>Holders: {auditResult.rawData?.holderCount?.toLocaleString() || "N/A"}</p>
                        </div>
                      </div>

                      {/* Summary Stats */}
                      <div className="grid grid-cols-5 gap-4">
                        {[
                          { label: t("dashboard.critical"), value: auditResult.summary?.critical || 0, color: "text-red-500" },
                          { label: t("dashboard.major"), value: auditResult.summary?.major || 0, color: "text-orange-500" },
                          { label: t("dashboard.medium"), value: auditResult.summary?.medium || 0, color: "text-yellow-500" },
                          { label: t("dashboard.minor"), value: auditResult.summary?.minor || 0, color: "text-blue-400" },
                          { label: t("dashboard.info"), value: auditResult.summary?.info || 0, color: "text-gray-400" },
                        ].map((stat) => (
                          <div key={stat.label} className="text-center p-4 rounded-lg bg-card border border-border">
                            <p className={`text-3xl font-bold ${stat.color}`}>{stat.value}</p>
                            <p className="text-sm text-muted-foreground">{stat.label}</p>
                          </div>
                        ))}
                      </div>

                      {/* Detailed Findings */}
                      <div className="space-y-4">
                        <h3 className="text-xl font-display font-semibold text-primary">{t("dashboard.detailedFindings")}</h3>
                        
                        {auditResult.findings?.length > 0 ? (
                          auditResult.findings.map((finding: any, i: number) => (
                            <div key={i} className="p-4 rounded-lg bg-card border border-border space-y-3">
                              <div className="flex items-center justify-between">
                                <span className={`px-3 py-1 rounded text-xs font-semibold uppercase ${
                                  finding.severity === "critical" ? "bg-red-500/20 text-red-400" :
                                  finding.severity === "major" ? "bg-orange-500/20 text-orange-400" :
                                  finding.severity === "medium" ? "bg-yellow-500/20 text-yellow-400" :
                                  finding.severity === "minor" ? "bg-blue-500/20 text-blue-400" :
                                  "bg-gray-500/20 text-gray-400"
                                }`}>
                                  {finding.severity}
                                </span>
                                <span className="text-sm text-muted-foreground">
                                  {finding.confidence === "high" ? t("dashboard.confidenceHigh") :
                                   finding.confidence === "medium" ? t("dashboard.confidenceMedium") :
                                   t("dashboard.confidenceLow")}
                                </span>
                              </div>
                              
                              <div className="p-3 rounded bg-primary/5 border-l-4 border-primary">
                                <p className="text-sm font-medium">{finding.title}</p>
                              </div>
                              
                              <div className="text-sm text-muted-foreground space-y-1">
                                <p className="flex items-center gap-2">
                                  <FileText className="w-4 h-4" />
                                  {finding.file || "contract.sol"}
                                </p>
                                {finding.lines && (
                                  <p className="flex items-center gap-2">
                                    # Lines: {finding.lines}
                                  </p>
                                )}
                              </div>
                              
                              <div className="pt-2">
                                <span className="px-2 py-1 rounded bg-primary/10 text-primary text-xs">
                                  Type: {finding.type}
                                </span>
                              </div>
                            </div>
                          ))
                        ) : (
                          <div className="p-6 rounded-lg bg-green-500/10 border border-green-500/30 text-center">
                            <CheckCircle className="w-12 h-12 text-green-400 mx-auto mb-3" />
                            <p className="text-green-400 font-medium">No security issues found!</p>
                            <p className="text-sm text-muted-foreground mt-1">This contract passed all security checks.</p>
                          </div>
                        )}
                      </div>

                      {/* Download Report Button */}
                      <Button
                        onClick={downloadAuditReport}
                        variant="outline"
                        className="w-full h-12 border-primary/30 text-primary"
                        data-testid="button-download-audit-report"
                      >
                        <Download className="w-5 h-5 mr-2" />
                        {t("dashboard.downloadReport")}
                      </Button>
                    </div>
                  )}
                </div>
              )}
            </div>
          </motion.div>
        </main>
      </div>
    </div>
  );
}
