import { createContext, useContext, useState, useEffect, ReactNode } from "react";

export type Language = "en" | "zh" | "ar";

type Translations = {
  [key: string]: {
    en: string;
    zh: string;
    ar: string;
  };
};

export const translations: Translations = {
  // Navigation
  "nav.home": { en: "Home", zh: "首页", ar: "الرئيسية" },
  "nav.tools": { en: "Tools", zh: "工具", ar: "الأدوات" },
  "nav.tokenomics": { en: "Tokenomics", zh: "代币经济", ar: "اقتصاديات الرمز" },
  "nav.roadmap": { en: "Roadmap", zh: "路线图", ar: "خارطة الطريق" },
  "nav.community": { en: "Community", zh: "社区", ar: "المجتمع" },
  "nav.docs": { en: "Docs", zh: "文档", ar: "المستندات" },
  "nav.launchApp": { en: "Launch App", zh: "启动应用", ar: "تشغيل التطبيق" },

  // Hero Section
  "hero.badge": { en: "AI-Powered Blockchain Utilities", zh: "AI驱动的区块链工具", ar: "أدوات بلوكتشين مدعومة بالذكاء الاصطناعي" },
  "hero.badgeAlt": { en: "AI-Powered Decentralized Finance", zh: "AI驱动的去中心化金融", ar: "التمويل اللامركزي المدعوم بالذكاء الاصطناعي" },
  "hero.launchDashboard": { en: "Launch Dashboard", zh: "启动仪表板", ar: "تشغيل لوحة التحكم" },
  "hero.viewChart": { en: "View Chart", zh: "查看图表", ar: "عرض الرسم البياني" },
  "hero.searchBtn": { en: "Search", zh: "搜索", ar: "بحث" },
  "hero.tagline": { en: "AI-Powered On-Chain Alpha Engine", zh: "AI驱动的链上Alpha引擎", ar: "محرك ألفا على السلسلة مدعوم بالذكاء الاصطناعي" },
  "hero.title1": { en: "Your", zh: "您的", ar: "مستشارك" },
  "hero.title2": { en: "Crypto Companion", zh: "加密货币伙伴", ar: "المشفر الموثوق" },
  "hero.description": { 
    en: "Astherafi here to help you navigate the crypto market by offering a suite of Sentiment, Technical, and On-Chain tools powered by AI.", 
    zh: "Astherafi 在这里通过 AI 驱动的情绪分析、技术分析和链上工具帮助您驾驭加密货币市场。", 
    ar: "Astherafi هنا لمساعدتك في التنقل في سوق العملات المشفرة من خلال تقديم مجموعة من أدوات التحليل العاطفي والتقني وعلى السلسلة المدعومة بالذكاء الاصطناعي." 
  },
  "hero.search": { en: "Search tokens....", zh: "搜索代币....", ar: "البحث عن الرموز...." },
  "hero.contract": { en: "Contract Address", zh: "合约地址", ar: "عنوان العقد" },
  "hero.copied": { en: "Copied!", zh: "已复制!", ar: "تم النسخ!" },

  // Tools Section
  "tools.title": { en: "Our Tools", zh: "我们的工具", ar: "أدواتنا" },
  "tools.tokenAnalyzer": { en: "Token Analyzer", zh: "代币分析器", ar: "محلل الرموز" },
  "tools.tokenAnalyzerDesc": { 
    en: "AI-powered token security analysis with sentiment, technical, and on-chain scores.", 
    zh: "AI驱动的代币安全分析，包含情绪、技术和链上评分。", 
    ar: "تحليل أمان الرموز المدعوم بالذكاء الاصطناعي مع درجات المشاعر والتقنية وعلى السلسلة." 
  },
  "tools.tokenGenerator": { en: "Token Generator", zh: "代币生成器", ar: "مولد الرموز" },
  "tools.tokenGeneratorDesc": { 
    en: "Create and deploy your own token on BSC with customizable parameters.", 
    zh: "在BSC上创建和部署具有可定制参数的代币。", 
    ar: "إنشاء ونشر الرمز الخاص بك على BSC مع معلمات قابلة للتخصيص." 
  },
  "tools.websiteBuilder": { en: "Website Builder", zh: "网站构建器", ar: "منشئ المواقع" },
  "tools.websiteBuilderDesc": { 
    en: "Generate professional token landing pages instantly with multiple themes.", 
    zh: "立即生成具有多种主题的专业代币登陆页面。", 
    ar: "إنشاء صفحات هبوط احترافية للرموز فورًا مع سمات متعددة." 
  },
  "tools.bundleChecker": { en: "Bundle Checker", zh: "捆绑检查器", ar: "مدقق الحزم" },
  "tools.bundleCheckerDesc": { 
    en: "Detect bundled transactions and identify potential rug pull patterns.", 
    zh: "检测捆绑交易并识别潜在的拉地毯模式。", 
    ar: "كشف المعاملات المجمعة وتحديد أنماط الاحتيال المحتملة." 
  },
  "tools.passwordScore": { en: "Password Score", zh: "密码评分", ar: "درجة كلمة المرور" },
  "tools.passwordScoreDesc": { 
    en: "Check the strength and security of your passwords with advanced scoring.", 
    zh: "使用高级评分检查密码的强度和安全性。", 
    ar: "تحقق من قوة وأمان كلمات المرور الخاصة بك مع تقييم متقدم." 
  },
  "tools.staking": { en: "Staking", zh: "质押", ar: "التخزين" },
  "tools.stakingDesc": { 
    en: "Stake your tokens to earn passive rewards with competitive APY.", 
    zh: "质押代币以赚取具有竞争力APY的被动收益。", 
    ar: "قم بتخزين رموزك لكسب مكافآت سلبية مع عائد سنوي تنافسي." 
  },

  // Tokenomics Section
  "tokenomics.title": { en: "Tokenomics", zh: "代币经济", ar: "اقتصاديات الرمز" },
  "tokenomics.totalSupply": { en: "Total Supply", zh: "总供应量", ar: "إجمالي العرض" },
  "tokenomics.tax": { en: "Tax", zh: "税率", ar: "الضريبة" },
  "tokenomics.chain": { en: "Chain", zh: "链", ar: "السلسلة" },
  "tokenomics.liquidity": { en: "Liquidity", zh: "流动性", ar: "السيولة" },
  "tokenomics.locked": { en: "Locked", zh: "已锁定", ar: "مقفل" },

  // Roadmap Section
  "roadmap.title": { en: "Roadmap", zh: "路线图", ar: "خارطة الطريق" },
  "roadmap.phase1": { en: "Phase 1", zh: "第一阶段", ar: "المرحلة الأولى" },
  "roadmap.foundation": { en: "Foundation", zh: "基础建设", ar: "الأساس" },
  "roadmap.phase1Item1": { en: "Launch Token Analyzer", zh: "启动代币分析器", ar: "إطلاق محلل الرموز" },
  "roadmap.phase1Item2": { en: "Deploy Token Generator", zh: "部署代币生成器", ar: "نشر مولد الرموز" },
  "roadmap.phase1Item3": { en: "Website Builder Release", zh: "网站构建器发布", ar: "إصدار منشئ المواقع" },
  "roadmap.phase1Item4": { en: "Community Building", zh: "社区建设", ar: "بناء المجتمع" },
  "roadmap.phase2": { en: "Phase 2", zh: "第二阶段", ar: "المرحلة الثانية" },
  "roadmap.expansion": { en: "Expansion", zh: "扩展", ar: "التوسع" },
  "roadmap.phase2Item1": { en: "Token Swap Integration", zh: "代币兑换集成", ar: "تكامل تبادل الرموز" },
  "roadmap.phase2Item2": { en: "Bundle Checker Launch", zh: "捆绑检查器启动", ar: "إطلاق مدقق الحزم" },
  "roadmap.phase2Item3": { en: "Multi-chain Support", zh: "多链支持", ar: "دعم متعدد السلاسل" },
  "roadmap.phase2Item4": { en: "Partnership Announcements", zh: "合作伙伴公告", ar: "إعلانات الشراكة" },
  "roadmap.phase3": { en: "Phase 3", zh: "第三阶段", ar: "المرحلة الثالثة" },
  "roadmap.growth": { en: "Growth", zh: "增长", ar: "النمو" },
  "roadmap.phase3Item1": { en: "Staking Platform Launch", zh: "质押平台启动", ar: "إطلاق منصة التخزين" },
  "roadmap.phase3Item2": { en: "Mobile App Development", zh: "移动应用开发", ar: "تطوير تطبيق الجوال" },
  "roadmap.phase3Item3": { en: "Advanced Analytics", zh: "高级分析", ar: "تحليلات متقدمة" },
  "roadmap.phase3Item4": { en: "CEX Listings", zh: "CEX上市", ar: "إدراج CEX" },
  "roadmap.phase4": { en: "Phase 4", zh: "第四阶段", ar: "المرحلة الرابعة" },
  "roadmap.innovation": { en: "Innovation", zh: "创新", ar: "الابتكار" },
  "roadmap.phase4Item1": { en: "AI Trading Signals", zh: "AI交易信号", ar: "إشارات التداول بالذكاء الاصطناعي" },
  "roadmap.phase4Item2": { en: "Cross-chain Bridge", zh: "跨链桥", ar: "جسر عبر السلاسل" },
  "roadmap.phase4Item3": { en: "API Access", zh: "API访问", ar: "الوصول إلى API" },
  "roadmap.phase4Item4": { en: "Global Expansion", zh: "全球扩展", ar: "التوسع العالمي" },

  // Community Section
  "community.title": { en: "Join Our Community", zh: "加入我们的社区", ar: "انضم إلى مجتمعنا" },
  "community.description": { 
    en: "Stay updated with the latest news and connect with fellow crypto enthusiasts.", 
    zh: "获取最新消息并与加密货币爱好者建立联系。", 
    ar: "ابق على اطلاع بأحدث الأخبار وتواصل مع عشاق العملات المشفرة." 
  },
  "community.twitter": { en: "Twitter", zh: "推特", ar: "تويتر" },
  "community.telegram": { en: "Telegram", zh: "电报", ar: "تيليجرام" },
  "community.discord": { en: "Discord", zh: "Discord", ar: "ديسكورد" },

  // Footer
  "footer.rights": { en: "All rights reserved.", zh: "版权所有。", ar: "جميع الحقوق محفوظة." },

  // Dashboard
  "dashboard.title": { en: "Dashboard", zh: "仪表板", ar: "لوحة التحكم" },
  "dashboard.overview": { en: "Dashboard Overview", zh: "仪表板概览", ar: "نظرة عامة على لوحة التحكم" },
  "dashboard.version": { en: "Dashboard v1.0", zh: "仪表板 v1.0", ar: "لوحة التحكم v1.0" },
  "dashboard.tokenAnalyzer": { en: "Token Analyzer", zh: "代币分析器", ar: "محلل الرموز" },
  "dashboard.tokenGenerator": { en: "Token Generator", zh: "代币生成器", ar: "مولد الرموز" },
  "dashboard.websiteBuilder": { en: "Website Builder", zh: "网站构建器", ar: "منشئ المواقع" },
  "dashboard.bundleChecker": { en: "Bundle Checker", zh: "捆绑检查器", ar: "مدقق الحزم" },
  "dashboard.passwordScore": { en: "Password Score", zh: "密码评分", ar: "درجة كلمة المرور" },
  "dashboard.staking": { en: "Staking", zh: "质押", ar: "التخزين" },
  "dashboard.tokenSwap": { en: "Token Swap", zh: "代币兑换", ar: "تبادل الرموز" },
  "dashboard.freeAudit": { en: "Free Audit", zh: "免费审计", ar: "تدقيق مجاني" },
  "dashboard.freeAuditDesc": { en: "Enter contract address and get AI-powered security analysis - completely free!", zh: "输入合约地址，获取AI驱动的安全分析 - 完全免费！", ar: "أدخل عنوان العقد واحصل على تحليل أمني مدعوم بالذكاء الاصطناعي - مجاني تمامًا!" },
  "dashboard.blockExplorer": { en: "Block Explorer", zh: "区块浏览器", ar: "مستكشف الكتل" },
  "dashboard.startAudit": { en: "Start Audit", zh: "开始审计", ar: "بدء التدقيق" },
  "dashboard.auditInProgress": { en: "Auditing...", zh: "审计中...", ar: "جاري التدقيق..." },
  "dashboard.contractAddress": { en: "Contract Address (BSC Mainnet)", zh: "合约地址 (BSC主网)", ar: "عنوان العقد (شبكة BSC الرئيسية)" },
  "dashboard.contractVerified": { en: "Contract must be verified on BscScan", zh: "合约必须在BscScan上验证", ar: "يجب التحقق من العقد على BscScan" },
  "dashboard.auditComplete": { en: "Audit completed successfully!", zh: "审计成功完成！", ar: "تم التدقيق بنجاح!" },
  "dashboard.detailedFindings": { en: "Detailed Findings", zh: "详细发现", ar: "النتائج التفصيلية" },
  "dashboard.downloadReport": { en: "Download Full Report", zh: "下载完整报告", ar: "تحميل التقرير الكامل" },
  "dashboard.critical": { en: "Critical", zh: "严重", ar: "حرج" },
  "dashboard.major": { en: "Major", zh: "重要", ar: "رئيسي" },
  "dashboard.medium": { en: "Medium", zh: "中等", ar: "متوسط" },
  "dashboard.minor": { en: "Minor", zh: "次要", ar: "ثانوي" },
  "dashboard.info": { en: "Info", zh: "信息", ar: "معلومات" },
  "dashboard.confidenceHigh": { en: "Confidence: High", zh: "置信度: 高", ar: "الثقة: عالية" },
  "dashboard.confidenceMedium": { en: "Confidence: Medium", zh: "置信度: 中", ar: "الثقة: متوسطة" },
  "dashboard.confidenceLow": { en: "Confidence: Low", zh: "置信度: 低", ar: "الثقة: منخفضة" },
  "dashboard.search": { en: "Search tokens...", zh: "搜索代币...", ar: "البحث عن الرموز..." },
  "dashboard.analyze": { en: "Analyze", zh: "分析", ar: "تحليل" },
  "dashboard.analyzing": { en: "Analyzing...", zh: "分析中...", ar: "جاري التحليل..." },
  "dashboard.selectChain": { en: "Select Chain", zh: "选择链", ar: "اختر السلسلة" },
  "dashboard.marketData": { en: "Market Data", zh: "市场数据", ar: "بيانات السوق" },
  "dashboard.securityFlags": { en: "Security Flags", zh: "安全标志", ar: "علامات الأمان" },
  "dashboard.aiAnalysis": { en: "AI Analysis", zh: "AI分析", ar: "تحليل الذكاء الاصطناعي" },
  "dashboard.simulationResults": { en: "Simulation Results", zh: "模拟结果", ar: "نتائج المحاكاة" },
  "dashboard.holderAnalysis": { en: "Holder Analysis", zh: "持有者分析", ar: "تحليل المالكين" },
  "dashboard.tradingSentiment": { en: "Trading Sentiment", zh: "交易情绪", ar: "معنويات التداول" },
  "dashboard.socialLinks": { en: "Social Links", zh: "社交链接", ar: "روابط التواصل" },
  "dashboard.price": { en: "Price", zh: "价格", ar: "السعر" },
  "dashboard.marketCap": { en: "Market Cap", zh: "市值", ar: "القيمة السوقية" },
  "dashboard.volume24h": { en: "Volume 24h", zh: "24小时交易量", ar: "حجم 24 ساعة" },
  "dashboard.liquidity": { en: "Liquidity", zh: "流动性", ar: "السيولة" },
  "dashboard.holders": { en: "Holders", zh: "持有者", ar: "المالكين" },
  "dashboard.enterPassword": { en: "Enter password to check", zh: "输入密码进行检查", ar: "أدخل كلمة المرور للتحقق" },
  "dashboard.checkPassword": { en: "Check Password", zh: "检查密码", ar: "تحقق من كلمة المرور" },
  "dashboard.passwordStrength": { en: "Password Strength", zh: "密码强度", ar: "قوة كلمة المرور" },
  "dashboard.weak": { en: "Weak", zh: "弱", ar: "ضعيف" },
  "dashboard.fair": { en: "Fair", zh: "一般", ar: "مقبول" },
  "dashboard.good": { en: "Good", zh: "良好", ar: "جيد" },
  "dashboard.strong": { en: "Strong", zh: "强", ar: "قوي" },
  "dashboard.veryStrong": { en: "Very Strong", zh: "非常强", ar: "قوي جداً" },
  "dashboard.enterWalletAddress": { en: "Enter wallet address", zh: "输入钱包地址", ar: "أدخل عنوان المحفظة" },
  "dashboard.checkBundle": { en: "Check Bundle", zh: "检查捆绑", ar: "تحقق من الحزمة" },
  "dashboard.checking": { en: "Checking...", zh: "检查中...", ar: "جاري التحقق..." },
  "dashboard.createToken": { en: "Create Token", zh: "创建代币", ar: "إنشاء رمز" },
  "dashboard.tokenName": { en: "Token Name", zh: "代币名称", ar: "اسم الرمز" },
  "dashboard.tokenSymbol": { en: "Token Symbol", zh: "代币符号", ar: "رمز العملة" },
  "dashboard.totalSupply": { en: "Total Supply", zh: "总供应量", ar: "إجمالي العرض" },
  "dashboard.deploy": { en: "Deploy Token", zh: "部署代币", ar: "نشر الرمز" },
  "dashboard.connectWallet": { en: "Connect Wallet", zh: "连接钱包", ar: "ربط المحفظة" },
  "dashboard.stakeTokens": { en: "Stake Tokens", zh: "质押代币", ar: "تخزين الرموز" },
  "dashboard.unstake": { en: "Unstake", zh: "取消质押", ar: "إلغاء التخزين" },
  "dashboard.rewards": { en: "Rewards", zh: "奖励", ar: "المكافآت" },
  "dashboard.apy": { en: "APY", zh: "年化收益率", ar: "العائد السنوي" },
  "dashboard.swap": { en: "Swap", zh: "兑换", ar: "تبادل" },
  "dashboard.from": { en: "From", zh: "从", ar: "من" },
  "dashboard.to": { en: "To", zh: "到", ar: "إلى" },
  "dashboard.slippage": { en: "Slippage", zh: "滑点", ar: "الانزلاق" },
  "dashboard.generateWebsite": { en: "Generate Website", zh: "生成网站", ar: "إنشاء موقع" },
  "dashboard.projectName": { en: "Project Name", zh: "项目名称", ar: "اسم المشروع" },
  "dashboard.projectDescription": { en: "Project Description", zh: "项目描述", ar: "وصف المشروع" },
  "dashboard.theme": { en: "Theme", zh: "主题", ar: "السمة" },
  "dashboard.selectTool": { en: "Select a tool from the sidebar to begin", zh: "从侧边栏选择工具开始", ar: "اختر أداة من الشريط الجانبي للبدء" },
  "dashboard.analyzeDesc": { en: "Analyze any token contract for security risks and metrics", zh: "分析任何代币合约的安全风险和指标", ar: "تحليل أي عقد رمز للمخاطر الأمنية والمقاييس" },
  "dashboard.generateDesc": { en: "Create and deploy your own token on blockchain", zh: "在区块链上创建和部署您自己的代币", ar: "إنشاء ونشر الرمز الخاص بك على البلوكتشين" },
  "dashboard.websiteDesc": { en: "Build a professional website for your project", zh: "为您的项目构建专业网站", ar: "بناء موقع احترافي لمشروعك" },
  "dashboard.bundleDesc": { en: "Check bundled transactions and wallet activity", zh: "检查捆绑交易和钱包活动", ar: "التحقق من المعاملات المجمعة ونشاط المحفظة" },
  "dashboard.passwordDesc": { en: "Evaluate the strength of your passwords", zh: "评估密码强度", ar: "تقييم قوة كلمات المرور" },
  "dashboard.stakingDesc": { en: "Stake your tokens and earn rewards", zh: "质押代币并获得奖励", ar: "قم بتخزين الرموز واكسب المكافآت" },
  "dashboard.swapDesc": { en: "Exchange tokens across different chains", zh: "在不同链之间交换代币", ar: "تبادل الرموز عبر سلاسل مختلفة" },
  "dashboard.home": { en: "Home", zh: "首页", ar: "الرئيسية" },

  // Docs/Whitepaper
  "docs.backToHome": { en: "Back to Home", zh: "返回首页", ar: "العودة للرئيسية" },
  "docs.whitepaper": { en: "Whitepaper v1.0", zh: "白皮书 v1.0", ar: "الورقة البيضاء v1.0" },
  "docs.officialDocs": { en: "Official Documentation", zh: "官方文档", ar: "الوثائق الرسمية" },
  "docs.subtitle": { en: "AI-Powered Blockchain Utilities Platform for Smart Crypto Investors", zh: "为智能加密投资者打造的AI驱动区块链工具平台", ar: "منصة أدوات البلوكتشين المدعومة بالذكاء الاصطناعي للمستثمرين الأذكياء" },
  "docs.introduction": { en: "Introduction", zh: "简介", ar: "المقدمة" },
  "docs.introText1": { en: "Astherafi is a comprehensive blockchain utilities platform designed to empower cryptocurrency investors with intelligent tools for token analysis, security verification, and portfolio management. Our platform combines advanced artificial intelligence with real-time blockchain data to provide actionable insights that help users make informed investment decisions.", zh: "Astherafi 是一个全面的区块链工具平台，旨在为加密货币投资者提供智能工具，用于代币分析、安全验证和投资组合管理。我们的平台将先进的人工智能与实时区块链数据相结合，提供可操作的见解，帮助用户做出明智的投资决策。", ar: "Astherafi هي منصة شاملة لأدوات البلوكتشين مصممة لتمكين مستثمري العملات المشفرة بأدوات ذكية لتحليل الرموز والتحقق من الأمان وإدارة المحافظ. تجمع منصتنا بين الذكاء الاصطناعي المتقدم وبيانات البلوكتشين في الوقت الفعلي لتوفير رؤى قابلة للتنفيذ تساعد المستخدمين على اتخاذ قرارات استثمارية مستنيرة." },
  "docs.introText2": { en: "In an ever-evolving crypto landscape filled with opportunities and risks, Astherafi serves as your trusted companion, offering sophisticated analysis tools that were once available only to institutional investors and professional traders.", zh: "在充满机遇和风险的不断发展的加密货币领域，Astherafi 是您值得信赖的伙伴，提供曾经只有机构投资者和专业交易员才能使用的复杂分析工具。", ar: "في مشهد التشفير المتطور باستمرار والمليء بالفرص والمخاطر، تعمل Astherafi كرفيقك الموثوق، حيث تقدم أدوات تحليل متطورة كانت متاحة فقط للمستثمرين المؤسسيين والمتداولين المحترفين." },
  "docs.ourVision": { en: "Our Vision", zh: "我们的愿景", ar: "رؤيتنا" },
  "docs.democratizeSecurity": { en: "Democratize Security", zh: "安全民主化", ar: "إضفاء الطابع الديمقراطي على الأمان" },
  "docs.democratizeSecurityDesc": { en: "Make professional-grade security analysis accessible to every crypto investor, regardless of technical expertise or portfolio size.", zh: "让每一位加密投资者都能获得专业级别的安全分析，无论技术专长或投资组合规模如何。", ar: "جعل تحليل الأمان الاحترافي متاحًا لكل مستثمر في العملات المشفرة، بغض النظر عن الخبرة التقنية أو حجم المحفظة." },
  "docs.aiFirstApproach": { en: "AI-First Approach", zh: "AI优先策略", ar: "نهج الذكاء الاصطناعي أولاً" },
  "docs.aiFirstApproachDesc": { en: "Leverage cutting-edge artificial intelligence to analyze complex blockchain data and provide clear, actionable recommendations.", zh: "利用尖端人工智能分析复杂的区块链数据，并提供清晰、可操作的建议。", ar: "الاستفادة من الذكاء الاصطناعي المتطور لتحليل بيانات البلوكتشين المعقدة وتقديم توصيات واضحة وقابلة للتنفيذ." },
  "docs.multiChainSupport": { en: "Multi-Chain Support", zh: "多链支持", ar: "دعم متعدد السلاسل" },
  "docs.multiChainSupportDesc": { en: "Support major blockchain networks including Ethereum, BNB Smart Chain, Solana, Polygon, Arbitrum, Base, and more.", zh: "支持主要区块链网络，包括以太坊、BNB智能链、Solana、Polygon、Arbitrum、Base等。", ar: "دعم شبكات البلوكتشين الرئيسية بما في ذلك إيثريوم و BNB Smart Chain و Solana و Polygon و Arbitrum و Base والمزيد." },
  "docs.communityDriven": { en: "Community Driven", zh: "社区驱动", ar: "مدفوع بالمجتمع" },
  "docs.communityDrivenDesc": { en: "Build tools based on real user needs and feedback, constantly evolving to address the challenges faced by the crypto community.", zh: "基于真实用户需求和反馈构建工具，不断发展以应对加密社区面临的挑战。", ar: "بناء أدوات بناءً على احتياجات المستخدمين الحقيقية وملاحظاتهم، والتطور المستمر لمعالجة التحديات التي يواجهها مجتمع التشفير." },
  "docs.platformFeatures": { en: "Platform Features", zh: "平台功能", ar: "ميزات المنصة" },
  "docs.securityPrivacy": { en: "Security & Privacy", zh: "安全与隐私", ar: "الأمان والخصوصية" },
  "docs.dataProtection": { en: "Data Protection", zh: "数据保护", ar: "حماية البيانات" },
  "docs.noPersonalData": { en: "No personal data collection required", zh: "无需收集个人数据", ar: "لا يلزم جمع بيانات شخصية" },
  "docs.walletReadOnly": { en: "Wallet connections are read-only by default", zh: "钱包连接默认为只读", ar: "اتصالات المحفظة للقراءة فقط افتراضياً" },
  "docs.explicitApproval": { en: "All sensitive operations require explicit approval", zh: "所有敏感操作需要明确批准", ar: "جميع العمليات الحساسة تتطلب موافقة صريحة" },
  "docs.platformSecurity": { en: "Platform Security", zh: "平台安全", ar: "أمان المنصة" },
  "docs.httpsEncryption": { en: "HTTPS encryption on all connections", zh: "所有连接均使用HTTPS加密", ar: "تشفير HTTPS على جميع الاتصالات" },
  "docs.noPrivateKeys": { en: "No private keys stored on servers", zh: "服务器上不存储私钥", ar: "لا يتم تخزين المفاتيح الخاصة على الخوادم" },
  "docs.regularAudits": { en: "Regular security audits and updates", zh: "定期安全审计和更新", ar: "عمليات تدقيق وتحديثات أمنية منتظمة" },
  "docs.roadmap": { en: "Roadmap", zh: "路线图", ar: "خارطة الطريق" },
  "docs.phase1": { en: "Phase 1", zh: "第一阶段", ar: "المرحلة 1" },
  "docs.foundation": { en: "Foundation", zh: "基础建设", ar: "الأساس" },
  "docs.phase2": { en: "Phase 2", zh: "第二阶段", ar: "المرحلة 2" },
  "docs.expansion": { en: "Expansion", zh: "扩展", ar: "التوسع" },
  "docs.phase3": { en: "Phase 3", zh: "第三阶段", ar: "المرحلة 3" },
  "docs.growth": { en: "Growth", zh: "增长", ar: "النمو" },
  "docs.phase4": { en: "Phase 4", zh: "第四阶段", ar: "المرحلة 4" },
  "docs.maturity": { en: "Maturity", zh: "成熟", ar: "النضج" },
  "docs.tokenomics": { en: "Tokenomics", zh: "代币经济学", ar: "اقتصاديات الرموز" },
  "docs.tokenomicsDesc": { en: "The ASTH token powers the Astherafi ecosystem, providing utility and governance capabilities.", zh: "ASTH代币为Astherafi生态系统提供动力，提供实用性和治理能力。", ar: "يعمل رمز ASTH على تشغيل نظام Astherafi البيئي، مما يوفر إمكانيات الاستخدام والحوكمة." },
  "docs.totalSupply": { en: "Total Supply", zh: "总供应量", ar: "إجمالي العرض" },
  "docs.tax": { en: "Tax", zh: "税率", ar: "الضريبة" },
  "docs.network": { en: "Network", zh: "网络", ar: "الشبكة" },
  "docs.liquidityStatus": { en: "Liquidity Status", zh: "流动性状态", ar: "حالة السيولة" },
  "docs.locked": { en: "Locked", zh: "已锁定", ar: "مقفل" },
  "docs.completed": { en: "Completed", zh: "已完成", ar: "مكتمل" },
  "docs.inProgress": { en: "In Progress", zh: "进行中", ar: "قيد التنفيذ" },
  "docs.upcoming": { en: "Upcoming", zh: "即将推出", ar: "قادم" },

  // Docs Features - Token Analyzer
  "docs.feature.tokenAnalyzer": { en: "Token Analyzer", zh: "代币分析器", ar: "محلل الرموز" },
  "docs.feature.tokenAnalyzerBadge": { en: "AI-Powered", zh: "AI驱动", ar: "مدعوم بالذكاء الاصطناعي" },
  "docs.feature.tokenAnalyzerDesc": { en: "Our flagship feature combines multiple analysis dimensions to give you a complete picture of any token.", zh: "我们的旗舰功能结合多个分析维度，为您提供任何代币的完整图景。", ar: "ميزتنا الرئيسية تجمع بين أبعاد تحليلية متعددة لتقديم صورة كاملة عن أي رمز." },
  "docs.feature.tokenAnalyzer1": { en: "Security risk assessment with 20+ vulnerability checks", zh: "包含20+漏洞检查的安全风险评估", ar: "تقييم المخاطر الأمنية مع أكثر من 20 فحص للثغرات" },
  "docs.feature.tokenAnalyzer2": { en: "AI-powered sentiment analysis using advanced language models", zh: "使用先进语言模型的AI情绪分析", ar: "تحليل المشاعر المدعوم بالذكاء الاصطناعي باستخدام نماذج لغوية متقدمة" },
  "docs.feature.tokenAnalyzer3": { en: "Technical analysis scores based on contract patterns", zh: "基于合约模式的技术分析评分", ar: "درجات التحليل الفني بناءً على أنماط العقود" },
  "docs.feature.tokenAnalyzer4": { en: "On-chain metrics including holder distribution and liquidity", zh: "包括持有者分布和流动性的链上指标", ar: "مقاييس على السلسلة بما في ذلك توزيع المالكين والسيولة" },
  "docs.feature.tokenAnalyzer5": { en: "Real-time market data with price change tracking", zh: "实时市场数据与价格变化跟踪", ar: "بيانات السوق في الوقت الفعلي مع تتبع تغير الأسعار" },
  "docs.feature.tokenAnalyzer6": { en: "Social presence verification (Twitter, Telegram, Discord)", zh: "社交存在验证（Twitter、Telegram、Discord）", ar: "التحقق من الوجود الاجتماعي (تويتر، تيليجرام، ديسكورد)" },

  // Docs Features - Token Generator
  "docs.feature.tokenGenerator": { en: "Token Generator", zh: "代币生成器", ar: "مولد الرموز" },
  "docs.feature.tokenGeneratorBadge": { en: "BSC", zh: "BSC", ar: "BSC" },
  "docs.feature.tokenGeneratorDesc": { en: "Create your own token on BNB Smart Chain with customizable parameters.", zh: "在BNB智能链上创建具有可定制参数的代币。", ar: "أنشئ الرمز الخاص بك على سلسلة BNB الذكية بمعلمات قابلة للتخصيص." },
  "docs.feature.tokenGenerator1": { en: "ERC-20 compliant token creation", zh: "符合ERC-20标准的代币创建", ar: "إنشاء رمز متوافق مع ERC-20" },
  "docs.feature.tokenGenerator2": { en: "Customizable name, symbol, and supply", zh: "可自定义名称、符号和供应量", ar: "اسم ورمز وعرض قابل للتخصيص" },
  "docs.feature.tokenGenerator3": { en: "Automatic liquidity pool setup", zh: "自动流动性池设置", ar: "إعداد مجمع السيولة التلقائي" },
  "docs.feature.tokenGenerator4": { en: "Trading fee configuration", zh: "交易费用配置", ar: "تكوين رسوم التداول" },
  "docs.feature.tokenGenerator5": { en: "Direct wallet integration", zh: "直接钱包集成", ar: "تكامل المحفظة المباشر" },

  // Docs Features - Bundle Checker
  "docs.feature.bundleChecker": { en: "Bundle Checker", zh: "捆绑检查器", ar: "مدقق الحزم" },
  "docs.feature.bundleCheckerBadge": { en: "Anti-Rug", zh: "防跑路", ar: "مضاد للاحتيال" },
  "docs.feature.bundleCheckerDesc": { en: "Detect suspicious wallet activity and potential rug pull patterns.", zh: "检测可疑钱包活动和潜在的跑路模式。", ar: "اكتشاف نشاط المحفظة المشبوه وأنماط الاحتيال المحتملة." },
  "docs.feature.bundleChecker1": { en: "Identify coordinated wallet groups", zh: "识别协调的钱包群组", ar: "تحديد مجموعات المحافظ المنسقة" },
  "docs.feature.bundleChecker2": { en: "Track bundled transaction patterns", zh: "跟踪捆绑交易模式", ar: "تتبع أنماط المعاملات المجمعة" },
  "docs.feature.bundleChecker3": { en: "Sniper bot detection", zh: "狙击机器人检测", ar: "اكتشاف روبوتات القناص" },
  "docs.feature.bundleChecker4": { en: "Whale wallet monitoring", zh: "巨鲸钱包监控", ar: "مراقبة محافظ الحيتان" },
  "docs.feature.bundleChecker5": { en: "Historical transaction analysis", zh: "历史交易分析", ar: "تحليل المعاملات التاريخية" },

  // Docs Features - Website Builder
  "docs.feature.websiteBuilder": { en: "Website Builder", zh: "网站构建器", ar: "منشئ المواقع" },
  "docs.feature.websiteBuilderBadge": { en: "No-Code", zh: "无代码", ar: "بدون كود" },
  "docs.feature.websiteBuilderDesc": { en: "Generate professional token landing pages instantly.", zh: "即时生成专业的代币登陆页面。", ar: "إنشاء صفحات هبوط احترافية للرموز على الفور." },
  "docs.feature.websiteBuilder1": { en: "Multiple theme options (Dark, Light, Cyber)", zh: "多种主题选项（深色、浅色、赛博）", ar: "خيارات سمات متعددة (داكن، فاتح، سايبر)" },
  "docs.feature.websiteBuilder2": { en: "Automatic content generation", zh: "自动内容生成", ar: "إنشاء المحتوى التلقائي" },
  "docs.feature.websiteBuilder3": { en: "Responsive design for all devices", zh: "适配所有设备的响应式设计", ar: "تصميم متجاوب لجميع الأجهزة" },
  "docs.feature.websiteBuilder4": { en: "Social links integration", zh: "社交链接集成", ar: "تكامل روابط التواصل الاجتماعي" },
  "docs.feature.websiteBuilder5": { en: "One-click export", zh: "一键导出", ar: "تصدير بنقرة واحدة" },

  // Docs Features - Password Security
  "docs.feature.passwordSecurity": { en: "Password Security", zh: "密码安全", ar: "أمان كلمة المرور" },
  "docs.feature.passwordSecurityBadge": { en: "Privacy", zh: "隐私", ar: "الخصوصية" },
  "docs.feature.passwordSecurityDesc": { en: "Check the strength of your passwords with our advanced scoring system.", zh: "使用我们先进的评分系统检查密码强度。", ar: "تحقق من قوة كلمات المرور باستخدام نظام التقييم المتقدم." },
  "docs.feature.passwordSecurity1": { en: "Real-time strength analysis", zh: "实时强度分析", ar: "تحليل القوة في الوقت الفعلي" },
  "docs.feature.passwordSecurity2": { en: "Pattern detection", zh: "模式检测", ar: "اكتشاف الأنماط" },
  "docs.feature.passwordSecurity3": { en: "Entropy calculation", zh: "熵值计算", ar: "حساب الإنتروبيا" },
  "docs.feature.passwordSecurity4": { en: "Improvement suggestions", zh: "改进建议", ar: "اقتراحات التحسين" },
  "docs.feature.passwordSecurity5": { en: "100% client-side processing", zh: "100%客户端处理", ar: "معالجة 100% من جانب العميل" },

  // Docs Features - Token Staking
  "docs.feature.tokenStaking": { en: "Token Staking", zh: "代币质押", ar: "تخزين الرموز" },
  "docs.feature.tokenStakingBadge": { en: "Earn", zh: "赚取", ar: "اكسب" },
  "docs.feature.tokenStakingDesc": { en: "Stake your tokens to earn passive rewards.", zh: "质押代币以获得被动奖励。", ar: "قم بتخزين الرموز للحصول على مكافآت سلبية." },
  "docs.feature.tokenStaking1": { en: "Flexible staking periods", zh: "灵活的质押期限", ar: "فترات تخزين مرنة" },
  "docs.feature.tokenStaking2": { en: "Competitive APY rates", zh: "有竞争力的年化收益率", ar: "معدلات عائد سنوي تنافسية" },
  "docs.feature.tokenStaking3": { en: "Real-time reward tracking", zh: "实时奖励跟踪", ar: "تتبع المكافآت في الوقت الفعلي" },
  "docs.feature.tokenStaking4": { en: "Easy unstaking process", zh: "简单的解除质押流程", ar: "عملية إلغاء التخزين السهلة" },

  // Docs Features - Token Swap
  "docs.feature.tokenSwap": { en: "Token Swap", zh: "代币兑换", ar: "تبادل الرموز" },
  "docs.feature.tokenSwapBadge": { en: "DEX", zh: "DEX", ar: "DEX" },
  "docs.feature.tokenSwapDesc": { en: "Swap tokens directly within the platform.", zh: "在平台内直接兑换代币。", ar: "تبادل الرموز مباشرة داخل المنصة." },
  "docs.feature.tokenSwap1": { en: "Multi-chain support", zh: "多链支持", ar: "دعم متعدد السلاسل" },
  "docs.feature.tokenSwap2": { en: "Best rate aggregation", zh: "最佳汇率聚合", ar: "تجميع أفضل الأسعار" },
  "docs.feature.tokenSwap3": { en: "Slippage protection", zh: "滑点保护", ar: "حماية الانزلاق" },
  "docs.feature.tokenSwap4": { en: "Transaction tracking", zh: "交易跟踪", ar: "تتبع المعاملات" },

  // Docs - Token Utilities
  "docs.tokenUtilities": { en: "Token Utilities", zh: "代币用途", ar: "استخدامات الرمز" },
  "docs.utility1": { en: "Access premium analysis features", zh: "访问高级分析功能", ar: "الوصول إلى ميزات التحليل المتميزة" },
  "docs.utility2": { en: "Reduced platform fees", zh: "降低平台费用", ar: "رسوم منصة مخفضة" },
  "docs.utility3": { en: "Staking rewards", zh: "质押奖励", ar: "مكافآت التخزين" },
  "docs.utility4": { en: "Governance voting rights", zh: "治理投票权", ar: "حقوق التصويت في الحوكمة" },
  "docs.utility5": { en: "Early access to new features", zh: "抢先体验新功能", ar: "الوصول المبكر للميزات الجديدة" },
  "docs.utility6": { en: "Priority customer support", zh: "优先客户支持", ar: "دعم العملاء ذو الأولوية" },

  // Docs - Disclaimer
  "docs.disclaimer": { en: "Disclaimer", zh: "免责声明", ar: "إخلاء المسؤولية" },
  "docs.disclaimerNotice": { en: "Important Notice:", zh: "重要提示：", ar: "إشعار هام:" },
  "docs.disclaimerText1": { en: "The information provided by Astherafi is for educational and informational purposes only. It should not be considered as financial, investment, or trading advice.", zh: "Astherafi提供的信息仅用于教育和信息目的。不应将其视为财务、投资或交易建议。", ar: "المعلومات المقدمة من Astherafi هي لأغراض تعليمية وإعلامية فقط. لا ينبغي اعتبارها نصيحة مالية أو استثمارية أو تجارية." },
  "docs.disclaimerText2": { en: "Cryptocurrency investments carry significant risks, including the potential loss of all invested capital. Always conduct your own research (DYOR) and consult with qualified financial advisors before making any investment decisions.", zh: "加密货币投资存在重大风险，包括可能损失所有投资资本。在做出任何投资决定之前，请始终进行自己的研究（DYOR）并咨询合格的财务顾问。", ar: "تحمل استثمارات العملات المشفرة مخاطر كبيرة، بما في ذلك الخسارة المحتملة لجميع رأس المال المستثمر. قم دائمًا بإجراء البحث الخاص بك (DYOR) واستشر المستشارين الماليين المؤهلين قبل اتخاذ أي قرارات استثمارية." },
  "docs.disclaimerText3": { en: "Astherafi makes no guarantees regarding the accuracy, completeness, or timeliness of the information provided. Users are solely responsible for their investment decisions and should understand that past performance is not indicative of future results.", zh: "Astherafi不对所提供信息的准确性、完整性或及时性做出任何保证。用户对其投资决定负全部责任，并应了解过去的表现不代表未来的结果。", ar: "لا تقدم Astherafi أي ضمانات فيما يتعلق بدقة أو اكتمال أو توقيت المعلومات المقدمة. المستخدمون مسؤولون وحدهم عن قراراتهم الاستثمارية ويجب أن يفهموا أن الأداء السابق لا يدل على النتائج المستقبلية." },

  // Docs - Roadmap Items
  "docs.roadmap.phase1.item1": { en: "Core platform development", zh: "核心平台开发", ar: "تطوير المنصة الأساسية" },
  "docs.roadmap.phase1.item2": { en: "Token Analyzer with security checks", zh: "带安全检查的代币分析器", ar: "محلل الرموز مع فحوصات الأمان" },
  "docs.roadmap.phase1.item3": { en: "AI integration for sentiment analysis", zh: "情绪分析AI集成", ar: "تكامل الذكاء الاصطناعي لتحليل المشاعر" },
  "docs.roadmap.phase1.item4": { en: "Multi-chain support (ETH, BSC, SOL)", zh: "多链支持（ETH、BSC、SOL）", ar: "دعم متعدد السلاسل (ETH، BSC، SOL)" },
  "docs.roadmap.phase2.item1": { en: "Token Generator on BSC", zh: "BSC代币生成器", ar: "مولد الرموز على BSC" },
  "docs.roadmap.phase2.item2": { en: "Bundle Checker tool", zh: "捆绑检查工具", ar: "أداة فحص الحزم" },
  "docs.roadmap.phase2.item3": { en: "Website Builder", zh: "网站构建器", ar: "منشئ المواقع" },
  "docs.roadmap.phase2.item4": { en: "Mobile-responsive design", zh: "移动响应式设计", ar: "تصميم متجاوب للجوال" },
  "docs.roadmap.phase3.item1": { en: "Token Staking platform", zh: "代币质押平台", ar: "منصة تخزين الرموز" },
  "docs.roadmap.phase3.item2": { en: "DEX aggregator integration", zh: "DEX聚合器集成", ar: "تكامل مجمع DEX" },
  "docs.roadmap.phase3.item3": { en: "Portfolio tracking", zh: "投资组合跟踪", ar: "تتبع المحفظة" },
  "docs.roadmap.phase3.item4": { en: "Alert notifications system", zh: "警报通知系统", ar: "نظام إشعارات التنبيه" },
  "docs.roadmap.phase4.item1": { en: "Advanced AI trading signals", zh: "高级AI交易信号", ar: "إشارات تداول ذكاء اصطناعي متقدمة" },
  "docs.roadmap.phase4.item2": { en: "Cross-chain bridge integration", zh: "跨链桥集成", ar: "تكامل جسر عبر السلاسل" },
  "docs.roadmap.phase4.item3": { en: "Mobile application", zh: "移动应用程序", ar: "تطبيق الجوال" },
  "docs.roadmap.phase4.item4": { en: "API access for developers", zh: "开发者API访问", ar: "وصول API للمطورين" },
  "docs.innovation": { en: "Innovation", zh: "创新", ar: "الابتكار" },

  // Docs - Ready to Get Started
  "docs.readyToStart": { en: "Ready to Get Started?", zh: "准备好开始了吗？", ar: "مستعد للبدء؟" },
  "docs.readyToStartDesc": { en: "Join thousands of crypto investors using Astherafi to make smarter decisions.", zh: "加入成千上万使用Astherafi做出更明智决策的加密投资者。", ar: "انضم إلى آلاف مستثمري العملات المشفرة الذين يستخدمون Astherafi لاتخاذ قرارات أذكى." },
  "docs.launchApp": { en: "Launch App", zh: "启动应用", ar: "تشغيل التطبيق" },
  "docs.backToHomeBtn": { en: "Back to Home", zh: "返回首页", ar: "العودة للرئيسية" },

  // Language names
  "lang.en": { en: "English", zh: "英语", ar: "الإنجليزية" },
  "lang.zh": { en: "中文", zh: "中文", ar: "الصينية" },
  "lang.ar": { en: "العربية", zh: "阿拉伯语", ar: "العربية" },
};

interface LanguageContextType {
  language: Language;
  setLanguage: (lang: Language) => void;
  t: (key: string) => string;
  dir: "ltr" | "rtl";
}

const LanguageContext = createContext<LanguageContextType | undefined>(undefined);

export function LanguageProvider({ children }: { children: ReactNode }) {
  const [language, setLanguageState] = useState<Language>(() => {
    if (typeof window !== "undefined") {
      const saved = localStorage.getItem("astherafi-lang");
      if (saved === "en" || saved === "zh" || saved === "ar") {
        return saved;
      }
    }
    return "en";
  });

  const setLanguage = (lang: Language) => {
    setLanguageState(lang);
    localStorage.setItem("astherafi-lang", lang);
  };

  useEffect(() => {
    document.documentElement.dir = language === "ar" ? "rtl" : "ltr";
    document.documentElement.lang = language;
  }, [language]);

  const t = (key: string): string => {
    const translation = translations[key];
    if (!translation) return key;
    return translation[language] || translation.en || key;
  };

  const dir = language === "ar" ? "rtl" : "ltr";

  return (
    <LanguageContext.Provider value={{ language, setLanguage, t, dir }}>
      {children}
    </LanguageContext.Provider>
  );
}

export function useLanguage() {
  const context = useContext(LanguageContext);
  if (!context) {
    throw new Error("useLanguage must be used within a LanguageProvider");
  }
  return context;
}
