import { useState } from "react";
import { motion } from "framer-motion";
import {
  Search,
  Layers,
  Coins,
  Shield,
  Percent,
  ChevronRight,
  ChevronLeft,
  Bell,
  Settings,
  User,
  TrendingUp,
  Zap,
  Activity,
  BarChart3,
  Home,
  PanelLeftClose,
  PanelLeft,
  Menu,
  Loader2,
  CheckCircle,
  XCircle,
  AlertCircle,
  Globe,
  Twitter,
  MessageCircle,
  ExternalLink,
} from "lucide-react";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";

type MenuItem = {
  id: string;
  label: string;
  icon: React.ReactNode;
  description: string;
};

const menuItems: MenuItem[] = [
  {
    id: "home",
    label: "Home",
    icon: <Home className="w-5 h-5" />,
    description: "Welcome to Astherafi - your blockchain utilities dashboard",
  },
  {
    id: "token-analyzer",
    label: "Token Analyzer",
    icon: <Search className="w-5 h-5" />,
    description: "Analyze any token contract for security risks and metrics",
  },
  {
    id: "bsc-bundler",
    label: "BSC Bundler",
    icon: <Layers className="w-5 h-5" />,
    description: "Bundle multiple transactions on Binance Smart Chain",
  },
  {
    id: "token-generator",
    label: "Token Generator",
    icon: <Coins className="w-5 h-5" />,
    description: "Create and deploy custom tokens in minutes",
  },
  {
    id: "password-score",
    label: "Password Score",
    icon: <Shield className="w-5 h-5" />,
    description: "Check the strength of your wallet passwords",
  },
  {
    id: "staking",
    label: "Staking",
    icon: <Percent className="w-5 h-5" />,
    description: "Stake tokens and earn passive rewards",
  },
];

const CHAINS = [
  { id: "bsc", name: "BSC", icon: "🔶" },
  { id: "ethereum", name: "Ethereum", icon: "⟠" },
  { id: "solana", name: "Solana", icon: "☀️" },
  { id: "polygon", name: "Polygon", icon: "🟣" },
  { id: "arbitrum", name: "Arbitrum", icon: "🔵" },
  { id: "base", name: "Base", icon: "🔷" },
];

export default function Dashboard() {
  const [activeItem, setActiveItem] = useState("home");
  const [sidebarOpen, setSidebarOpen] = useState(true);
  const [tokenAddress, setTokenAddress] = useState("");
  const [selectedChain, setSelectedChain] = useState("bsc");
  const [tokenAnalysis, setTokenAnalysis] = useState<any>(null);
  const [isAnalyzing, setIsAnalyzing] = useState(false);
  const [analysisError, setAnalysisError] = useState<string | null>(null);
  const [homeSearchAddress, setHomeSearchAddress] = useState("");

  const activeMenu = menuItems.find((item) => item.id === activeItem);

  const handleHomeSearch = () => {
    if (!homeSearchAddress.trim()) return;
    setTokenAddress(homeSearchAddress.trim());
    setActiveItem("token-analyzer");
    setTimeout(() => {
      analyzeTokenWithAddress(homeSearchAddress.trim());
    }, 100);
  };

  const analyzeTokenWithAddress = async (address: string) => {
    if (!address.trim()) {
      setAnalysisError("Please enter a token address");
      return;
    }

    setIsAnalyzing(true);
    setAnalysisError(null);
    setTokenAnalysis(null);

    try {
      const response = await fetch("/api/analyze-token", {
        method: "POST",
        headers: {
          "Content-Type": "application/json",
        },
        body: JSON.stringify({ 
          tokenAddress: address.trim(),
          chain: selectedChain,
        }),
      });

      if (!response.ok) {
        const errorData = await response.json();
        throw new Error(errorData.error || "Failed to analyze token");
      }

      const data = await response.json();
      setTokenAnalysis(data);
    } catch (error: any) {
      setAnalysisError(error.message || "Failed to analyze token");
    } finally {
      setIsAnalyzing(false);
    }
  };

  const analyzeToken = async () => {
    if (!tokenAddress.trim()) {
      setAnalysisError("Please enter a token address");
      return;
    }

    setIsAnalyzing(true);
    setAnalysisError(null);
    setTokenAnalysis(null);

    try {
      const response = await fetch("/api/analyze-token", {
        method: "POST",
        headers: {
          "Content-Type": "application/json",
        },
        body: JSON.stringify({ 
          tokenAddress: tokenAddress.trim(),
          chain: selectedChain,
        }),
      });

      if (!response.ok) {
        const errorData = await response.json();
        throw new Error(errorData.error || "Failed to analyze token");
      }

      const data = await response.json();
      setTokenAnalysis(data);
    } catch (error) {
      setAnalysisError(error instanceof Error ? error.message : "An error occurred");
    } finally {
      setIsAnalyzing(false);
    }
  };

  return (
    <div className="h-screen bg-background flex overflow-hidden relative">
      {/* Mobile Overlay */}
      {sidebarOpen && (
        <div 
          className="fixed inset-0 bg-black/50 z-40 md:hidden"
          onClick={() => setSidebarOpen(false)}
        />
      )}
      
      {/* Sidebar - Mobile: slides in/out, Desktop: always visible */}
      <motion.aside
        initial={false}
        animate={{ x: sidebarOpen ? 0 : -288 }}
        transition={{ duration: 0.3, ease: "easeInOut" }}
        className="fixed left-0 top-0 h-full w-72 border-r border-border/50 bg-sidebar flex-col noise-overlay overflow-hidden z-50 flex sidebar-desktop"
      >
        {/* Logo */}
        <div className="h-16 px-4 flex items-center border-b border-border/50">
          <button
            onClick={() => setSidebarOpen(!sidebarOpen)}
            className="flex items-center gap-3 cursor-pointer hover:opacity-80 transition-opacity"
            data-testid="button-toggle-sidebar"
          >
            <img src="/logo.png" alt="Astherafi" className="w-10 h-10 rounded-xl animate-spin-slow flex-shrink-0" />
            {sidebarOpen && (
              <div className="text-left">
                <h1 className="font-display font-bold text-lg tracking-tight whitespace-nowrap">Astherafi</h1>
                <p className="text-xs text-muted-foreground">Dashboard v1.0</p>
              </div>
            )}
          </button>
        </div>

        {/* Navigation */}
        <nav className="flex-1 p-4 space-y-1 relative z-10">
          {sidebarOpen && (
            <p className="text-xs font-medium text-muted-foreground uppercase tracking-wider px-3 mb-3">
              Tools
            </p>
          )}
          {menuItems.map((item, index) => (
            <motion.button
              key={item.id}
              data-testid={`menu-${item.id}`}
              initial={{ x: -10, opacity: 0 }}
              animate={{ x: 0, opacity: 1 }}
              transition={{ delay: index * 0.05, duration: 0.3 }}
              onClick={() => setActiveItem(item.id)}
              className={`w-full flex items-center gap-3 px-3 py-3 rounded-lg text-left transition-all duration-200 group ${
                activeItem === item.id
                  ? "bg-primary/10 text-primary glow-cyan"
                  : "text-muted-foreground hover:text-foreground hover:bg-muted/50"
              }`}
            >
              <span
                className={`${
                  activeItem === item.id ? "text-primary" : "text-muted-foreground group-hover:text-foreground"
                } transition-colors`}
              >
                {item.icon}
              </span>
              {sidebarOpen && (
                <>
                  <span className="font-medium text-sm">{item.label}</span>
                  {activeItem === item.id && (
                    <ChevronRight className="w-4 h-4 ml-auto text-primary" />
                  )}
                </>
              )}
            </motion.button>
          ))}
        </nav>

        {/* User Section */}
        <div className="p-4 border-t border-border/50 relative z-10">
          <div className={`flex items-center gap-3 p-3 rounded-lg bg-muted/30 ${!sidebarOpen ? 'justify-center' : ''}`}>
            <div className="w-9 h-9 rounded-full cyber-gradient flex items-center justify-center flex-shrink-0">
              <User className="w-4 h-4 text-background" />
            </div>
            {sidebarOpen && (
              <>
                <div className="flex-1 min-w-0">
                  <p className="text-sm font-medium truncate">0x7a2...f9d4</p>
                  <p className="text-xs text-muted-foreground">Connected</p>
                </div>
                <div className="w-2 h-2 rounded-full bg-green-500 animate-pulse" />
              </>
            )}
          </div>
        </div>
      </motion.aside>

      {/* Main Content */}
      <div className="flex-1 flex flex-col h-full overflow-hidden">
        {/* Header */}
        <motion.header
          initial={{ y: -10, opacity: 0 }}
          animate={{ y: 0, opacity: 1 }}
          transition={{ duration: 0.4, delay: 0.1 }}
          className="h-16 border-b border-border/50 flex items-center justify-between px-6 bg-sidebar flex-shrink-0"
        >
          <Button
            variant="ghost"
            size="icon"
            className="md:hidden text-muted-foreground hover:text-foreground"
            onClick={() => setSidebarOpen(true)}
          >
            <Menu className="w-5 h-5" />
          </Button>
          <div className="hidden md:block"></div>
          <div className="flex items-center gap-2">
            <Button
              data-testid="button-notifications"
              variant="ghost"
              size="icon"
              className="text-muted-foreground hover:text-foreground relative"
            >
              <Bell className="w-5 h-5" />
              <span className="absolute top-1 right-1 w-2 h-2 bg-primary rounded-full" />
            </Button>
            <Button
              data-testid="button-settings"
              variant="ghost"
              size="icon"
              className="text-muted-foreground hover:text-foreground"
            >
              <Settings className="w-5 h-5" />
            </Button>
          </div>
        </motion.header>

        {/* Content Area */}
        <main className="flex-1 p-6 overflow-y-auto">
          {/* Page Title - hidden on home */}
          {activeItem !== "home" && (
            <motion.div
              key={activeItem}
              initial={{ y: 10, opacity: 0 }}
              animate={{ y: 0, opacity: 1 }}
              transition={{ duration: 0.3 }}
              className="mb-8"
            >
              <h2 className="text-3xl font-display font-bold text-glow-cyan">
                {activeMenu?.label}
              </h2>
              <p className="text-muted-foreground mt-1">{activeMenu?.description}</p>
            </motion.div>
          )}

          {/* Tool Content Area */}
          <motion.div
            key={`content-${activeItem}`}
            initial={{ opacity: 0, scale: 0.98 }}
            animate={{ opacity: 1, scale: 1 }}
            transition={{ duration: 0.3 }}
            className="rounded-2xl bg-card border border-border/50 p-8 min-h-[400px] relative overflow-hidden"
          >
            {/* Gradient accent */}
            <div className="absolute top-0 right-0 w-96 h-96 bg-gradient-to-bl from-primary/5 to-transparent rounded-full blur-3xl pointer-events-none" />
            <div className="absolute bottom-0 left-0 w-64 h-64 bg-gradient-to-tr from-secondary/5 to-transparent rounded-full blur-3xl pointer-events-none" />

            <div className="relative z-10">
              {activeItem !== "home" && (
                <div className="flex items-center gap-3 mb-6">
                  <div className="w-12 h-12 rounded-xl cyber-gradient flex items-center justify-center glow-cyan">
                    {activeMenu?.icon}
                  </div>
                  <div>
                    <h3 className="font-display font-semibold text-xl">{activeMenu?.label}</h3>
                    <p className="text-sm text-muted-foreground">Ready to use</p>
                  </div>
                </div>
              )}

              {/* Placeholder content based on selected tool */}
              {activeItem === "home" && (
                <div className="flex flex-col items-center justify-center text-center py-12 space-y-6">
                  {/* Social Links - Hidden */}
                  <div className="hidden flex gap-4 mb-10">
                    <a href="https://www.dextools.io/app/en/" target="_blank" rel="noopener noreferrer" className="social-icon-link w-12 h-12 rounded-full bg-[#00242b] flex items-center justify-center hover:bg-[#003540] transition-colors">
                      <svg width="20" height="20" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                        <path d="M12 4L6 8v8l6 4 6-4V8l-6-4z" stroke="#30d0f2" strokeWidth="1.5" fill="none"/>
                        <path d="M12 12L6 8M12 12l6-4M12 12v8" stroke="#30d0f2" strokeWidth="1.5"/>
                        <circle cx="12" cy="12" r="2" fill="#30d0f2"/>
                      </svg>
                    </a>
                    <a href="https://coinmarketcap.com/currencies/" target="_blank" rel="noopener noreferrer" className="social-icon-link w-12 h-12 rounded-full bg-[#00242b] flex items-center justify-center hover:bg-[#003540] transition-colors">
                      <svg width="20" height="20" viewBox="0 0 24 24" fill="#30d0f2"><circle cx="12" cy="12" r="10" stroke="#30d0f2" strokeWidth="2" fill="none"/><path d="M12 6v6l4 2" stroke="#30d0f2" strokeWidth="2" fill="none"/></svg>
                    </a>
                    <a href="https://www.coingecko.com/en/coins/" target="_blank" rel="noopener noreferrer" className="social-icon-link w-12 h-12 rounded-full bg-[#00242b] flex items-center justify-center hover:bg-[#003540] transition-colors">
                      <svg width="20" height="20" viewBox="0 0 24 24" fill="#30d0f2"><circle cx="12" cy="12" r="8" fill="#30d0f2"/><circle cx="14" cy="10" r="2" fill="#00242b"/></svg>
                    </a>
                    <a href="https://www.linkedin.com/company/" target="_blank" rel="noopener noreferrer" className="social-icon-link w-12 h-12 rounded-full bg-[#00242b] flex items-center justify-center hover:bg-[#003540] transition-colors">
                      <svg width="16" height="16" viewBox="0 0 24 24" fill="#30d0f2"><path d="M19 0h-14c-2.761 0-5 2.239-5 5v14c0 2.761 2.239 5 5 5h14c2.762 0 5-2.239 5-5v-14c0-2.761-2.238-5-5-5zm-11 19h-3v-11h3v11zm-1.5-12.268c-.966 0-1.75-.79-1.75-1.764s.784-1.764 1.75-1.764 1.75.79 1.75 1.764-.783 1.764-1.75 1.764zm13.5 12.268h-3v-5.604c0-3.368-4-3.113-4 0v5.604h-3v-11h3v1.765c1.396-2.586 7-2.777 7 2.476v6.759z"/></svg>
                    </a>
                    <a href="https://medium.com/" target="_blank" rel="noopener noreferrer" className="social-icon-link w-12 h-12 rounded-full bg-[#00242b] flex items-center justify-center hover:bg-[#003540] transition-colors">
                      <svg width="20" height="20" viewBox="0 0 24 24" fill="#30d0f2"><ellipse cx="6" cy="12" rx="4" ry="6" fill="#30d0f2"/><ellipse cx="14" cy="12" rx="2.5" ry="5.5" fill="#30d0f2"/><ellipse cx="20" cy="12" rx="1" ry="5" fill="#30d0f2"/></svg>
                    </a>
                    <a href="https://www.youtube.com/@" target="_blank" rel="noopener noreferrer" className="social-icon-link w-12 h-12 rounded-full bg-[#00242b] flex items-center justify-center hover:bg-[#003540] transition-colors">
                      <svg width="20" height="20" viewBox="0 0 24 24" fill="#30d0f2"><path d="M19.615 3.184c-3.604-.246-11.631-.245-15.23 0-3.897.266-4.356 2.62-4.385 8.816.029 6.185.484 8.549 4.385 8.816 3.6.245 11.626.246 15.23 0 3.897-.266 4.356-2.62 4.385-8.816-.029-6.185-.484-8.549-4.385-8.816zm-10.615 12.816v-8l8 3.993-8 4.007z"/></svg>
                    </a>
                    <a href="https://x.com/astherafihq" target="_blank" rel="noopener noreferrer" className="social-icon-link w-12 h-12 rounded-full bg-[#00242b] flex items-center justify-center hover:bg-[#003540] transition-colors">
                      <svg width="18" height="18" viewBox="0 0 24 24" fill="#30d0f2"><path d="M18.244 2.25h3.308l-7.227 8.26 8.502 11.24H16.17l-5.214-6.817L4.99 21.75H1.68l7.73-8.835L1.254 2.25H8.08l4.713 6.231zm-1.161 17.52h1.833L7.084 4.126H5.117z"/></svg>
                    </a>
                    <a href="https://t.me/AstheraFi_Official" target="_blank" rel="noopener noreferrer" className="social-icon-link w-12 h-12 rounded-full bg-[#00242b] flex items-center justify-center hover:bg-[#003540] transition-colors">
                      <svg width="20" height="20" viewBox="0 0 24 24" fill="#30d0f2"><path d="M11.944 0A12 12 0 0 0 0 12a12 12 0 0 0 12 12 12 12 0 0 0 12-12A12 12 0 0 0 12 0a12 12 0 0 0-.056 0zm4.962 7.224c.1-.002.321.023.465.14a.506.506 0 0 1 .171.325c.016.093.036.306.02.472-.18 1.898-.962 6.502-1.36 8.627-.168.9-.499 1.201-.82 1.23-.696.065-1.225-.46-1.9-.902-1.056-.693-1.653-1.124-2.678-1.8-1.185-.78-.417-1.21.258-1.91.177-.184 3.247-2.977 3.307-3.23.007-.032.014-.15-.056-.212s-.174-.041-.249-.024c-.106.024-1.793 1.14-5.061 3.345-.48.33-.913.49-1.302.48-.428-.008-1.252-.241-1.865-.44-.752-.245-1.349-.374-1.297-.789.027-.216.325-.437.893-.663 3.498-1.524 5.83-2.529 6.998-3.014 3.332-1.386 4.025-1.627 4.476-1.635z"/></svg>
                    </a>
                  </div>

                  <h2 className="font-display font-bold text-4xl text-foreground tracking-tight">
                    Astherafi
                  </h2>
                  <h3 
                    className="font-display font-bold text-3xl bg-clip-text text-transparent"
                    style={{ backgroundImage: "linear-gradient(135deg, rgb(44, 206, 240), rgb(48, 208, 242), rgb(93, 92, 222))" }}
                  >
                    AI-Powered On-Chain Alpha Engine
                  </h3>
                  <p className="text-muted-foreground text-lg max-w-2xl leading-relaxed">
                    Astherafi here to help you navigate the crypto market by offering a suite of Sentiment, Technical, and On-Chain tools powered by AI.
                  </p>
                  <div className="flex gap-4 mt-4">
                    <Button 
                      data-testid="button-buy-now"
                      className="px-8 h-12 font-semibold text-base text-white border-0 rounded-lg shadow-lg"
                      style={{ background: "linear-gradient(135deg, rgb(48, 208, 242), rgb(93, 92, 222))" }}
                    >
                      Buy Now
                    </Button>
                    <Button 
                      data-testid="button-view-chart"
                      className="px-8 h-12 font-semibold text-base text-white border-0 rounded-lg shadow-lg"
                      style={{ background: "linear-gradient(135deg, rgb(48, 208, 242), rgb(93, 92, 222))" }}
                    >
                      View Chart
                    </Button>
                  </div>

                  <div className="flex w-full max-w-2xl mt-8">
                    <Input
                      data-testid="input-home-search"
                      placeholder="Search for tokens, pairs, or addresses..."
                      className="flex-1 h-12 bg-[hsl(200,30%,12%)] rounded-l-full rounded-r-none text-foreground placeholder:text-muted-foreground px-6 focus:outline-none focus:ring-0"
                      style={{ border: "2px solid rgb(48, 208, 242)" }}
                      value={homeSearchAddress}
                      onChange={(e) => setHomeSearchAddress(e.target.value)}
                      onKeyDown={(e) => e.key === "Enter" && handleHomeSearch()}
                    />
                    <Button
                      data-testid="button-home-search"
                      className="h-12 px-8 rounded-l-none rounded-r-full font-semibold text-white border-0"
                      style={{ background: "linear-gradient(135deg, rgb(48, 208, 242), rgb(93, 92, 222))" }}
                      onClick={handleHomeSearch}
                    >
                      Search
                    </Button>
                  </div>
                </div>
              )}

              {activeItem === "token-analyzer" && (
                <div className="space-y-4">
                  {/* Chain Selector + Input */}
                  <div className="flex gap-3">
                    <select
                      data-testid="select-chain"
                      value={selectedChain}
                      onChange={(e) => setSelectedChain(e.target.value)}
                      className="bg-muted/30 border border-border/50 rounded-lg px-4 h-12 text-sm font-medium focus:outline-none focus:ring-2 focus:ring-primary/50"
                    >
                      {CHAINS.map((chain) => (
                        <option key={chain.id} value={chain.id}>
                          {chain.icon} {chain.name}
                        </option>
                      ))}
                    </select>
                    <Input
                      data-testid="input-token-address"
                      placeholder="Enter token contract address"
                      className="flex-1 bg-muted/30 border-border/50 h-12 font-mono text-sm"
                      value={tokenAddress}
                      onChange={(e) => setTokenAddress(e.target.value)}
                      onKeyDown={(e) => e.key === "Enter" && analyzeToken()}
                    />
                    <Button 
                      data-testid="button-analyze" 
                      className="cyber-gradient text-background font-semibold px-6 h-12 glow-cyan"
                      onClick={analyzeToken}
                      disabled={isAnalyzing}
                    >
                      {isAnalyzing ? (
                        <>
                          <Loader2 className="w-4 h-4 mr-2 animate-spin" />
                          Analyzing...
                        </>
                      ) : (
                        "Analyze"
                      )}
                    </Button>
                  </div>

                  {analysisError && (
                    <div className="p-4 rounded-lg bg-red-500/10 border border-red-500/30 flex items-center gap-3">
                      <XCircle className="w-5 h-5 text-red-400 flex-shrink-0" />
                      <p className="text-sm text-red-300">{analysisError}</p>
                    </div>
                  )}

                  {tokenAnalysis && (
                    <div className="space-y-6 mt-6">
                      {/* Token Header */}
                      <div className="p-4 rounded-lg bg-gradient-to-r from-primary/10 to-secondary/10 border border-primary/20">
                        <div className="flex items-center justify-between">
                          <div className="flex items-center gap-3">
                            {tokenAnalysis.logo ? (
                              <img 
                                src={tokenAnalysis.logo} 
                                alt={tokenAnalysis.symbol}
                                className="w-12 h-12 rounded-full object-cover bg-muted"
                                onError={(e) => {
                                  (e.target as HTMLImageElement).style.display = 'none';
                                  (e.target as HTMLImageElement).nextElementSibling?.classList.remove('hidden');
                                }}
                              />
                            ) : null}
                            <div className={`w-12 h-12 rounded-full cyber-gradient flex items-center justify-center text-background font-bold text-lg ${tokenAnalysis.logo ? 'hidden' : ''}`}>
                              {tokenAnalysis.symbol?.[0] || "?"}
                            </div>
                            <div>
                              <h4 className="font-display font-semibold text-lg">{tokenAnalysis.name}</h4>
                              <p className="text-sm text-muted-foreground font-mono">{tokenAnalysis.symbol} • {tokenAnalysis.chain}</p>
                            </div>
                          </div>
                          {tokenAnalysis.simulation?.isHoneypot && (
                            <div className="px-3 py-1 rounded-full bg-red-500/20 border border-red-500/50 text-red-400 text-sm font-medium flex items-center gap-2">
                              <XCircle className="w-4 h-4" />
                              HONEYPOT
                            </div>
                          )}
                          {!tokenAnalysis.simulation?.isHoneypot && (
                            <div className="px-3 py-1 rounded-full bg-green-500/20 border border-green-500/50 text-green-400 text-sm font-medium flex items-center gap-2">
                              <CheckCircle className="w-4 h-4" />
                              SAFE
                            </div>
                          )}
                        </div>
                        
                        {/* Social Links */}
                        {tokenAnalysis.socials && Object.keys(tokenAnalysis.socials).length > 0 && (
                          <div className="flex items-center gap-3 mt-4 pt-4 border-t border-border/30">
                            <span className="text-xs text-muted-foreground uppercase tracking-wider">Links:</span>
                            {tokenAnalysis.socials.website && (
                              <a 
                                href={tokenAnalysis.socials.website.startsWith('http') ? tokenAnalysis.socials.website : `https://${tokenAnalysis.socials.website}`}
                                target="_blank"
                                rel="noopener noreferrer"
                                className="flex items-center gap-1.5 px-2.5 py-1.5 rounded-md bg-muted/30 hover:bg-muted/50 text-sm text-muted-foreground hover:text-foreground transition-colors"
                                data-testid="link-website"
                              >
                                <Globe className="w-4 h-4" />
                                Website
                              </a>
                            )}
                            {tokenAnalysis.socials.twitter && (
                              <a 
                                href={tokenAnalysis.socials.twitter.startsWith('http') ? tokenAnalysis.socials.twitter : `https://twitter.com/${tokenAnalysis.socials.twitter.replace('@', '')}`}
                                target="_blank"
                                rel="noopener noreferrer"
                                className="flex items-center gap-1.5 px-2.5 py-1.5 rounded-md bg-muted/30 hover:bg-muted/50 text-sm text-muted-foreground hover:text-foreground transition-colors"
                                data-testid="link-twitter"
                              >
                                <Twitter className="w-4 h-4" />
                                Twitter
                              </a>
                            )}
                            {tokenAnalysis.socials.telegram && (
                              <a 
                                href={tokenAnalysis.socials.telegram.startsWith('http') ? tokenAnalysis.socials.telegram : `https://t.me/${tokenAnalysis.socials.telegram}`}
                                target="_blank"
                                rel="noopener noreferrer"
                                className="flex items-center gap-1.5 px-2.5 py-1.5 rounded-md bg-muted/30 hover:bg-muted/50 text-sm text-muted-foreground hover:text-foreground transition-colors"
                                data-testid="link-telegram"
                              >
                                <MessageCircle className="w-4 h-4" />
                                Telegram
                              </a>
                            )}
                            {tokenAnalysis.socials.discord && (
                              <a 
                                href={tokenAnalysis.socials.discord.startsWith('http') ? tokenAnalysis.socials.discord : `https://discord.gg/${tokenAnalysis.socials.discord}`}
                                target="_blank"
                                rel="noopener noreferrer"
                                className="flex items-center gap-1.5 px-2.5 py-1.5 rounded-md bg-muted/30 hover:bg-muted/50 text-sm text-muted-foreground hover:text-foreground transition-colors"
                                data-testid="link-discord"
                              >
                                <ExternalLink className="w-4 h-4" />
                                Discord
                              </a>
                            )}
                          </div>
                        )}
                      </div>

                      {/* Simulation Results */}
                      <div className="p-5 rounded-lg bg-[hsl(40,30%,15%)] border border-[hsl(40,30%,25%)]">
                        <h5 className="text-[hsl(40,30%,60%)] font-display font-semibold text-sm uppercase tracking-wider mb-4">
                          Simulation Results
                        </h5>
                        <div className="grid grid-cols-3 gap-x-8 gap-y-4">
                          <div>
                            <p className="text-[hsl(40,30%,50%)] text-xs uppercase tracking-wider">Buy Tax</p>
                            <p className="text-green-400 text-xl font-display font-bold">{tokenAnalysis.simulation?.buyTax || "0.0%"}</p>
                          </div>
                          <div>
                            <p className="text-[hsl(40,30%,50%)] text-xs uppercase tracking-wider">Buy Gas</p>
                            <p className="text-green-400 text-xl font-display font-bold">{tokenAnalysis.simulation?.buyGas || "N/A"}</p>
                          </div>
                          <div>
                            <p className="text-[hsl(40,30%,50%)] text-xs uppercase tracking-wider">Buy Limit</p>
                            <p className="text-foreground text-xl font-display font-bold">{tokenAnalysis.simulation?.buyLimit || "NONE DETECTED"}</p>
                          </div>
                          <div>
                            <p className="text-[hsl(40,30%,50%)] text-xs uppercase tracking-wider">Sell Tax</p>
                            <p className="text-green-400 text-xl font-display font-bold">{tokenAnalysis.simulation?.sellTax || "0.0%"}</p>
                          </div>
                          <div>
                            <p className="text-[hsl(40,30%,50%)] text-xs uppercase tracking-wider">Sell Gas</p>
                            <p className="text-green-400 text-xl font-display font-bold">{tokenAnalysis.simulation?.sellGas || "N/A"}</p>
                          </div>
                          <div>
                            <p className="text-[hsl(40,30%,50%)] text-xs uppercase tracking-wider">Sell Limit</p>
                            <p className="text-foreground text-xl font-display font-bold">{tokenAnalysis.simulation?.sellLimit || "NONE DETECTED"}</p>
                          </div>
                          <div>
                            <p className="text-[hsl(40,30%,50%)] text-xs uppercase tracking-wider">Transfer Tax</p>
                            <p className="text-green-400 text-xl font-display font-bold">{tokenAnalysis.simulation?.transferTax || "0.0%"}</p>
                          </div>
                          <div>
                            <p className="text-[hsl(40,30%,50%)] text-xs uppercase tracking-wider">Source Code</p>
                            <p className="text-foreground text-sm font-display font-bold flex items-center gap-2">
                              {tokenAnalysis.simulation?.sourceCode === "OPEN SOURCE" ? (
                                <>
                                  <CheckCircle className="w-4 h-4 text-green-400" />
                                  OPEN SOURCE
                                </>
                              ) : (
                                tokenAnalysis.simulation?.sourceCode || "N/A"
                              )}
                            </p>
                          </div>
                        </div>
                      </div>

                      {/* Holder Analysis */}
                      <div className="p-5 rounded-lg bg-[hsl(40,30%,15%)] border border-[hsl(40,30%,25%)]">
                        <div className="flex items-center gap-4 mb-4">
                          <h5 className="text-[hsl(40,30%,60%)] font-display font-semibold text-sm uppercase tracking-wider">
                            Recent Holder Analysis
                          </h5>
                          <div className="relative group">
                            <span className="text-[hsl(40,30%,50%)] text-xs underline cursor-help">What's this?</span>
                            <div className="absolute left-0 top-full mt-2 w-80 p-3 rounded-lg bg-background border border-border shadow-xl opacity-0 invisible group-hover:opacity-100 group-hover:visible transition-all duration-200 z-50">
                              <p className="text-sm text-foreground leading-relaxed">
                                Astherafi analyses the last buyers of the token to check if they can sell or not. Each wallet is checked multiple times at different intervals.
                              </p>
                              <p className="text-sm text-muted-foreground mt-2 leading-relaxed">
                                <strong className="text-foreground">Siphoned</strong> means the user's tokens disappeared without them sending.
                              </p>
                            </div>
                          </div>
                        </div>
                        <div className="grid grid-cols-3 gap-x-8 gap-y-4">
                          <div>
                            <p className="text-[hsl(40,30%,50%)] text-xs uppercase tracking-wider">Holders Analysed</p>
                            <p className="text-green-400 text-xl font-display font-bold">{tokenAnalysis.holders?.total?.toLocaleString() || "0"}</p>
                          </div>
                          <div>
                            <p className="text-[hsl(40,30%,50%)] text-xs uppercase tracking-wider">Can Sell</p>
                            <p className="text-green-400 text-xl font-display font-bold">{tokenAnalysis.holders?.canSell?.toLocaleString() || "0"}</p>
                          </div>
                          <div>
                            <p className="text-[hsl(40,30%,50%)] text-xs uppercase tracking-wider">Can't Sell</p>
                            <p className="text-foreground text-xl font-display font-bold">{tokenAnalysis.holders?.cantSell || "0"}</p>
                          </div>
                          <div>
                            <p className="text-[hsl(40,30%,50%)] text-xs uppercase tracking-wider">Siphoned</p>
                            <p className="text-foreground text-xl font-display font-bold">{tokenAnalysis.holders?.siphoned || "0"}</p>
                          </div>
                          <div>
                            <p className="text-[hsl(40,30%,50%)] text-xs uppercase tracking-wider">Average Tax</p>
                            <p className="text-green-400 text-xl font-display font-bold">{tokenAnalysis.holders?.averageTax || "0.0%"}</p>
                          </div>
                          <div>
                            <p className="text-[hsl(40,30%,50%)] text-xs uppercase tracking-wider">Highest Tax</p>
                            <p className="text-foreground text-xl font-display font-bold">{tokenAnalysis.holders?.highestTax || "0.0%"}</p>
                          </div>
                          <div>
                            <p className="text-[hsl(40,30%,50%)] text-xs uppercase tracking-wider">Average Gas</p>
                            <p className="text-green-400 text-xl font-display font-bold">{tokenAnalysis.holders?.averageGas || "N/A"}</p>
                          </div>
                        </div>
                      </div>

                      {/* Sentiment Analysis */}
                      <div className="p-5 rounded-lg bg-[hsl(270,40%,12%)] border border-[hsl(270,40%,25%)]">
                        <h5 className="text-[hsl(270,40%,70%)] font-display font-semibold text-sm uppercase tracking-wider mb-4">
                          Sentiment Analysis
                        </h5>
                        <div className="grid grid-cols-3 gap-x-8 gap-y-4">
                          <div>
                            <p className="text-[hsl(270,30%,50%)] text-xs uppercase tracking-wider">Overall Sentiment</p>
                            <p className={`text-xl font-display font-bold ${
                              tokenAnalysis.sentiment?.overall === "Bullish" ? "text-green-400" :
                              tokenAnalysis.sentiment?.overall === "Bearish" ? "text-red-400" : "text-foreground"
                            }`}>
                              {tokenAnalysis.sentiment?.overall || "N/A"}
                            </p>
                          </div>
                          <div>
                            <p className="text-[hsl(270,30%,50%)] text-xs uppercase tracking-wider">Buy Pressure (24h)</p>
                            <p className="text-[hsl(var(--cyber-purple))] text-xl font-display font-bold">
                              {tokenAnalysis.sentiment?.buyPressure || "N/A"}
                            </p>
                          </div>
                          <div>
                            <p className="text-[hsl(270,30%,50%)] text-xs uppercase tracking-wider">Total Txns (24h)</p>
                            <p className="text-foreground text-xl font-display font-bold">
                              {tokenAnalysis.sentiment?.totalTxns24h?.toLocaleString() || "0"}
                            </p>
                          </div>
                          <div>
                            <p className="text-[hsl(270,30%,50%)] text-xs uppercase tracking-wider">Buys (24h)</p>
                            <p className="text-green-400 text-xl font-display font-bold">
                              {tokenAnalysis.sentiment?.buys24h?.toLocaleString() || "0"}
                            </p>
                          </div>
                          <div>
                            <p className="text-[hsl(270,30%,50%)] text-xs uppercase tracking-wider">Sells (24h)</p>
                            <p className="text-red-400 text-xl font-display font-bold">
                              {tokenAnalysis.sentiment?.sells24h?.toLocaleString() || "0"}
                            </p>
                          </div>
                        </div>
                      </div>

                      {/* Technical Analysis */}
                      <div className="p-5 rounded-lg bg-[hsl(200,50%,12%)] border border-[hsl(200,50%,25%)]">
                        <h5 className="text-[hsl(200,50%,70%)] font-display font-semibold text-sm uppercase tracking-wider mb-4">
                          Technical Analysis
                        </h5>
                        <div className="grid grid-cols-4 gap-x-8 gap-y-4">
                          <div>
                            <p className="text-[hsl(200,30%,50%)] text-xs uppercase tracking-wider">Price USD</p>
                            <p className="text-foreground text-xl font-display font-bold">
                              ${tokenAnalysis.technical?.priceUsd ? parseFloat(tokenAnalysis.technical.priceUsd).toFixed(6) : "N/A"}
                            </p>
                          </div>
                          <div>
                            <p className="text-[hsl(200,30%,50%)] text-xs uppercase tracking-wider">5m Change</p>
                            <p className={`text-xl font-display font-bold ${
                              (tokenAnalysis.technical?.priceChange5m || 0) >= 0 ? "text-green-400" : "text-red-400"
                            }`}>
                              {tokenAnalysis.technical?.priceChange5m !== undefined 
                                ? `${tokenAnalysis.technical.priceChange5m >= 0 ? "+" : ""}${tokenAnalysis.technical.priceChange5m.toFixed(2)}%`
                                : "N/A"}
                            </p>
                          </div>
                          <div>
                            <p className="text-[hsl(200,30%,50%)] text-xs uppercase tracking-wider">1h Change</p>
                            <p className={`text-xl font-display font-bold ${
                              (tokenAnalysis.technical?.priceChange1h || 0) >= 0 ? "text-green-400" : "text-red-400"
                            }`}>
                              {tokenAnalysis.technical?.priceChange1h !== undefined 
                                ? `${tokenAnalysis.technical.priceChange1h >= 0 ? "+" : ""}${tokenAnalysis.technical.priceChange1h.toFixed(2)}%`
                                : "N/A"}
                            </p>
                          </div>
                          <div>
                            <p className="text-[hsl(200,30%,50%)] text-xs uppercase tracking-wider">24h Change</p>
                            <p className={`text-xl font-display font-bold ${
                              (tokenAnalysis.technical?.priceChange24h || 0) >= 0 ? "text-green-400" : "text-red-400"
                            }`}>
                              {tokenAnalysis.technical?.priceChange24h !== undefined 
                                ? `${tokenAnalysis.technical.priceChange24h >= 0 ? "+" : ""}${tokenAnalysis.technical.priceChange24h.toFixed(2)}%`
                                : "N/A"}
                            </p>
                          </div>
                          <div>
                            <p className="text-[hsl(200,30%,50%)] text-xs uppercase tracking-wider">Volume (24h)</p>
                            <p className="text-[hsl(var(--cyber-cyan))] text-xl font-display font-bold">
                              ${tokenAnalysis.technical?.volume24h 
                                ? tokenAnalysis.technical.volume24h >= 1000000 
                                  ? `${(tokenAnalysis.technical.volume24h / 1000000).toFixed(2)}M`
                                  : tokenAnalysis.technical.volume24h >= 1000
                                  ? `${(tokenAnalysis.technical.volume24h / 1000).toFixed(2)}K`
                                  : tokenAnalysis.technical.volume24h.toFixed(2)
                                : "N/A"}
                            </p>
                          </div>
                          <div>
                            <p className="text-[hsl(200,30%,50%)] text-xs uppercase tracking-wider">Liquidity</p>
                            <p className="text-[hsl(var(--cyber-cyan))] text-xl font-display font-bold">
                              ${tokenAnalysis.technical?.liquidity 
                                ? tokenAnalysis.technical.liquidity >= 1000000 
                                  ? `${(tokenAnalysis.technical.liquidity / 1000000).toFixed(2)}M`
                                  : tokenAnalysis.technical.liquidity >= 1000
                                  ? `${(tokenAnalysis.technical.liquidity / 1000).toFixed(2)}K`
                                  : tokenAnalysis.technical.liquidity.toFixed(2)
                                : "N/A"}
                            </p>
                          </div>
                          <div>
                            <p className="text-[hsl(200,30%,50%)] text-xs uppercase tracking-wider">Market Cap</p>
                            <p className="text-foreground text-xl font-display font-bold">
                              ${tokenAnalysis.technical?.marketCap 
                                ? tokenAnalysis.technical.marketCap >= 1000000000
                                  ? `${(tokenAnalysis.technical.marketCap / 1000000000).toFixed(2)}B`
                                  : tokenAnalysis.technical.marketCap >= 1000000 
                                  ? `${(tokenAnalysis.technical.marketCap / 1000000).toFixed(2)}M`
                                  : tokenAnalysis.technical.marketCap >= 1000
                                  ? `${(tokenAnalysis.technical.marketCap / 1000).toFixed(2)}K`
                                  : tokenAnalysis.technical.marketCap.toFixed(2)
                                : "N/A"}
                            </p>
                          </div>
                          <div>
                            <p className="text-[hsl(200,30%,50%)] text-xs uppercase tracking-wider">6h Change</p>
                            <p className={`text-xl font-display font-bold ${
                              (tokenAnalysis.technical?.priceChange6h || 0) >= 0 ? "text-green-400" : "text-red-400"
                            }`}>
                              {tokenAnalysis.technical?.priceChange6h !== undefined 
                                ? `${tokenAnalysis.technical.priceChange6h >= 0 ? "+" : ""}${tokenAnalysis.technical.priceChange6h.toFixed(2)}%`
                                : "N/A"}
                            </p>
                          </div>
                        </div>
                      </div>
                    </div>
                  )}
                </div>
              )}

              {activeItem === "bsc-bundler" && (
                <div className="space-y-4">
                  <p className="text-muted-foreground">Add transactions to bundle and execute them atomically on BSC.</p>
                  <div className="grid grid-cols-2 gap-4">
                    <div className="p-4 rounded-lg bg-muted/20 border border-dashed border-border/50 text-center py-12">
                      <Layers className="w-8 h-8 mx-auto text-muted-foreground mb-2" />
                      <p className="text-sm text-muted-foreground">Drop transactions here</p>
                    </div>
                    <div className="p-4 rounded-lg bg-muted/20 border border-border/30">
                      <p className="text-sm font-medium mb-3">Bundle Summary</p>
                      <div className="space-y-2 text-sm text-muted-foreground">
                        <p>Transactions: 0</p>
                        <p>Estimated Gas: --</p>
                        <p>Total Value: --</p>
                      </div>
                    </div>
                  </div>
                </div>
              )}

              {activeItem === "token-generator" && (
                <div className="space-y-4">
                  <div className="grid grid-cols-2 gap-4">
                    <Input data-testid="input-token-name" placeholder="Token Name" className="bg-muted/30 border-border/50 h-11" />
                    <Input data-testid="input-token-symbol" placeholder="Symbol" className="bg-muted/30 border-border/50 h-11" />
                    <Input data-testid="input-token-supply" placeholder="Total Supply" className="bg-muted/30 border-border/50 h-11" />
                    <Input data-testid="input-token-decimals" placeholder="Decimals" className="bg-muted/30 border-border/50 h-11" />
                  </div>
                  <Button data-testid="button-generate" className="cyber-gradient text-background font-semibold w-full h-12 glow-cyan mt-4">
                    Generate Token
                  </Button>
                </div>
              )}

              {activeItem === "password-score" && (
                <div className="space-y-4">
                  <Input
                    data-testid="input-password"
                    type="password"
                    placeholder="Enter password to check strength"
                    className="bg-muted/30 border-border/50 h-12"
                  />
                  <div className="p-4 rounded-lg bg-muted/20 border border-border/30">
                    <div className="flex items-center justify-between mb-2">
                      <p className="text-sm text-muted-foreground">Strength Score</p>
                      <p className="text-sm font-mono">--/100</p>
                    </div>
                    <div className="h-2 bg-muted rounded-full overflow-hidden">
                      <div className="h-full w-0 cyber-gradient rounded-full transition-all" />
                    </div>
                  </div>
                </div>
              )}

              {activeItem === "staking" && (
                <div className="space-y-4">
                  <div className="grid grid-cols-3 gap-4">
                    <div className="p-4 rounded-lg bg-muted/20 border border-border/30 text-center">
                      <p className="text-sm text-muted-foreground">APY</p>
                      <p className="text-2xl font-display font-bold text-primary mt-1">12.5%</p>
                    </div>
                    <div className="p-4 rounded-lg bg-muted/20 border border-border/30 text-center">
                      <p className="text-sm text-muted-foreground">Your Stake</p>
                      <p className="text-2xl font-display font-bold mt-1">0.00</p>
                    </div>
                    <div className="p-4 rounded-lg bg-muted/20 border border-border/30 text-center">
                      <p className="text-sm text-muted-foreground">Rewards</p>
                      <p className="text-2xl font-display font-bold text-green-400 mt-1">0.00</p>
                    </div>
                  </div>
                  <div className="flex gap-3 mt-6">
                    <Button data-testid="button-stake" className="flex-1 cyber-gradient text-background font-semibold h-12 glow-cyan">
                      Stake
                    </Button>
                    <Button data-testid="button-unstake" variant="outline" className="flex-1 h-12 border-primary/30 hover:bg-primary/10">
                      Unstake
                    </Button>
                  </div>
                </div>
              )}
            </div>
          </motion.div>
        </main>
      </div>
    </div>
  );
}
