import type { Express } from "express";
import { createServer, type Server } from "http";
import { storage } from "./storage";

const CHAIN_IDS: Record<string, string> = {
  ethereum: "1",
  bsc: "56",
  polygon: "137",
  arbitrum: "42161",
  base: "8453",
  avalanche: "43114",
  fantom: "250",
  optimism: "10",
};

const CHAIN_NAMES: Record<string, string> = {
  "1": "Ethereum",
  "56": "BSC",
  "137": "Polygon",
  "42161": "Arbitrum",
  "8453": "Base",
  "43114": "Avalanche",
  "250": "Fantom",
  "10": "Optimism",
};

const TRUST_WALLET_CHAIN_NAMES: Record<string, string> = {
  "1": "ethereum",
  "56": "smartchain",
  "137": "polygon",
  "42161": "arbitrum",
  "8453": "base",
  "43114": "avalanchec",
  "250": "fantom",
  "10": "optimism",
};

function getTokenLogo(address: string, chainId: string): string {
  const chainName = TRUST_WALLET_CHAIN_NAMES[chainId] || "ethereum";
  return `https://raw.githubusercontent.com/trustwallet/assets/master/blockchains/${chainName}/assets/${address}/logo.png`;
}

const DEXSCREENER_CHAIN_IDS: Record<string, string> = {
  "1": "ethereum",
  "56": "bsc",
  "137": "polygon",
  "42161": "arbitrum",
  "8453": "base",
  "43114": "avalanche",
  "250": "fantom",
  "10": "optimism",
};

async function fetchDexScreenerData(tokenAddress: string, chainId: string) {
  try {
    const dexChain = DEXSCREENER_CHAIN_IDS[chainId] || "bsc";
    const response = await fetch(
      `https://api.dexscreener.com/latest/dex/tokens/${tokenAddress}`
    );
    
    if (response.ok) {
      const data = await response.json();
      if (data.pairs && data.pairs.length > 0) {
        const matchingPair = data.pairs.find((p: any) => p.chainId === dexChain) || data.pairs[0];
        const txns = matchingPair?.txns || {};
        const priceChange = matchingPair?.priceChange || {};
        
        const buys24h = txns.h24?.buys || 0;
        const sells24h = txns.h24?.sells || 0;
        const totalTxns = buys24h + sells24h;
        const buyPressure = totalTxns > 0 ? ((buys24h / totalTxns) * 100).toFixed(1) : "50.0";
        
        let sentiment = "Neutral";
        if (parseFloat(buyPressure) > 60) sentiment = "Bullish";
        else if (parseFloat(buyPressure) < 40) sentiment = "Bearish";
        
        return {
          logo: matchingPair?.info?.imageUrl || null,
          website: matchingPair?.info?.websites?.[0]?.url || null,
          twitter: matchingPair?.info?.socials?.find((s: any) => s.type === "twitter")?.url || null,
          telegram: matchingPair?.info?.socials?.find((s: any) => s.type === "telegram")?.url || null,
          discord: matchingPair?.info?.socials?.find((s: any) => s.type === "discord")?.url || null,
          technical: {
            priceUsd: matchingPair?.priceUsd || null,
            priceChange5m: priceChange.m5 || 0,
            priceChange1h: priceChange.h1 || 0,
            priceChange6h: priceChange.h6 || 0,
            priceChange24h: priceChange.h24 || 0,
            volume24h: matchingPair?.volume?.h24 || 0,
            liquidity: matchingPair?.liquidity?.usd || 0,
            marketCap: matchingPair?.marketCap || matchingPair?.fdv || 0,
          },
          sentiment: {
            overall: sentiment,
            buyPressure: `${buyPressure}%`,
            buys24h,
            sells24h,
            totalTxns24h: totalTxns,
          },
        };
      }
    }
  } catch (e) {
    console.log("DexScreener fetch failed, continuing without social data");
  }
  return null;
}

async function analyzeEvmToken(tokenAddress: string, chainId: string) {
  const [goPlusResponse, dexScreenerData] = await Promise.all([
    fetch(`https://api.gopluslabs.io/api/v1/token_security/${chainId}?contract_addresses=${tokenAddress}`, {
      headers: { "Accept": "application/json" },
    }),
    fetchDexScreenerData(tokenAddress, chainId),
  ]);

  if (!goPlusResponse.ok) {
    throw new Error("Failed to fetch token data from GoPlus");
  }

  const data = await goPlusResponse.json();
  
  if (data.code !== 1 || !data.result || !data.result[tokenAddress.toLowerCase()]) {
    throw new Error("Token not found or invalid address");
  }

  const tokenData = data.result[tokenAddress.toLowerCase()];
  const checksumAddress = tokenAddress;

  const socials: Record<string, string> = {};
  if (dexScreenerData?.website) socials.website = dexScreenerData.website;
  if (dexScreenerData?.twitter) socials.twitter = dexScreenerData.twitter;
  if (dexScreenerData?.telegram) socials.telegram = dexScreenerData.telegram;
  if (dexScreenerData?.discord) socials.discord = dexScreenerData.discord;

  return {
    chain: CHAIN_NAMES[chainId] || `Chain ${chainId}`,
    address: tokenAddress,
    name: tokenData.token_name || "Unknown",
    symbol: tokenData.token_symbol || "N/A",
    logo: dexScreenerData?.logo || getTokenLogo(checksumAddress, chainId),
    socials,
    technical: dexScreenerData?.technical || null,
    sentiment: dexScreenerData?.sentiment || null,
    simulation: {
      buyTax: tokenData.buy_tax ? `${(parseFloat(tokenData.buy_tax) * 100).toFixed(1)}%` : "0.0%",
      sellTax: tokenData.sell_tax ? `${(parseFloat(tokenData.sell_tax) * 100).toFixed(1)}%` : "0.0%",
      transferTax: tokenData.transfer_tax ? `${(parseFloat(tokenData.transfer_tax) * 100).toFixed(1)}%` : "0.0%",
      buyGas: tokenData.buy_gas || "N/A",
      sellGas: tokenData.sell_gas || "N/A",
      buyLimit: tokenData.cannot_buy === "1" ? "BLOCKED" : "NONE DETECTED",
      sellLimit: tokenData.cannot_sell_all === "1" ? "RESTRICTED" : "NONE DETECTED",
      isHoneypot: tokenData.is_honeypot === "1",
      sourceCode: tokenData.is_open_source === "1" ? "OPEN SOURCE" : "NOT VERIFIED",
    },
    holders: {
      total: parseInt(tokenData.holder_count) || 0,
      canSell: parseInt(tokenData.holder_count) - (parseInt(tokenData.holders_cannot_sell_count) || 0),
      cantSell: parseInt(tokenData.holders_cannot_sell_count) || 0,
      siphoned: 0,
      averageTax: tokenData.sell_tax ? `${(parseFloat(tokenData.sell_tax) * 100).toFixed(1)}%` : "0.0%",
      highestTax: tokenData.sell_tax ? `${(parseFloat(tokenData.sell_tax) * 100).toFixed(1)}%` : "0.0%",
      averageGas: tokenData.sell_gas || "N/A",
    },
    security: {
      isProxy: tokenData.is_proxy === "1",
      isMintable: tokenData.is_mintable === "1",
      canTakeBackOwnership: tokenData.can_take_back_ownership === "1",
      ownerCanChangeBalance: tokenData.owner_change_balance === "1",
      hiddenOwner: tokenData.hidden_owner === "1",
      hasTradingCooldown: tokenData.trading_cooldown === "1",
      isBlacklisted: tokenData.is_blacklisted === "1",
      isWhitelisted: tokenData.is_whitelisted === "1",
      antiWhaleModifiable: tokenData.anti_whale_modifiable === "1",
      lpLocked: tokenData.lp_holder_count ? parseInt(tokenData.lp_holder_count) > 0 : false,
    },
    raw: tokenData,
  };
}

async function analyzeSolanaToken(tokenAddress: string) {
  const HELIUS_API_KEY = process.env.HELIUS_API_KEY;
  if (!HELIUS_API_KEY) {
    throw new Error("Helius API key not configured");
  }

  const metadataResponse = await fetch(
    `https://api.helius.xyz/v0/token-metadata?api-key=${HELIUS_API_KEY}`,
    {
      method: "POST",
      headers: { "Content-Type": "application/json" },
      body: JSON.stringify({ mintAccounts: [tokenAddress] }),
    }
  );

  if (!metadataResponse.ok) {
    throw new Error("Failed to fetch token metadata from Helius");
  }

  const metadataData = await metadataResponse.json();
  if (!metadataData || metadataData.length === 0) {
    throw new Error("Token not found");
  }

  const tokenInfo = metadataData[0];
  const parsedInfo = tokenInfo.onChainAccountInfo?.accountInfo?.data?.parsed?.info;
  const offChainMeta = tokenInfo.offChainMetadata?.metadata;
  const legacyMeta = tokenInfo.legacyMetadata;

  const socials: Record<string, string> = {};
  if (offChainMeta?.external_url) socials.website = offChainMeta.external_url;
  if (legacyMeta?.website) socials.website = legacyMeta.website;
  if (legacyMeta?.twitter) socials.twitter = legacyMeta.twitter;
  if (legacyMeta?.telegram) socials.telegram = legacyMeta.telegram;
  if (legacyMeta?.discord) socials.discord = legacyMeta.discord;

  return {
    chain: "Solana",
    address: tokenAddress,
    name: parsedInfo?.name || offChainMeta?.name || legacyMeta?.name || "Unknown",
    symbol: parsedInfo?.symbol || offChainMeta?.symbol || legacyMeta?.symbol || "N/A",
    logo: offChainMeta?.image || legacyMeta?.logoURI || null,
    socials,
    simulation: {
      buyTax: "0.0%",
      sellTax: "0.0%",
      transferTax: "0.0%",
      buyGas: "5000",
      sellGas: "5000",
      buyLimit: "NONE DETECTED",
      sellLimit: "NONE DETECTED",
      isHoneypot: false,
      sourceCode: "N/A (Solana)",
    },
    holders: {
      total: 0,
      canSell: 0,
      cantSell: 0,
      siphoned: 0,
      averageTax: "0.0%",
      highestTax: "0.0%",
      averageGas: "5000",
    },
    security: {
      mintAuthority: parsedInfo?.mintAuthority || null,
      freezeAuthority: parsedInfo?.freezeAuthority || null,
      isMintable: !!parsedInfo?.mintAuthority,
      isFreezable: !!parsedInfo?.freezeAuthority,
    },
  };
}

function detectChain(address: string): "solana" | "evm" {
  if (address.startsWith("0x") && address.length === 42) {
    return "evm";
  }
  return "solana";
}

export async function registerRoutes(
  httpServer: Server,
  app: Express
): Promise<Server> {
  app.post("/api/analyze-token", async (req, res) => {
    try {
      const { tokenAddress, chain } = req.body;

      if (!tokenAddress) {
        return res.status(400).json({ error: "Token address is required" });
      }

      const detectedChain = chain || (detectChain(tokenAddress) === "evm" ? "bsc" : "solana");
      
      let result;
      
      if (detectedChain === "solana") {
        result = await analyzeSolanaToken(tokenAddress);
      } else {
        const chainId = CHAIN_IDS[detectedChain] || "56";
        result = await analyzeEvmToken(tokenAddress, chainId);
      }

      return res.json(result);
    } catch (error) {
      console.error("Token analysis error:", error);
      const message = error instanceof Error ? error.message : "Internal server error";
      return res.status(500).json({ error: message });
    }
  });

  app.get("/api/supported-chains", (_req, res) => {
    res.json({
      chains: [
        { id: "solana", name: "Solana", icon: "☀️" },
        { id: "ethereum", name: "Ethereum", icon: "⟠" },
        { id: "bsc", name: "BSC", icon: "🔶" },
        { id: "polygon", name: "Polygon", icon: "🟣" },
        { id: "arbitrum", name: "Arbitrum", icon: "🔵" },
        { id: "base", name: "Base", icon: "🔷" },
        { id: "avalanche", name: "Avalanche", icon: "🔺" },
        { id: "fantom", name: "Fantom", icon: "👻" },
        { id: "optimism", name: "Optimism", icon: "🔴" },
      ],
    });
  });

  return httpServer;
}
