//SPDX-License-Identifier: UNLICENSED

pragma solidity 0.8.11;

contract Context {
    constructor() {}

    function _msgSender() internal view returns (address payable) {
        return payable(msg.sender);
    }

    function _msgData() internal view returns (bytes memory) {
        this;
        return msg.data;
    }
}

contract Ownable is Context {
    address private _owner;

    event OwnershipTransferred(address indexed previousOwner, address indexed newOwner);

    constructor() {
        address msgSender = _msgSender();
        _owner = msgSender;
        emit OwnershipTransferred(address(0), msgSender);
    }

    function owner() public view returns (address) {
        return _owner;
    }

    modifier onlyOwner() {
        require(_owner == _msgSender(), 'Ownable: caller is not the owner');
        _;
    }

    function renounceOwnership() public onlyOwner {
        emit OwnershipTransferred(_owner, address(0));
        _owner = address(0);
    }

    function transferOwnership(address newOwner) public onlyOwner {
        _transferOwnership(newOwner);
    }

    function _transferOwnership(address newOwner) internal {
        require(newOwner != address(0), 'Ownable: new owner is the zero address');
        emit OwnershipTransferred(_owner, newOwner);
        _owner = newOwner;
    }
}

library SafeMath {
    function add(uint256 a, uint256 b) internal pure returns (uint256) {
        uint256 c = a + b;
        require(c >= a, 'SafeMath: addition overflow');
        return c;
    }

    function sub(uint256 a, uint256 b) internal pure returns (uint256) {
        return sub(a, b, 'SafeMath: subtraction overflow');
    }

    function sub(uint256 a, uint256 b, string memory errorMessage) internal pure returns (uint256) {
        require(b <= a, errorMessage);
        uint256 c = a - b;
        return c;
    }

    function mul(uint256 a, uint256 b) internal pure returns (uint256) {
        if (a == 0) {
            return 0;
        }
        uint256 c = a * b;
        require(c / a == b, 'SafeMath: multiplication overflow');
        return c;
    }

    function div(uint256 a, uint256 b) internal pure returns (uint256) {
        return div(a, b, 'SafeMath: division by zero');
    }

    function div(uint256 a, uint256 b, string memory errorMessage) internal pure returns (uint256) {
        require(b > 0, errorMessage);
        uint256 c = a / b;
        return c;
    }

    function mod(uint256 a, uint256 b) internal pure returns (uint256) {
        return mod(a, b, 'SafeMath: modulo by zero');
    }

    function mod(uint256 a, uint256 b, string memory errorMessage) internal pure returns (uint256) {
        require(b != 0, errorMessage);
        return a % b;
    }
}

interface IBEP20 {
    function totalSupply() external view returns (uint256);
    function decimals() external view returns (uint8);
    function symbol() external view returns (string memory);
    function name() external view returns (string memory);
    function getOwner() external view returns (address);
    function balanceOf(address account) external view returns (uint256);
    function transfer(address recipient, uint256 amount) external returns (bool);
    function allowance(address _owner, address spender) external view returns (uint256);
    function approve(address spender, uint256 amount) external returns (bool);
    function transferFrom(address sender, address recipient, uint256 amount) external returns (bool);
    event Transfer(address indexed from, address indexed to, uint256 value);
    event Approval(address indexed owner, address indexed spender, uint256 value);
}

library SafeBEP20 {
    using SafeMath for uint256;
    using Address for address;

    function safeTransfer(IBEP20 token, address to, uint256 value) internal {
        _callOptionalReturn(token, abi.encodeWithSelector(token.transfer.selector, to, value));
    }

    function safeTransferFrom(IBEP20 token, address from, address to, uint256 value) internal {
        _callOptionalReturn(token, abi.encodeWithSelector(token.transferFrom.selector, from, to, value));
    }

    function safeApprove(IBEP20 token, address spender, uint256 value) internal {
        require(
            (value == 0) || (token.allowance(address(this), spender) == 0),
            'SafeBEP20: approve from non-zero to non-zero allowance'
        );
        _callOptionalReturn(token, abi.encodeWithSelector(token.approve.selector, spender, value));
    }

    function _callOptionalReturn(IBEP20 token, bytes memory data) private {
        bytes memory returndata = address(token).functionCall(data, 'SafeBEP20: low-level call failed');
        if (returndata.length > 0) {
            require(abi.decode(returndata, (bool)), 'SafeBEP20: BEP20 operation did not succeed');
        }
    }
}

library Address {
    function isContract(address account) internal view returns (bool) {
        bytes32 codehash;
        bytes32 accountHash = 0xc5d2460186f7233c927e7db2dcc703c0e500b653ca82273b7bfad8045d85a470;
        assembly {
            codehash := extcodehash(account)
        }
        return (codehash != accountHash && codehash != 0x0);
    }

    function sendValue(address payable recipient, uint256 amount) internal {
        require(address(this).balance >= amount, 'Address: insufficient balance');
        (bool success, ) = recipient.call{value: amount}('');
        require(success, 'Address: unable to send value, recipient may have reverted');
    }

    function functionCall(address target, bytes memory data) internal returns (bytes memory) {
        return functionCall(target, data, 'Address: low-level call failed');
    }

    function functionCall(address target, bytes memory data, string memory errorMessage) internal returns (bytes memory) {
        return _functionCallWithValue(target, data, 0, errorMessage);
    }

    function _functionCallWithValue(address target, bytes memory data, uint256 weiValue, string memory errorMessage) private returns (bytes memory) {
        require(isContract(target), 'Address: call to non-contract');
        (bool success, bytes memory returndata) = target.call{value: weiValue}(data);
        if (success) {
            return returndata;
        } else {
            if (returndata.length > 0) {
                assembly {
                    let returndata_size := mload(returndata)
                    revert(add(32, returndata), returndata_size)
                }
            } else {
                revert(errorMessage);
            }
        }
    }
}

abstract contract ReentrancyGuard {
    uint256 private constant _NOT_ENTERED = 1;
    uint256 private constant _ENTERED = 2;
    uint256 private _status;

    constructor() {
        _status = _NOT_ENTERED;
    }

    modifier nonReentrant() {
        require(_status != _ENTERED, "ReentrancyGuard: reentrant call");
        _status = _ENTERED;
        _;
        _status = _NOT_ENTERED;
    }
}

contract ASTHERAFIStakingMultiPool is Ownable, ReentrancyGuard {
    using SafeMath for uint256;
    using SafeBEP20 for IBEP20;

    struct UserInfo {
        uint256 amount;
        uint256 rewardDebt;
        uint256 unlockTime;
    }

    struct PoolInfo {
        string name;
        uint256 apy;
        uint256 lockDuration;
        uint256 exitPenaltyPerc;
        uint256 totalStaked;
        uint256 lastRewardTimestamp;
        uint256 accTokensPerShare;
        bool isActive;
    }

    IBEP20 public immutable stakingToken;
    IBEP20 public immutable rewardToken;

    PoolInfo[] public poolInfo;
    mapping(uint256 => mapping(address => UserInfo)) public userInfo;

    event Deposit(address indexed user, uint256 indexed pid, uint256 amount);
    event Withdraw(address indexed user, uint256 indexed pid, uint256 amount);
    event EmergencyWithdraw(address indexed user, uint256 indexed pid, uint256 amount);
    event PoolAdded(uint256 indexed pid, string name, uint256 apy, uint256 lockDuration);
    event PoolUpdated(uint256 indexed pid, uint256 apy, uint256 lockDuration, uint256 exitPenalty);
    event BNBWithdrawn(address indexed owner, uint256 amount);
    event TokenWithdrawn(address indexed token, address indexed owner, uint256 amount);

    constructor(address _stakingToken) {
        stakingToken = IBEP20(_stakingToken);
        rewardToken = IBEP20(_stakingToken);

        // Pool 0: Flexible (0 days lock, 60% APY)
        poolInfo.push(PoolInfo({
            name: "Flexible",
            apy: 60,
            lockDuration: 0,
            exitPenaltyPerc: 10,
            totalStaked: 0,
            lastRewardTimestamp: 21616747,
            accTokensPerShare: 0,
            isActive: true
        }));

        // Pool 1: 3 Days Lock (90% APY)
        poolInfo.push(PoolInfo({
            name: "3 Days Lock",
            apy: 90,
            lockDuration: 3 days,
            exitPenaltyPerc: 10,
            totalStaked: 0,
            lastRewardTimestamp: 21616747,
            accTokensPerShare: 0,
            isActive: true
        }));

        // Pool 2: 7 Days Lock (150% APY)
        poolInfo.push(PoolInfo({
            name: "7 Days Lock",
            apy: 150,
            lockDuration: 7 days,
            exitPenaltyPerc: 10,
            totalStaked: 0,
            lastRewardTimestamp: 21616747,
            accTokensPerShare: 0,
            isActive: true
        }));
    }

    // Accept BNB deposits
    receive() external payable {}

    // Get number of pools
    function poolLength() external view returns (uint256) {
        return poolInfo.length;
    }

    // Get total staked across all pools
    function getTotalStaked() public view returns (uint256) {
        uint256 total = 0;
        for (uint256 i = 0; i < poolInfo.length; i++) {
            total = total.add(poolInfo[i].totalStaked);
        }
        return total;
    }

    // Get pool info
    function getPoolInfo(uint256 _pid) external view returns (
        string memory name,
        uint256 apy,
        uint256 lockDuration,
        uint256 exitPenaltyPerc,
        uint256 totalStaked,
        bool isActive
    ) {
        PoolInfo storage pool = poolInfo[_pid];
        return (pool.name, pool.apy, pool.lockDuration, pool.exitPenaltyPerc, pool.totalStaked, pool.isActive);
    }

    // Get user info for a pool
    function getUserInfo(uint256 _pid, address _user) external view returns (
        uint256 amount,
        uint256 unlockTime,
        uint256 pendingRewards
    ) {
        UserInfo storage user = userInfo[_pid][_user];
        return (user.amount, user.unlockTime, this.pendingReward(_pid, _user));
    }

    // Add a new pool - Owner only
    function addPool(
        string memory _name,
        uint256 _apy,
        uint256 _lockDurationDays,
        uint256 _exitPenaltyPerc
    ) external onlyOwner {
        require(_apy <= 10000, "APY too high");
        require(_exitPenaltyPerc <= 30, "Penalty too high");

        poolInfo.push(PoolInfo({
            name: _name,
            apy: _apy,
            lockDuration: _lockDurationDays * 1 days,
            exitPenaltyPerc: _exitPenaltyPerc,
            totalStaked: 0,
            lastRewardTimestamp: block.timestamp,
            accTokensPerShare: 0,
            isActive: true
        }));

        emit PoolAdded(poolInfo.length - 1, _name, _apy, _lockDurationDays);
    }

    // Update pool parameters - Owner only
    function updatePool(
        uint256 _pid,
        uint256 _apy,
        uint256 _lockDurationDays,
        uint256 _exitPenaltyPerc,
        bool _isActive
    ) external onlyOwner {
        require(_pid < poolInfo.length, "Invalid pool");
        require(_apy <= 10000, "APY too high");
        require(_exitPenaltyPerc <= 30, "Penalty too high");

        _updatePoolRewards(_pid);

        PoolInfo storage pool = poolInfo[_pid];
        pool.apy = _apy;
        pool.lockDuration = _lockDurationDays * 1 days;
        pool.exitPenaltyPerc = _exitPenaltyPerc;
        pool.isActive = _isActive;

        emit PoolUpdated(_pid, _apy, _lockDurationDays, _exitPenaltyPerc);
    }

    // Start rewards for a pool - Owner only
    function startReward(uint256 _pid) external onlyOwner {
        require(_pid < poolInfo.length, "Invalid pool");
        require(poolInfo[_pid].lastRewardTimestamp == 21616747, "Already started");
        poolInfo[_pid].lastRewardTimestamp = block.timestamp;
    }

    // Start rewards for all pools - Owner only
    function startAllRewards() external onlyOwner {
        for (uint256 i = 0; i < poolInfo.length; i++) {
            if (poolInfo[i].lastRewardTimestamp == 21616747) {
                poolInfo[i].lastRewardTimestamp = block.timestamp;
            }
        }
    }

    // Stop rewards for a pool - Owner only
    function stopReward(uint256 _pid) external onlyOwner {
        require(_pid < poolInfo.length, "Invalid pool");
        _updatePoolRewards(_pid);
        poolInfo[_pid].apy = 0;
    }

    // View pending rewards
    function pendingReward(uint256 _pid, address _user) external view returns (uint256) {
        require(_pid < poolInfo.length, "Invalid pool");
        PoolInfo storage pool = poolInfo[_pid];
        UserInfo storage user = userInfo[_pid][_user];

        if (pool.lastRewardTimestamp == 21616747 || user.amount == 0) {
            return 0;
        }

        uint256 accTokensPerShare = pool.accTokensPerShare;
        if (block.timestamp > pool.lastRewardTimestamp && pool.totalStaked != 0) {
            uint256 tokenReward = _calculateRewards(_pid);
            accTokensPerShare = accTokensPerShare.add(tokenReward.mul(1e12).div(pool.totalStaked));
        }
        return user.amount.mul(accTokensPerShare).div(1e12).sub(user.rewardDebt);
    }

    // Calculate rewards for a pool
    function _calculateRewards(uint256 _pid) internal view returns (uint256) {
        PoolInfo storage pool = poolInfo[_pid];
        if (pool.lastRewardTimestamp >= block.timestamp) {
            return 0;
        }
        uint256 timePassed = block.timestamp - pool.lastRewardTimestamp;
        return (timePassed * pool.totalStaked * pool.apy) / 100 / 365 days;
    }

    // Update pool rewards
    function _updatePoolRewards(uint256 _pid) internal {
        PoolInfo storage pool = poolInfo[_pid];
        if (block.timestamp <= pool.lastRewardTimestamp) {
            return;
        }
        if (pool.totalStaked == 0) {
            pool.lastRewardTimestamp = block.timestamp;
            return;
        }
        uint256 tokenReward = _calculateRewards(_pid);
        pool.accTokensPerShare = pool.accTokensPerShare.add(tokenReward.mul(1e12).div(pool.totalStaked));
        pool.lastRewardTimestamp = block.timestamp;
    }

    // Deposit tokens to a pool
    function deposit(uint256 _pid, uint256 _amount) public nonReentrant {
        require(_pid < poolInfo.length, "Invalid pool");
        PoolInfo storage pool = poolInfo[_pid];
        UserInfo storage user = userInfo[_pid][msg.sender];

        require(pool.isActive, "Pool not active");

        _updatePoolRewards(_pid);

        // Pay pending rewards
        if (user.amount > 0) {
            uint256 pending = user.amount.mul(pool.accTokensPerShare).div(1e12).sub(user.rewardDebt);
            if (pending > 0) {
                require(pending <= rewardsRemaining(), "Insufficient reward tokens");
                rewardToken.safeTransfer(msg.sender, pending);
            }
        }

        // Transfer tokens
        if (_amount > 0) {
            uint256 initialBalance = stakingToken.balanceOf(address(this));
            stakingToken.safeTransferFrom(msg.sender, address(this), _amount);
            uint256 amountTransferred = stakingToken.balanceOf(address(this)) - initialBalance;
            
            user.amount = user.amount.add(amountTransferred);
            pool.totalStaked = pool.totalStaked.add(amountTransferred);

            // Set/extend unlock time
            if (pool.lockDuration > 0) {
                user.unlockTime = block.timestamp + pool.lockDuration;
            }
        }

        user.rewardDebt = user.amount.mul(pool.accTokensPerShare).div(1e12);
        emit Deposit(msg.sender, _pid, _amount);
    }

    // Withdraw tokens from a pool (with rewards)
    function withdraw(uint256 _pid) public nonReentrant {
        require(_pid < poolInfo.length, "Invalid pool");
        PoolInfo storage pool = poolInfo[_pid];
        UserInfo storage user = userInfo[_pid][msg.sender];

        require(user.amount > 0, "Nothing to withdraw");
        require(user.unlockTime <= block.timestamp || pool.lockDuration == 0, "Tokens still locked");

        _updatePoolRewards(_pid);

        uint256 _amount = user.amount;

        // Pay pending rewards
        uint256 pending = user.amount.mul(pool.accTokensPerShare).div(1e12).sub(user.rewardDebt);
        if (pending > 0) {
            require(pending <= rewardsRemaining(), "Insufficient reward tokens");
            rewardToken.safeTransfer(msg.sender, pending);
        }

        // Transfer staked tokens
        user.amount = 0;
        pool.totalStaked = pool.totalStaked.sub(_amount);
        user.unlockTime = 0;
        user.rewardDebt = 0;
        stakingToken.safeTransfer(msg.sender, _amount);

        emit Withdraw(msg.sender, _pid, _amount);
    }

    // Emergency withdraw (forfeits rewards, may incur penalty)
    function emergencyWithdraw(uint256 _pid) external nonReentrant {
        require(_pid < poolInfo.length, "Invalid pool");
        PoolInfo storage pool = poolInfo[_pid];
        UserInfo storage user = userInfo[_pid][msg.sender];

        require(user.amount > 0, "Nothing to withdraw");

        uint256 _amount = user.amount;
        pool.totalStaked = pool.totalStaked.sub(_amount);

        // Apply penalty if still locked
        if (user.unlockTime > block.timestamp && pool.lockDuration > 0) {
            uint256 penalty = _amount.mul(pool.exitPenaltyPerc).div(100);
            _amount = _amount.sub(penalty);
        }

        user.amount = 0;
        user.unlockTime = 0;
        user.rewardDebt = 0;
        stakingToken.safeTransfer(msg.sender, _amount);

        emit EmergencyWithdraw(msg.sender, _pid, _amount);
    }

    // Get remaining rewards available
    function rewardsRemaining() public view returns (uint256) {
        uint256 totalStaked = getTotalStaked();
        uint256 balance = rewardToken.balanceOf(address(this));
        if (balance <= totalStaked) {
            return 0;
        }
        return balance - totalStaked;
    }

    // ========== OWNER FUNCTIONS ==========

    // Withdraw all BNB from contract
    function withdrawBNB() external onlyOwner {
        uint256 balance = address(this).balance;
        require(balance > 0, "No BNB to withdraw");
        payable(owner()).transfer(balance);
        emit BNBWithdrawn(owner(), balance);
    }

    // Withdraw specific amount of BNB
    function withdrawBNBAmount(uint256 _amount) external onlyOwner {
        require(_amount <= address(this).balance, "Insufficient BNB");
        payable(owner()).transfer(_amount);
        emit BNBWithdrawn(owner(), _amount);
    }

    // Withdraw any token except staking token
    function withdrawToken(address _tokenAddress, uint256 _amount) external onlyOwner {
        require(_tokenAddress != address(stakingToken), "Use withdrawRewardTokens");
        IBEP20 token = IBEP20(_tokenAddress);
        uint256 balance = token.balanceOf(address(this));
        if (_amount == 0) {
            _amount = balance;
        }
        require(_amount <= balance, "Insufficient balance");
        token.safeTransfer(owner(), _amount);
        emit TokenWithdrawn(_tokenAddress, owner(), _amount);
    }

    // Withdraw excess reward tokens (not user staked)
    function withdrawRewardTokens(uint256 _amount) external onlyOwner {
        uint256 available = rewardsRemaining();
        require(_amount <= available, "Cannot withdraw user funds");
        rewardToken.safeTransfer(owner(), _amount);
        emit TokenWithdrawn(address(rewardToken), owner(), _amount);
    }

    // Emergency: withdraw all tokens - DANGEROUS
    function emergencyWithdrawAllTokens() external onlyOwner {
        uint256 balance = rewardToken.balanceOf(address(this));
        rewardToken.safeTransfer(owner(), balance);
        emit TokenWithdrawn(address(rewardToken), owner(), balance);
    }

    // Withdraw deposited/staked tokens - Owner only
    function withdrawDepositedTokens(uint256 _amount) external onlyOwner {
        uint256 balance = stakingToken.balanceOf(address(this));
        require(_amount <= balance, "Insufficient token balance");
        if (_amount == 0) {
            _amount = balance;
        }
        stakingToken.safeTransfer(owner(), _amount);
        emit TokenWithdrawn(address(stakingToken), owner(), _amount);
    }

    // Withdraw all deposited tokens from all pools - Owner only
    function withdrawAllDepositedTokens() external onlyOwner {
        uint256 balance = stakingToken.balanceOf(address(this));
        require(balance > 0, "No tokens to withdraw");
        stakingToken.safeTransfer(owner(), balance);
        emit TokenWithdrawn(address(stakingToken), owner(), balance);
    }

    // View contract balances
    function getBNBBalance() external view returns (uint256) {
        return address(this).balance;
    }

    function getTokenBalance(address _token) external view returns (uint256) {
        return IBEP20(_token).balanceOf(address(this));
    }
}
