"use strict";

process.on("uncaughtException", (err) => { console.error("UNCAUGHT:", err); });
process.on("unhandledRejection", (err) => { console.error("UNHANDLED:", err); });

console.log("Astherafi server starting...");
console.log("Node version:", process.version);
console.log("CWD:", process.cwd());
console.log("__dirname:", __dirname);

const express = require("express");
const path = require("path");

const app = express();
const PORT = process.env.PORT || 5000;

const publicPath = path.join(__dirname, "public");
console.log("Serving static files from:", publicPath);

const fs = require("fs");
if (fs.existsSync(publicPath)) {
  console.log("Public directory exists. Contents:", fs.readdirSync(publicPath).join(", "));
} else {
  console.error("ERROR: Public directory does NOT exist at", publicPath);
}

app.use(express.static(publicPath));
app.use(express.json());

const CHAIN_IDS = {
  ethereum: "1", bsc: "56", polygon: "137", arbitrum: "42161",
  base: "8453", avalanche: "43114", fantom: "250", optimism: "10",
};
const CHAIN_NAMES = {
  "1": "Ethereum", "56": "BSC", "137": "Polygon", "42161": "Arbitrum",
  "8453": "Base", "43114": "Avalanche", "250": "Fantom", "10": "Optimism",
};
const TRUST_WALLET_CHAIN_NAMES = {
  "1": "ethereum", "56": "smartchain", "137": "polygon", "42161": "arbitrum",
  "8453": "base", "43114": "avalanchec", "250": "fantom", "10": "optimism",
};
const DEXSCREENER_CHAIN_IDS = {
  "1": "ethereum", "56": "bsc", "137": "polygon", "42161": "arbitrum",
  "8453": "base", "43114": "avalanche", "250": "fantom", "10": "optimism",
};

function getTokenLogo(address, chainId) {
  const chainName = TRUST_WALLET_CHAIN_NAMES[chainId] || "ethereum";
  return "https://raw.githubusercontent.com/trustwallet/assets/master/blockchains/" + chainName + "/assets/" + address + "/logo.png";
}

async function fetchDexScreenerData(tokenAddress, chainId) {
  try {
    const dexChain = DEXSCREENER_CHAIN_IDS[chainId] || "bsc";
    const response = await fetch("https://api.dexscreener.com/latest/dex/tokens/" + tokenAddress);
    if (response.ok) {
      const data = await response.json();
      if (data.pairs && data.pairs.length > 0) {
        const matchingPair = data.pairs.find(function(p) { return p.chainId === dexChain; }) || data.pairs[0];
        const txns = matchingPair && matchingPair.txns ? matchingPair.txns : {};
        const priceChange = matchingPair && matchingPair.priceChange ? matchingPair.priceChange : {};
        const buys24h = txns.h24 && txns.h24.buys ? txns.h24.buys : 0;
        const sells24h = txns.h24 && txns.h24.sells ? txns.h24.sells : 0;
        const totalTxns = buys24h + sells24h;
        const buyPressure = totalTxns > 0 ? ((buys24h / totalTxns) * 100).toFixed(1) : "50.0";
        var sentiment = "Neutral";
        if (parseFloat(buyPressure) > 60) sentiment = "Bullish";
        else if (parseFloat(buyPressure) < 40) sentiment = "Bearish";
        return {
          logo: matchingPair && matchingPair.info ? matchingPair.info.imageUrl : null,
          website: matchingPair && matchingPair.info && matchingPair.info.websites && matchingPair.info.websites[0] ? matchingPair.info.websites[0].url : null,
          twitter: matchingPair && matchingPair.info && matchingPair.info.socials ? (matchingPair.info.socials.find(function(s) { return s.type === "twitter"; }) || {}).url || null : null,
          telegram: matchingPair && matchingPair.info && matchingPair.info.socials ? (matchingPair.info.socials.find(function(s) { return s.type === "telegram"; }) || {}).url || null : null,
          discord: matchingPair && matchingPair.info && matchingPair.info.socials ? (matchingPair.info.socials.find(function(s) { return s.type === "discord"; }) || {}).url || null : null,
          technical: {
            priceUsd: matchingPair ? matchingPair.priceUsd : null,
            priceChange5m: priceChange.m5 || 0, priceChange1h: priceChange.h1 || 0,
            priceChange6h: priceChange.h6 || 0, priceChange24h: priceChange.h24 || 0,
            volume24h: matchingPair && matchingPair.volume ? matchingPair.volume.h24 : 0,
            liquidity: matchingPair && matchingPair.liquidity ? matchingPair.liquidity.usd : 0,
            marketCap: matchingPair ? (matchingPair.marketCap || matchingPair.fdv || 0) : 0,
          },
          sentiment: { overall: sentiment, buyPressure: buyPressure + "%", buys24h: buys24h, sells24h: sells24h, totalTxns24h: totalTxns },
        };
      }
    }
  } catch (e) {
    console.log("DexScreener fetch failed:", e.message);
  }
  return null;
}

async function analyzeEvmToken(tokenAddress, chainId) {
  var goPlusResponse, dexScreenerData;
  try {
    var results = await Promise.all([
      fetch("https://api.gopluslabs.io/api/v1/token_security/" + chainId + "?contract_addresses=" + tokenAddress, {
        headers: { "Accept": "application/json" },
      }),
      fetchDexScreenerData(tokenAddress, chainId),
    ]);
    goPlusResponse = results[0];
    dexScreenerData = results[1];
  } catch (err) {
    throw new Error("Failed to fetch token data");
  }
  if (!goPlusResponse.ok) throw new Error("Failed to fetch token data from GoPlus");
  var data = await goPlusResponse.json();
  if (data.code !== 1 || !data.result || !data.result[tokenAddress.toLowerCase()])
    throw new Error("Token not found or invalid address");
  var tokenData = data.result[tokenAddress.toLowerCase()];
  var socials = {};
  if (dexScreenerData && dexScreenerData.website) socials.website = dexScreenerData.website;
  if (dexScreenerData && dexScreenerData.twitter) socials.twitter = dexScreenerData.twitter;
  if (dexScreenerData && dexScreenerData.telegram) socials.telegram = dexScreenerData.telegram;
  if (dexScreenerData && dexScreenerData.discord) socials.discord = dexScreenerData.discord;
  return {
    chain: CHAIN_NAMES[chainId] || ("Chain " + chainId), address: tokenAddress,
    name: tokenData.token_name || "Unknown", symbol: tokenData.token_symbol || "N/A",
    logo: (dexScreenerData && dexScreenerData.logo) || getTokenLogo(tokenAddress, chainId), socials: socials,
    technical: dexScreenerData ? dexScreenerData.technical : null,
    sentiment: dexScreenerData ? dexScreenerData.sentiment : null,
    simulation: {
      buyTax: tokenData.buy_tax ? (parseFloat(tokenData.buy_tax) * 100).toFixed(1) + "%" : "0.0%",
      sellTax: tokenData.sell_tax ? (parseFloat(tokenData.sell_tax) * 100).toFixed(1) + "%" : "0.0%",
      transferTax: tokenData.transfer_tax ? (parseFloat(tokenData.transfer_tax) * 100).toFixed(1) + "%" : "0.0%",
      buyGas: tokenData.buy_gas || "N/A", sellGas: tokenData.sell_gas || "N/A",
      buyLimit: tokenData.cannot_buy === "1" ? "BLOCKED" : "NONE DETECTED",
      sellLimit: tokenData.cannot_sell_all === "1" ? "RESTRICTED" : "NONE DETECTED",
      isHoneypot: tokenData.is_honeypot === "1",
      sourceCode: tokenData.is_open_source === "1" ? "OPEN SOURCE" : "NOT VERIFIED",
    },
    holders: {
      total: parseInt(tokenData.holder_count) || 0,
      canSell: parseInt(tokenData.holder_count) - (parseInt(tokenData.holders_cannot_sell_count) || 0),
      cantSell: parseInt(tokenData.holders_cannot_sell_count) || 0, siphoned: 0,
      averageTax: tokenData.sell_tax ? (parseFloat(tokenData.sell_tax) * 100).toFixed(1) + "%" : "0.0%",
      highestTax: tokenData.sell_tax ? (parseFloat(tokenData.sell_tax) * 100).toFixed(1) + "%" : "0.0%",
      averageGas: tokenData.sell_gas || "N/A",
    },
    security: {
      isProxy: tokenData.is_proxy === "1", isMintable: tokenData.is_mintable === "1",
      canTakeBackOwnership: tokenData.can_take_back_ownership === "1",
      ownerCanChangeBalance: tokenData.owner_change_balance === "1",
      hiddenOwner: tokenData.hidden_owner === "1", hasTradingCooldown: tokenData.trading_cooldown === "1",
      isBlacklisted: tokenData.is_blacklisted === "1", isWhitelisted: tokenData.is_whitelisted === "1",
      antiWhaleModifiable: tokenData.anti_whale_modifiable === "1",
      lpLocked: tokenData.lp_holder_count ? parseInt(tokenData.lp_holder_count) > 0 : false,
    },
    raw: tokenData,
  };
}

async function analyzeSolanaToken(tokenAddress) {
  var HELIUS_API_KEY = process.env.HELIUS_API_KEY;
  if (!HELIUS_API_KEY) throw new Error("Helius API key not configured");
  var metadataResponse = await fetch("https://api.helius.xyz/v0/token-metadata?api-key=" + HELIUS_API_KEY, {
    method: "POST", headers: { "Content-Type": "application/json" },
    body: JSON.stringify({ mintAccounts: [tokenAddress] }),
  });
  if (!metadataResponse.ok) throw new Error("Failed to fetch token metadata from Helius");
  var metadataData = await metadataResponse.json();
  if (!metadataData || metadataData.length === 0) throw new Error("Token not found");
  var tokenInfo = metadataData[0];
  var parsedInfo = tokenInfo.onChainAccountInfo && tokenInfo.onChainAccountInfo.accountInfo && tokenInfo.onChainAccountInfo.accountInfo.data && tokenInfo.onChainAccountInfo.accountInfo.data.parsed ? tokenInfo.onChainAccountInfo.accountInfo.data.parsed.info : null;
  var offChainMeta = tokenInfo.offChainMetadata ? tokenInfo.offChainMetadata.metadata : null;
  var legacyMeta = tokenInfo.legacyMetadata || null;
  var socials = {};
  if (offChainMeta && offChainMeta.external_url) socials.website = offChainMeta.external_url;
  if (legacyMeta && legacyMeta.website) socials.website = legacyMeta.website;
  if (legacyMeta && legacyMeta.twitter) socials.twitter = legacyMeta.twitter;
  if (legacyMeta && legacyMeta.telegram) socials.telegram = legacyMeta.telegram;
  if (legacyMeta && legacyMeta.discord) socials.discord = legacyMeta.discord;
  return {
    chain: "Solana", address: tokenAddress,
    name: (parsedInfo && parsedInfo.name) || (offChainMeta && offChainMeta.name) || (legacyMeta && legacyMeta.name) || "Unknown",
    symbol: (parsedInfo && parsedInfo.symbol) || (offChainMeta && offChainMeta.symbol) || (legacyMeta && legacyMeta.symbol) || "N/A",
    logo: (offChainMeta && offChainMeta.image) || (legacyMeta && legacyMeta.logoURI) || null, socials: socials,
    simulation: { buyTax: "0.0%", sellTax: "0.0%", transferTax: "0.0%", buyGas: "5000", sellGas: "5000", buyLimit: "NONE DETECTED", sellLimit: "NONE DETECTED", isHoneypot: false, sourceCode: "N/A (Solana)" },
    holders: { total: 0, canSell: 0, cantSell: 0, siphoned: 0, averageTax: "0.0%", highestTax: "0.0%", averageGas: "5000" },
    security: { mintAuthority: parsedInfo ? parsedInfo.mintAuthority : null, freezeAuthority: parsedInfo ? parsedInfo.freezeAuthority : null, isMintable: parsedInfo ? !!parsedInfo.mintAuthority : false, isFreezable: parsedInfo ? !!parsedInfo.freezeAuthority : false },
  };
}

function detectChain(address) {
  if (address.startsWith("0x") && address.length === 42) return "evm";
  return "solana";
}

app.post("/api/analyze-token", async function(req, res) {
  try {
    var tokenAddress = req.body.tokenAddress;
    var chain = req.body.chain;
    if (!tokenAddress) return res.status(400).json({ error: "Token address is required" });
    var detectedChain = chain || (detectChain(tokenAddress) === "evm" ? "bsc" : "solana");
    var result;
    if (detectedChain === "solana") result = await analyzeSolanaToken(tokenAddress);
    else { var chainId = CHAIN_IDS[detectedChain] || "56"; result = await analyzeEvmToken(tokenAddress, chainId); }
    return res.json(result);
  } catch (error) {
    console.error("Token analysis error:", error);
    var message = error instanceof Error ? error.message : "Internal server error";
    return res.status(500).json({ error: message });
  }
});

app.get("/api/supported-chains", function(req, res) {
  res.json({
    chains: [
      { id: "solana", name: "Solana", icon: "S" }, { id: "ethereum", name: "Ethereum", icon: "E" },
      { id: "bsc", name: "BSC", icon: "B" }, { id: "polygon", name: "Polygon", icon: "P" },
      { id: "arbitrum", name: "Arbitrum", icon: "A" }, { id: "base", name: "Base", icon: "Ba" },
      { id: "avalanche", name: "Avalanche", icon: "Av" }, { id: "fantom", name: "Fantom", icon: "F" },
      { id: "optimism", name: "Optimism", icon: "O" },
    ],
  });
});

app.get("/api/totalsupply", async function(req, res) {
  try {
    var tokenAddress = req.query.address || "0x277204675524B49D417bD0B374A1FB09465F7777";
    var chain = (req.query.chain || "bsc").toLowerCase();
    var decimals = parseInt(req.query.decimals) || 18;
    var apiKey = process.env.BSCSCAN_API_KEY || "";
    var chainIds = { bsc: 56, ethereum: 1, polygon: 137, arbitrum: 42161, base: 8453 };
    var chainId = chainIds[chain];
    if (!chainId) return res.status(400).send("Unsupported chain");
    var apiUrl = "https://api.etherscan.io/v2/api?chainid=" + chainId + "&module=stats&action=tokensupply&contractaddress=" + tokenAddress + "&apikey=" + apiKey;
    var response = await fetch(apiUrl);
    var data = await response.json();
    if (data.status === "1" && data.result) {
      var totalSupply = parseFloat(data.result) / Math.pow(10, decimals);
      res.type("text/plain").send(totalSupply.toString());
    } else {
      var rpcUrls = { bsc: "https://bsc-dataseed.binance.org/", ethereum: "https://eth.llamarpc.com" };
      var rpcUrl = rpcUrls[chain];
      if (rpcUrl) {
        var rpcResponse = await fetch(rpcUrl, { method: "POST", headers: { "Content-Type": "application/json" },
          body: JSON.stringify({ jsonrpc: "2.0", method: "eth_call", params: [{ to: tokenAddress, data: "0x18160ddd" }, "latest"], id: 1 }) });
        var rpcData = await rpcResponse.json();
        if (rpcData.result && rpcData.result !== "0x") {
          var supplyBigInt = BigInt(rpcData.result);
          var totalSupply2 = Number(supplyBigInt) / Math.pow(10, decimals);
          return res.type("text/plain").send(totalSupply2.toString());
        }
      }
      res.status(500).send("Error fetching supply");
    }
  } catch (error) { console.error("Total supply error:", error); res.status(500).send("Error fetching supply"); }
});

app.get("/api/circulatingsupply", async function(req, res) {
  try {
    var tokenAddress = req.query.address || "0x277204675524B49D417bD0B374A1FB09465F7777";
    var chain = (req.query.chain || "bsc").toLowerCase();
    var decimals = parseInt(req.query.decimals) || 18;
    var rpcUrls = { bsc: "https://bsc-dataseed.binance.org/", ethereum: "https://eth.llamarpc.com" };
    var rpcUrl = rpcUrls[chain];
    if (!rpcUrl) return res.status(400).send("Unsupported chain");
    var rpcResponse = await fetch(rpcUrl, { method: "POST", headers: { "Content-Type": "application/json" },
      body: JSON.stringify({ jsonrpc: "2.0", method: "eth_call", params: [{ to: tokenAddress, data: "0x18160ddd" }, "latest"], id: 1 }) });
    var rpcData = await rpcResponse.json();
    if (rpcData.result && rpcData.result !== "0x") {
      var supplyBigInt = BigInt(rpcData.result);
      var totalSupply = Number(supplyBigInt) / Math.pow(10, decimals);
      return res.type("text/plain").send(totalSupply.toString());
    }
    res.status(500).send("Error fetching supply");
  } catch (error) { console.error("Circulating supply error:", error); res.status(500).send("Error fetching supply"); }
});

var chatRateLimit = new Map();
function checkChatRateLimit(ip) {
  var now = Date.now();
  var entry = chatRateLimit.get(ip);
  if (!entry || now > entry.resetAt) { chatRateLimit.set(ip, { count: 1, resetAt: now + 60000 }); return true; }
  if (entry.count >= 10) return false;
  entry.count++; return true;
}
setInterval(function() { var now = Date.now(); chatRateLimit.forEach(function(_e, ip) { if (now > chatRateLimit.get(ip).resetAt) chatRateLimit.delete(ip); }); }, 300000);

var chatHistoryMap = new Map();
var MAX_SESSIONS = 20;
var MAX_MSGS = 50;

function getOrCreateSession(ip, sessionId) {
  var sessions = chatHistoryMap.get(ip);
  if (!sessions) { sessions = []; chatHistoryMap.set(ip, sessions); }
  if (sessionId) { var existing = sessions.find(function(s) { return s.id === sessionId; }); if (existing) return existing; }
  var newSession = { id: Date.now().toString(36) + Math.random().toString(36).slice(2, 6), title: "", messages: [], createdAt: Date.now(), updatedAt: Date.now() };
  sessions.unshift(newSession);
  if (sessions.length > MAX_SESSIONS) sessions.pop();
  return newSession;
}

app.get("/api/chat/history", function(req, res) {
  var clientIp = req.ip || req.socket.remoteAddress || "unknown";
  var sessions = chatHistoryMap.get(clientIp) || [];
  var summary = sessions.map(function(s) { return { id: s.id, title: s.title || (s.messages[0] ? s.messages[0].text.substring(0, 50) + "..." : "New Chat"), messageCount: s.messages.length, createdAt: s.createdAt, updatedAt: s.updatedAt }; });
  return res.json({ sessions: summary });
});

app.get("/api/chat/history/:sessionId", function(req, res) {
  var clientIp = req.ip || req.socket.remoteAddress || "unknown";
  var sessions = chatHistoryMap.get(clientIp) || [];
  var session = sessions.find(function(s) { return s.id === req.params.sessionId; });
  if (!session) return res.status(404).json({ error: "Session not found" });
  return res.json({ session: session });
});

app.delete("/api/chat/history/:sessionId", function(req, res) {
  var clientIp = req.ip || req.socket.remoteAddress || "unknown";
  var sessions = chatHistoryMap.get(clientIp) || [];
  var idx = -1;
  for (var i = 0; i < sessions.length; i++) { if (sessions[i].id === req.params.sessionId) { idx = i; break; } }
  if (idx === -1) return res.status(404).json({ error: "Session not found" });
  sessions.splice(idx, 1);
  return res.json({ success: true });
});

var freeApiUsers = new Map();
var DAILY_LIMIT = 15;
var ASTHER_BSC_CONTRACT = "0x277204675524B49D417bD0B374A1FB09465F7777";

function generateApiKey() { var chars = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789"; var key = "sk-asth-"; for (var i = 0; i < 32; i++) key += chars.charAt(Math.floor(Math.random() * chars.length)); return key; }
function getTodayStr() { return new Date().toISOString().split("T")[0]; }
function resetDailyUsageIfNeeded(user) { var today = getTodayStr(); if (user.lastResetDate !== today) { user.usedToday = 0; user.lastResetDate = today; } }

async function getBscTokenBalance(walletAddress) {
  try {
    var apiKey = process.env.BSCSCAN_API_KEY || "";
    var url = "https://api.bscscan.com/api?module=account&action=tokenbalance&contractaddress=" + ASTHER_BSC_CONTRACT + "&address=" + walletAddress + "&tag=latest&apikey=" + apiKey;
    var response = await fetch(url);
    var data = await response.json();
    if (data.status === "1" && data.result) {
      return parseFloat(data.result) / 1e18;
    }
    return 0;
  } catch (e) { console.error("BSC balance check error:", e); return 0; }
}

app.post("/api/free-api/nonce", function(req, res) {
  var wallet = req.body.wallet;
  if (!wallet || typeof wallet !== "string" || !wallet.startsWith("0x") || wallet.length !== 42) return res.status(400).json({ error: "Valid BSC wallet address required" });
  var chars = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789";
  var nonce = "";
  for (var i = 0; i < 32; i++) nonce += chars.charAt(Math.floor(Math.random() * chars.length));
  var message = "Astherafi Free API Authentication\nWallet: " + wallet + "\nNonce: " + nonce;
  return res.json({ nonce: nonce, message: message });
});

app.post("/api/free-api/balance", async function(req, res) {
  try {
    var wallet = req.body.wallet;
    if (!wallet || typeof wallet !== "string" || !wallet.startsWith("0x") || wallet.length !== 42) return res.status(400).json({ error: "Valid BSC wallet address required" });
    var balance = await getBscTokenBalance(wallet);
    return res.json({ balance: balance, hasTokens: balance > 0 });
  } catch (e) { return res.status(500).json({ error: "Failed to check balance" }); }
});

app.post("/api/free-api/key", async function(req, res) {
  try {
    var wallet = req.body.wallet;
    if (!wallet || typeof wallet !== "string" || !wallet.startsWith("0x") || wallet.length !== 42) return res.status(400).json({ error: "Valid BSC wallet address required" });
    var existing = freeApiUsers.get(wallet.toLowerCase());
    if (existing) {
      resetDailyUsageIfNeeded(existing);
      return res.json({ apiKey: existing.apiKey, usedToday: existing.usedToday, dailyLimit: DAILY_LIMIT });
    }
    var balance = await getBscTokenBalance(wallet);
    if (balance <= 0) return res.status(403).json({ error: "You must hold ASTHER tokens on BSC to get a free API key. Buy $ASTHER on BNB Smart Chain first." });
    var apiKey = generateApiKey();
    var user = { wallet: wallet, apiKey: apiKey, createdAt: Date.now(), usedToday: 0, lastResetDate: getTodayStr() };
    freeApiUsers.set(wallet.toLowerCase(), user);
    return res.json({ apiKey: apiKey, usedToday: 0, dailyLimit: DAILY_LIMIT });
  } catch (e) { console.error("Free API key error:", e); return res.status(500).json({ error: "Server error generating API key" }); }
});

app.post("/api/free-api/regenerate", async function(req, res) {
  try {
    var wallet = req.body.wallet;
    if (!wallet || typeof wallet !== "string" || !wallet.startsWith("0x") || wallet.length !== 42) return res.status(400).json({ error: "Valid BSC wallet address required" });
    var balance = await getBscTokenBalance(wallet);
    if (balance <= 0) return res.status(403).json({ error: "You must hold ASTHER tokens to regenerate your API key." });
    var existing = freeApiUsers.get(wallet.toLowerCase());
    var apiKey = generateApiKey();
    var user = { wallet: wallet, apiKey: apiKey, createdAt: existing ? existing.createdAt : Date.now(), usedToday: existing ? existing.usedToday : 0, lastResetDate: existing ? existing.lastResetDate : getTodayStr() };
    freeApiUsers.set(wallet.toLowerCase(), user);
    return res.json({ apiKey: apiKey, usedToday: user.usedToday, dailyLimit: DAILY_LIMIT });
  } catch (e) { console.error("Free API regenerate error:", e); return res.status(500).json({ error: "Server error regenerating API key" }); }
});

app.post("/api/v1/chat/completions", async function(req, res) {
  var authHeader = req.headers.authorization;
  if (!authHeader || !authHeader.startsWith("Bearer sk-asth-")) return res.status(401).json({ error: "Invalid API key" });
  var apiKey = authHeader.replace("Bearer ", "");
  var foundUser = null;
  freeApiUsers.forEach(function(u) { if (u.apiKey === apiKey) foundUser = u; });
  if (!foundUser) return res.status(401).json({ error: "API key not found" });
  resetDailyUsageIfNeeded(foundUser);
  if (foundUser.usedToday >= DAILY_LIMIT) return res.status(429).json({ error: "Daily usage limit reached ($15/day). Your limit resets at midnight UTC.", usedToday: foundUser.usedToday.toFixed(4), limit: DAILY_LIMIT });
  var OPENROUTER_KEY = process.env.OPENROUTER_API_KEY;
  if (!OPENROUTER_KEY) return res.status(500).json({ error: "API service not configured" });
  try {
    var orRes = await fetch("https://openrouter.ai/api/v1/chat/completions", { method: "POST",
      headers: { "Content-Type": "application/json", "Authorization": "Bearer " + OPENROUTER_KEY, "HTTP-Referer": "https://astherafi.com", "X-Title": "Astherafi Free API" },
      body: JSON.stringify(req.body) });
    var data = await orRes.json();
    var promptTokens = data.usage && data.usage.prompt_tokens ? data.usage.prompt_tokens : 0;
    var completionTokens = data.usage && data.usage.completion_tokens ? data.usage.completion_tokens : 0;
    var totalCost = (data.usage && data.usage.total_cost) ? data.usage.total_cost : ((promptTokens * 0.000001) + (completionTokens * 0.000002));
    foundUser.usedToday += totalCost;
    data._usage_info = { usedToday: foundUser.usedToday.toFixed(4), dailyLimit: DAILY_LIMIT, remaining: (DAILY_LIMIT - foundUser.usedToday).toFixed(4) };
    return res.json(data);
  } catch (e) { return res.status(502).json({ error: "API request failed" }); }
});

app.post("/api/chat", async function(req, res) {
  try {
    var clientIp = req.ip || req.socket.remoteAddress || "unknown";
    if (!checkChatRateLimit(clientIp)) return res.status(429).json({ error: "Too many requests. Please wait a moment and try again." });
    var message = req.body.message;
    var lang = req.body.lang;
    var sessionId = req.body.sessionId;
    if (!message || typeof message !== "string" || message.length > 2000) return res.status(400).json({ error: "Message is required and must be under 2000 characters" });
    var DEEPSEEK_API_KEY = process.env.DEEPSEEK_API_KEY;
    if (!DEEPSEEK_API_KEY) return res.status(500).json({ error: "AI service not configured" });
    var langInstructions = { en: "Respond in English.", zh: "Respond in Chinese (简体中文).", ar: "Respond in Arabic (العربية)." };
    var langHint = langInstructions[lang] || langInstructions.en;
    var response = await fetch("https://api.deepseek.com/chat/completions", { method: "POST",
      headers: { "Content-Type": "application/json", "Authorization": "Bearer " + DEEPSEEK_API_KEY },
      body: JSON.stringify({ model: "deepseek-chat", messages: [
        { role: "system", content: "You are Asthera AI, an expert cryptocurrency and DeFi assistant built by Astherafi. Astherafi helps users navigate the crypto market with Sentiment, Technical, and On-Chain tools powered by AI. The ASTHER token powers the ecosystem with staking pools (Flexible 60% APY, 3-Day 90% APY, 7-Day 150% APY). Tools include Token Analyzer, Token Generator, BSC Bundler, Password Score, Staking, Token Swap, Block Explorer, and more. Help with blockchain analysis, token security, staking strategies, DeFi, market analysis, and crypto education. Keep responses concise. IMPORTANT: When users ask about the ASTHER token contract address, CA, smart contract, Astherafi contract, or anything related to the token address, ALWAYS respond with the BSC contract address: 0x277204675524B49D417bD0B374A1FB09465F7777. The ASTHER token is on BNB Smart Chain (BSC). If they just type 'ca' or 'CA', respond with the contract address. " + langHint },
        { role: "user", content: message }
      ], max_tokens: 1024, temperature: 0.7 }) });
    if (!response.ok) { console.error("DeepSeek error:", await response.text()); return res.status(502).json({ error: "AI service temporarily unavailable" }); }
    var data = await response.json();
    var reply = data.choices && data.choices[0] && data.choices[0].message ? data.choices[0].message.content : "I couldn't generate a response. Please try again.";
    var session = getOrCreateSession(clientIp, sessionId);
    session.messages.push({ role: "user", text: message, timestamp: Date.now() });
    session.messages.push({ role: "ai", text: reply, timestamp: Date.now() });
    if (!session.title) session.title = message.substring(0, 50);
    session.updatedAt = Date.now();
    if (session.messages.length > MAX_MSGS * 2) session.messages = session.messages.slice(-MAX_MSGS * 2);
    return res.json({ reply: reply, sessionId: session.id });
  } catch (error) { console.error("Chat error:", error); return res.status(500).json({ error: "Failed to process chat request" }); }
});

app.get("/models", function(req, res) { res.sendFile(path.join(publicPath, "models.html")); });

var modelsCache = { data: [], ts: 0 };
app.get("/api/models", async function(req, res) {
  try {
    var now = Date.now();
    if (modelsCache.data.length > 0 && now - modelsCache.ts < 3600000) return res.json({ data: modelsCache.data });
    var resp = await fetch("https://openrouter.ai/api/v1/models");
    var json = await resp.json();
    var models = (json.data || []).map(function(m) { return { id: m.id, name: m.name, description: m.description || "", context_length: m.context_length, pricing: m.pricing, architecture: m.architecture }; });
    modelsCache = { data: models, ts: now };
    res.json({ data: models });
  } catch (e) { res.status(500).json({ error: "Failed to fetch models" }); }
});

app.get("*", function(req, res) { res.sendFile(path.join(publicPath, "index.html")); });

app.listen(PORT, function() { console.log("Astherafi server running on port " + PORT); });
